/* mml-base64.c -- base64 routines

	Copyright 2005,2006
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Functions for base64 encoding/decoding.  See .h file for notes.

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>

#include <mml/mml.h>

#include <mml/mml-base64.h>


/* Local definitions */



/* External data referenced */



/* External routines used */



/* Local data publicly available */



/* Local routines and forward references */



/* Private data */

    /* Table of base64-encoded ASCII characters, i.e. the ASCII character
       corresponding to each 6-bit encoded value, plus a final one for
       the separator value.  We don't care that the string is NUL-terminated.
    */
static	char	b64enc[] = 
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";


    /* Table of data values associated with base64-encoded ASCII
       characters.  Invalid encoded characters are represented by
       0xff, and the PAD character is represented by the pad value.
       This table is produced by a q&d program that may or may not
       sill be laying somewhere around here.
    */
static	UBYTE	b64dec[256] = {
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0x3e,  0xff,  0xff,  0xff,  0x3f,
    0x34,  0x35,  0x36,  0x37,  0x38,  0x39,  0x3a,  0x3b,
    0x3c,  0x3d,  0xff,  0xff,  0xff,  0x40,  0xff,  0xff,
    0xff,  0x00,  0x01,  0x02,  0x03,  0x04,  0x05,  0x06,
    0x07,  0x08,  0x09,  0x0a,  0x0b,  0x0c,  0x0d,  0x0e,
    0x0f,  0x10,  0x11,  0x12,  0x13,  0x14,  0x15,  0x16,
    0x17,  0x18,  0x19,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0x1a,  0x1b,  0x1c,  0x1d,  0x1e,  0x1f,  0x20,
    0x21,  0x22,  0x23,  0x24,  0x25,  0x26,  0x27,  0x28,
    0x29,  0x2a,  0x2b,  0x2c,  0x2d,  0x2e,  0x2f,  0x30,
    0x31,  0x32,  0x33,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,
    0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff,  0xff
};


/*

*//* b64_decode( srcP, srcL, dstP, dstL )

	Completely decode a base64-encoded buffer

Accepts :

	srcP		Ptr to encoded data
	srcL		Number of bytes in encoded buffer
	dstP		Ptr to destination buffer
	dstL		Size of destination buffer

Returns :

	<value>		number of bytes decoded into destination buffer,
			-1 if overflow,
			-2 if other error (internal error)

Notes :

	This function simply makes use of the other individual functions
	to do a complete decode operation.

*/

int
b64_decode ARGLIST( (srcP, srcL, dstP, dstL) )
   NFARG( UBYTE		*srcP )		/* Ptr to encoded data */
   NFARG( int		srcL )		/* # of encoded bytes */
   NFARG( char		*dstP )		/* Where to put decoded chars */
    FARG( int		dstL )		/* Size of destination buffer */
{
	MMLB64		b64;		/* base64 handle */

    /* Init the handle for the decode operation */
    b64_init( &b64 );

    /* Decode the entire buffer as the final part. */
    return ( b64_decode_rest( &b64, srcP, srcL, dstP, dstL ) );
}
/*

*//* b64_decode_byte( b64P, ch, dstP, dstL )

	Decode [another] byte of encoded data

Accepts :

	b64P		Ptr to a b64 handle
	ch		The next byte of encoded data
	dstP		Ptr to destination buffer
	dstL		Size of destination buffer

Returns :

	<value>		number of bytes decoded into destination buffer,
			-1 if overflow,
			-2 if other error.

Notes :


*/

int
b64_decode_byte ARGLIST( (b64P, ch, dstP, dstL) )
   NFARG( MMLB64	*b64P )		/* base64 decode/decode handle */
   NFARG( UBYTE		ch )		/* Source byte to decode */
   NFARG( char		*dstP )		/* Where to put decoded chars */
    FARG( int		dstL )		/* Size of destination buffer */
{
	UBYTE		dbyte;		/* Decoded byte value */
	int		len;

    dbyte = b64dec[ ch ];
    if ( dbyte == 0xff )
	return ( -2 );			/* Invalid encoding */

    /* Special handling for pad bytes */
    if ( dbyte == B64PADX ) {
	if ( b64P->b64_pX >= 2 )
	    return ( -2 );		/* Too much pad. */

	if ( b64P->b64_bX == 0 )
	    return ( -2 );		/* Can't pad after zero bytes */

	/* Count the pad byte(s) and return; the 'finish' processing
	   will emit the final data.
	*/
	++b64P->b64_pX;
	return ( 0 );
    }

    /* Here if non-pad, valid decoded 6-bit byte. */
    if ( b64P->b64_pX != 0 )		/* Can't have data after pad */
	return ( -2 );

    /* The buffered index (bX) is the number of encoded bytes that have
        been processed.  We assemble the decoded bytes in the array.
    */
    switch ( b64P->b64_bX ) {
	case 0:				/* First encoded byte */
	    b64P->b64_b[0] = ( dbyte << 2 );
	    ++b64P->b64_bX;
	    len = 0;			/* No bytes emitted */
	    break;
	
	case 1:				/* Second encoded byte */
	    b64P->b64_b[0] |= ( dbyte >> 4 );
	    b64P->b64_b[1] =  ( dbyte << 4 ); 
	    ++b64P->b64_bX;
	    len = 0;			/* No bytes emitted */
	    break;

	case 2:				/* Third encoded byte */
	    b64P->b64_b[1] |= ( dbyte >> 2 );
	    b64P->b64_b[2] =  ( dbyte << 6 );
	    ++b64P->b64_bX;
	    len = 0;			/* No bytes emitted */
	    break;

	case 3:				/* Fourth and final encoded byte */
	    b64P->b64_b[2] |= dbyte;

	    /* Emit the 3 bytes */
	    if ( dstL < 3 )
		len = -1;
	    else {
		for ( len = 0; len < 3; ++len ) {
		    dstP[ len ] = b64P->b64_b[ len ];
		}
	    }
	    b64P->b64_bX = 0;		/* For next time */
	    break;

	default:			/* Some error */
	    len = -2;
	    break;
    }

    return ( len );
}
/*

*//* b64_decode_finish( b64P, dstP, dstL )

	Finish base64 decoding

Accepts :

	b64P		Ptr to a b64 handle
	dstP		Ptr to destination buffer
	dstL		Size of destination buffer

Returns :

	<value>		number of bytes decoded into destination buffer,
			-1 if overflow,
			-2 if other error.

Notes :

	Takes care of any buffered/incomplete decoding.

*/

int
b64_decode_finish ARGLIST( (b64P, dstP, dstL) )
   NFARG( MMLB64	*b64P )		/* base64 decode/decode handle */
   NFARG( char		*dstP )		/* Where to put decoded chars */
    FARG( int		dstL )		/* Size of destination buffer */
{
	int		bC;		/* # of bytes ready */
	int		len;		/* Output counter */

    /* Note that we should only be here if 0, 2, or 3 encoded bytes have
       been buffered, (it is impossible to have 1 encoded byte).  What we
       emit depends on how much we have buffered.  Also we must ensure
       that the correct number of pad bytes have been seen.
    */

    switch ( b64P->b64_bX ) {
	case 0:				/* No bytes buffered */
	    bC = 0;			/* Nothing to emit. */
	    break;

	case 1:				/* 1 encoded byte buffered */
	    return ( -2 );		/* This is impossible. */

	case 2:				/* 2 encoded bytes buffered */
	    if ( b64P->b64_pX != 2 )	/* 2 pad bytes required */
		return ( -2 );
	    bC = 2;			/* Emit 2 bytes */
	    break;

	case 3:				/* 3 encoded bytes buffered */
	    if ( b64P->b64_pX != 1 )	/* 1 pad byte required */
		return ( -2 );
	    bC = 3;			/* Emit 3 bytes */
	    break;

	default:			/* Some error */
	    return ( -2 );
    }

    /* Emit the bytes */
    if ( dstL < bC )
	len = -1;
    else {
	for ( len = 0; len < bC; ++len )
	    dstP[ len ] = b64P->b64_b[ len ];
    }

    return ( len );
}
/*

*//* b64_decode_part( b64P, srcP, srcL, dstP, dstL )

	Partly decode base64 data

Accepts :

	b64P		Ptr to a b64 handle
	srcP		Ptr to base64-encoded data
	srcL		Number of bytes in encoded buffer
	dstP		Ptr to destination buffer
	dstL		Size of destination buffer

Returns :

	<value>		number of bytes decoded into destination buffer,
			-1 if overflow,
			-2 if some other error.

Notes :

	On error (or partial) return, some bytes may have been placed in
	output buffer.  The decoded data should not be used.

*/

int
b64_decode_part ARGLIST( (b64P, srcP, srcL, dstP, dstL) )
   NFARG( MMLB64	*b64P )		/* base64 decode/decode handle */
   NFARG( UBYTE		*srcP )		/* Ptr to base64-encoded data */
   NFARG( int		srcL )		/* # of bytes in source buffer */
   NFARG( char		*dstP )		/* Where to put decoded chars */
    FARG( int		dstL )		/* Size of destination buffer */
{
	int		i, len;

    /* Loop through encoded bytes */
    for ( len = 0; srcL-- > 0; ) {
	i = b64_decode_byte( b64P, *srcP++, dstP, dstL );
	if ( i < 0 )
	    return ( i );
	if ( i > 0 ) {
	    dstP += i;
	    dstL -= i;
	    len += i;
	}
    }

    return ( len );
}
/*

*//* b64_decode_rest( b64P, srcP, srcL, dstP, dstL )

	Decode the final part (the rest of) base64-encoded data

Accepts :

	b64P		Ptr to a b64 handle
	srcP		Ptr to encoded data
	srcL		Number of bytes in source buffer
	dstP		Ptr to destination buffer
	dstL		Size of destination buffer

Returns :

	<value>		number of bytes decoded into destination buffer,
			-1 if overflow;
			-2 if other error.

Notes :

	This is a final decoding, where there might have been some other
	data previously and partly decoded.  Partial decoding state is
	maintained in the base64 handle.  The source length (srcL) may
	be zero, in which case the source pointer (srcP) may be NULL.
	(However, caller would be more likely to use a 'finish' function
	for this case.)

*/

int
b64_decode_rest ARGLIST( (b64P, srcP, srcL, dstP, dstL) )
   NFARG( MMLB64	*b64P )		/* base64 decode/decode handle */
   NFARG( UBYTE		*srcP )		/* Ptr to base64-encoded data */
   NFARG( int		srcL )		/* # of source bytes */
   NFARG( char		*dstP )		/* Where to put decoded chars */
    FARG( int		dstL )		/* Size of destination buffer */
{
	int		len, finlen;

    /* Decode the source as a partial decoding, and then finish it. */
    if ( srcL > 0 ) {
	len = b64_decode_part( b64P, srcP, srcL, dstP, dstL );
	if ( len < 0 )
	    return ( len );
	dstP += len;
	dstL -= len;
    }

    finlen = b64_decode_finish( b64P, dstP, dstL );
    if ( finlen < 0 )
	return ( finlen );

    return ( len + finlen );
}
/*

*//* b64_decode_size( srcL )

	Determine a buffer size required to hold decoded data

Accepts :

	srcL		Length of the source data (# of bytes)

Returns :

	<value>		Number of bytes of buffer required for the decoding.

Notes :

	The returned value is a number of bytes large enough to hold a
	full or partial decoding of the given source length.  The value
	takes into consideration the possibility that some part of the
	decoding might already be buffered, and that the finalization of
	the decoding might require more extra byte(s).  Note that not all
	combinations are additive, but it's pragmatic to be safe.

*/

int
b64_decode_size ARGLIST( (srcL) )
    FARG( int		srcL )		/* Source data length (bytes) */
{
    /* The first byte of source might consume 4 decoded bytes if there is
       already a partial decoding in the pipeline.  The remaining bytes
       can potentially consume 4 bytes for every 3, rounded up.  So
       we have:

           (((srcL -1)+3)/3)*4 +4
       or
           ((srcL+2)/3)*4 +4
       or
           ((srcL +5)/3) * 4
    */
    return ( ((srcL +5)/3) * 4 );
}
/*

*//* b64_encode( srcP, srcL, dstP, dstL )

	Completely encode a source buffer into base64

Accepts :

	srcP		Ptr to source data
	srcL		Number of bytes in source data
	dstP		Ptr to destination buffer
	dstL		Size of destination buffer

Returns :

	<value>		number of bytes encoded into destination buffer,
			-1 if overflow,
			-2 if other error (internal error)

Notes :

	This function simply makes use of the other individual functions
	to do a complete encode operation.

*/

int
b64_encode ARGLIST( (srcP, srcL, dstP, dstL) )
   NFARG( UBYTE		*srcP )		/* Ptr to source data */
   NFARG( int		srcL )		/* # of source bytes */
   NFARG( char		*dstP )		/* Where to put encoded chars */
    FARG( int		dstL )		/* Size of destination buffer */
{
	MMLB64		b64;		/* base64 handle */

    /* Init the handle for the encode operation */
    b64_init( &b64 );

    /* The source data is the entire data, so it is encoded as the final
       part.
    */
    return ( b64_encode_rest( &b64, srcP, srcL, dstP, dstL ) );
}
/*

*//* b64_encode_byte( b64P, ch, dstP, dstL )

	Encode [another] byte of source data

Accepts :

	b64P		Ptr to a b64 handle
	ch		The byte to add to the encoded output
	dstP		Ptr to destination buffer
	dstL		Size of destination buffer

Returns :

	<value>		number of bytes encoded into destination buffer,
			-1 if overflow,
			-2 if other error.

Notes :


*/

int
b64_encode_byte ARGLIST( (b64P, ch, dstP, dstL) )
   NFARG( MMLB64	*b64P )		/* base64 encode/decode handle */
   NFARG( UBYTE		ch )		/* Source byte to encode */
   NFARG( char		*dstP )		/* Where to put encoded chars */
    FARG( int		dstL )		/* Size of destination buffer */
{
	int		len;

    /* The buffered index is the number of input bytes that have been
        processed.  We assemble the 6-bit bytes in the buffer array.
    */
    switch ( b64P->b64_bX ) {
	case 0:				/* First buffered byte */
	    b64P->b64_b[0] = ( ch >> 2 );
	    b64P->b64_b[1] = ( ch << 4 ) & 0x3f;
	    ++b64P->b64_bX;
	    len = 0;			/* No bytes emitted */
	    break;

	case 1:				/* Second buffered byte */
	    b64P->b64_b[1] |= ( ch >> 4 );
	    b64P->b64_b[2] =  ( ch << 2 ) & 0x3f;
	    ++b64P->b64_bX;
	    len = 0;			/* No bytes emitted */
	    break;

	case 2:				/* Third and final buffered byte */
	    b64P->b64_b[2] |= ( ch >> 6 );
	    b64P->b64_b[3] =  ( ch & 0x3f );

	    /* Emit the 4 bytes */
	    if ( dstL < 4 )
		len = -1;
	    else {
		for ( len = 0; len < 4; ++len ) {
		    dstP[ len ] = b64enc[ b64P->b64_b[ len ] ];
		}
	    }
	    b64P->b64_bX = 0;		/* For next time */
	    break;

	default:			/* Error */
	    len = -2;
	    break;
    }

    return ( len );
}
/*

*//* b64_encode_finish( b64P, dstP, dstL )

	Finish base64 encoding

Accepts :

	b64P		Ptr to a b64 handle
	dstP		Ptr to destination buffer
	dstL		Size of destination buffer

Returns :

	<value>		number of bytes encoded into destination buffer,
			-1 if overflow,
			-2 if other error.

Notes :

	Takes care of any buffered/incomplete encoding.

*/

int
b64_encode_finish ARGLIST( (b64P, dstP, dstL) )
   NFARG( MMLB64	*b64P )		/* base64 encode/decode handle */
   NFARG( char		*dstP )		/* Where to put encoded chars */
    FARG( int		dstL )		/* Size of destination buffer */
{
	int		bC;		/* # of bytes ready */
	int		len;		/* Output counter */

    /* Finish preparing the buffered base64 bytes */
    switch ( b64P->b64_bX ) {
	case 0:				/* No bytes buffered, nothing to do. */
	    return ( 0 );

	case 1:				/* 3rd & 4th bytes will be pad */
	    b64P->b64_b[2] = B64PADX;
		/* Fall Through */
	case 2:				/* 4th byte will be pad */
	    b64P->b64_b[3] = B64PADX;
	    bC = 4;			/* 4 bytes to emit */
	    break;

	default:			/* Some error. */
	    return ( -2 );
    }

    /* Emit the bytes */
    if ( dstL < bC )
	len = -1;
    else {
	for ( len = 0; len < bC; ++len ) {
	    dstP[ len ] = b64enc[ b64P->b64_b[ len ] ];
	}
    }

    return ( len );
}
/*

*//* b64_encode_part( b64P, srcP, srcL, dstP, dstL )

	Partly encode source data into base64

Accepts :

	b64P		Ptr to a b64 handle
	srcP		Ptr to source data
	srcL		Number of bytes in source data
	dstP		Ptr to destination buffer
	dstL		Size of destination buffer

Returns :

	<value>		number of bytes encoded into destination buffer,
			-1 if overflow,
			-2 if some other error.

Notes :

	On error (or partial) return, some bytes may have been placed in
	output buffer.  The encoded data should not be used.

*/

int
b64_encode_part ARGLIST( (b64P, srcP, srcL, dstP, dstL) )
   NFARG( MMLB64	*b64P )		/* base64 encode/decode handle */
   NFARG( UBYTE		*srcP )		/* Ptr to source data */
   NFARG( int		srcL )		/* # of source bytes */
   NFARG( char		*dstP )		/* Where to put encoded chars */
    FARG( int		dstL )		/* Size of destination buffer */
{
	int		i, len;

    /* Loop through input bytes */
    for ( len = 0; srcL-- > 0; ) {
	i = b64_encode_byte( b64P, *srcP++, dstP, dstL );
	if ( i < 0 )
	    return ( i );
	if ( i > 0 ) {
	    dstP += i;
	    dstL -= i;
	    len += i;
	}
    }

    return ( len );
}
/*

*//* b64_encode_rest( b64P, srcP, srcL, dstP, dstL )

	Encode the final part (the rest of) source data into base64

Accepts :

	b64P		Ptr to a b64 handle
	srcP		Ptr to source data
	srcL		Number of bytes in source data
	dstP		Ptr to destination buffer
	dstL		Size of destination buffer

Returns :

	<value>		number of bytes encoded into destination buffer,
			-1 if overflow;
			-2 if other error.

Notes :

	This is a final encoding, where there might have been some other
	data previously and partly encoded.  Partial encoding state is
	maintained in the base64 handle.  The source length (srcL) may
	be zero, in which case the source pointer (srcP) may be NULL.
	(However, caller would be more likely to use a 'finish' function
	for this case.)

*/

int
b64_encode_rest ARGLIST( (b64P, srcP, srcL, dstP, dstL) )
   NFARG( MMLB64	*b64P )		/* base64 encode/decode handle */
   NFARG( UBYTE		*srcP )		/* Ptr to source data */
   NFARG( int		srcL )		/* # of source bytes */
   NFARG( char		*dstP )		/* Where to put encoded chars */
    FARG( int		dstL )		/* Size of destination buffer */
{
	int		len, finlen;

    /* Encode the source as a partial encoding, and then finish it. */
    if ( srcL > 0 ) {
	len = b64_encode_part( b64P, srcP, srcL, dstP, dstL );
	if ( len < 0 )
	    return ( len );
	dstP += len;
	dstL -= len;
    }

    finlen = b64_encode_finish( b64P, dstP, dstL );
    if ( finlen < 0 )
	return ( finlen );

    return ( len + finlen );
}
/*

*//* b64_encode_size( srcL )

	Determine a buffer size required to hold a base64 encoding

Accepts :

	srcL		Length of the source data (# of bytes)

Returns :

	<value>		Number of bytes of buffer required for the encoding.

Notes :

	The returned value is a number of bytes large enough to hold a
	full or partial encoding of the given source length.  The value
	takes into consideration the possibility that some part of the
	encoding might already be buffered, and that the finalization of
	the encoding might require more extra byte(s).  Note that not all
	combinations are additive, but it's pragmatic to be safe.

*/

int
b64_encode_size ARGLIST( (srcL) )
    FARG( int		srcL )		/* Source data length (bytes) */
{
    /* The first byte of source might consume 4 encoded bytes if there is
       already a partial encoding in the pipeline.  The remaining bytes
       can potentially consume 4 bytes for every 3, rounded up.  So
       we have:

           (((srcL -1)+3)/3)*4 +4
       or
           ((srcL+2)/3)*4 +4
       or
           ((srcL +5)/3) * 4
    */
    return ( ((srcL +5)/3) * 4 );
}
/*

*//* b64_init( b64P )

	Initialize a b64 handle

Accepts :

	b64P		Ptr to a b64 handle

Returns :

	<nothing>

Notes :


*/

void
b64_init ARGLIST( (b64P) )
    FARG( MMLB64	*b64P )		/* base64 encode/decode handle */
{
    b64P->b64_bX = 0;
    b64P->b64_pX = 0;
}
