/* mml-dns.c -- Rudimentary interface to DNS resolver routines

	Copyright 2003,2004,2005,2006
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Contains functions to simplify access to resolver routines.
Simplicity tends to be limiting, and so are these functions.
However they provide basic DNS resolver capabilities.

Functions are named "mdns_" rather than "dns_" to avoid what are
likely to be natural naming conflicts.

*/

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <string.h>

#include <mml/mml.h>
#include <mml/mml-ll.h>
#include <mml/mml-str.h>
#include <mml/mml-alloc.h>
#include <mml/mml-dns.h>

/* Local definitions */

    /* A tuple that is a query type / query name association */
typedef struct {
    char	*qt_nameP;		/* Name */
    int		qt_type;		/* Type, coerced to int */
} QTNT;


/* External data referenced */



/* External routines used */



/* Local data publicly available */



/* Local routines and forward references */

static	BOOL	mdns_ans_set PROTO( (MML_DNS *dnsP) );
static	int	mdns_val_word PROTO( (UBYTE *bP) );

/* Private data */

    /* Query type names for class ns_c_in, ending with a NULL name
       and -1 query type value.
    */

static	QTNT	Qtnt_in[] = {
/*	Name		Class			*/
{	"A",		ns_t_a		},	/* Host address. */
{	"NS",		ns_t_ns		},	/* Authoritative server. */

#ifdef	MDNS_ALL_IN_TYPE_NAMES
{	"MD",		ns_t_md		},	/* Mail destination. */
{	"MF",		ns_t_mf		},	/* Mail forwarder. */
#endif	/* MDNS_ALL_IN_TYPE_NAMES */

{	"CNAME",	ns_t_cname	},	/* Canonical name. */
{	"SOA",		ns_t_soa	},	/* Start of authority zone. */

#ifdef	MDNS_ALL_IN_TYPE_NAMES
{	"MB",		ns_t_mb		},	/* Mailbox domain name. */
{	"MG",		ns_t_mg		},	/* Mail group member. */
{	"MR",		ns_t_mr		},	/* Mail rename name. */
{	"NULL",		ns_t_null	},	/* Null resource record. */
{	"WKS",		ns_t_wks	},	/* Well known service. */
#endif	/* MDNS_ALL_IN_TYPE_NAMES */

{	"PTR",		ns_t_ptr	},	/* Domain name pointer. */

#ifdef	MDNS_ALL_IN_TYPE_NAMES
{	"HINFO",	ns_t_hinfo	},	/* Host information. */
{	"MINFO",	ns_t_minfo	},	/* Mailbox information. */
#endif	/* MDNS_ALL_IN_TYPE_NAMES */

{	"MX",		ns_t_mx		},	/* Mail routing information. */
{	"TXT",		ns_t_txt	},	/* Text strings. */

#ifdef	MDNS_ALL_IN_TYPE_NAMES
{	"RP",		ns_t_rp		},	/* Responsible person. */
{	"AFSDB",	ns_t_afsdb	},	/* AFS cell database. */
{	"X25",		ns_t_x25	},	/* X_25 calling address. */
{	"ISDN",		ns_t_isdn	},	/* ISDN calling address. */
{	"RT",		ns_t_rt		},	/* Router. */
{	"NSAP",		ns_t_nsap	},	/* NSAP address. */
{	"NSAP_ptr",	ns_t_nsap_ptr	},	/* Reverse NSAP lookup (deprecated). */
{	"SIG",		ns_t_sig	},	/* Security signature. */
{	"KEY",		ns_t_key	},	/* Security key. */
{	"PX",		ns_t_px		},	/* X.400 mail mapping. */
{	"GPOS",		ns_t_gpos	},	/* Geographical position (withdrawn). */
{	"AAAA",		ns_t_aaaa	},	/* Ip6 Address. */
{	"LOC",		ns_t_loc	},	/* Location Information. */
{	"NXT",		ns_t_nxt	},	/* Next domain (security). */
{	"EID",		ns_t_eid	},	/* Endpoint identifier. */
{	"NIMLOC",	ns_t_nimloc	},	/* Nimrod Locator. */
#endif	/* MDNS_ALL_IN_TYPE_NAMES */

{	"SRV",		ns_t_srv	},	/* Server Selection. */

#ifdef	MDNS_ALL_IN_TYPE_NAMES
{	"ATMA",		ns_t_atma	},	/* ATM Address */
{	"NAPTR",	ns_t_naptr	},	/* Naming Authority PoinTeR */
{	"KX",		ns_t_kx		},	/* Key Exchange */
{	"CERT",		ns_t_cert	},	/* Certification record */
{	"A6",		ns_t_a6		},	/* IPv6 address (deprecates AAAA) */
{	"DNAME",	ns_t_dname	},	/* Non-terminal DNAME (for IPv6) */
{	"SINK",		ns_t_sink	},	/* Kitchen sink (experimentatl) */
{	"OPT",		ns_t_opt	},	/* EDNS0 option (meta-RR) */
{	"TSIG",		ns_t_tsig	},	/* Transaction signature. */
#endif	/* MDNS_ALL_IN_TYPE_NAMES */

#if 0	/* BIND-internal stuff, here for commentary only */
{	"IXFR",		ns_t_ixfr	},	/* Incremental zone transfer. */
{	"AXFR",		ns_t_axfr	},	/* Transfer zone of authority. */
{	"MAILB",	ns_t_mailb	},	/* Transfer mailbox records. */
{	"MAILA",	ns_t_maila	},	/* Transfer mail agent records. */
{	"ANY",		ns_t_any	},	/* Wildcard match. */
{	"ZXFR",		ns_t_zxfr	},	/* BIND-specific, nonstandard. */
{	"MAX",		ns_t_max	},
#endif	/* 0 */

{	NULL,		-1		}

};


#ifndef	MML_HAVE_RES_NINIT

    /* The resolver does not have res_ninit; this means it keeps global
       static context.  All allocated MML_DNS structures are therefor
       co-dependant.  When an application locks any one of these structures,
       it must lock access to the entire mml-dns module.
    */
# ifdef	MML_PTHREADS
static	pthread_mutex_t Mdns_lock = PTHREAD_MUTEX_INITIALIZER;
# endif	/* MML_PTHREADS */

#endif	/* MML_HAVE_RES_NINIT */


/*

*//* mdns_ans_first(dnsP)

	Reset to the first answer from last query result

Accepts :

	dnsP		PTR to dns handle

Returns :

	<value>		TRUE if OK
			FALSE if not (i.e. no previous answer)

Notes :


*/

BOOL
mdns_ans_first ARGLIST( (dnsP) )
    FARG( MML_DNS	*dnsP )		/* Ptr to dns access block */
{
    dnsP->dn_ansX = 0;			/* Set to first one */
    dnsP->dn_rP = dnsP->dn_rfP;		/* Position to first result header */

    /* Now just get the current answer (stepping result point to the next) */
    return ( mdns_ans_next( dnsP ) );
}
/*

*//* mdns_ans_next(dnsP)

	Get the next answer from last query result

Accepts :

	dnsP		PTR to dns handle

Returns :

	<value>		TRUE if OK
			FALSE if not (i.e. no next answer)

Notes :

	Fetching an answer always sets up the dnsP to point to
	the next result header in the query response buffer.

*/

BOOL
mdns_ans_next ARGLIST( (dnsP) )
    FARG( MML_DNS	*dnsP )		/* Ptr to dns access block */
{
    if ( dnsP->dn_ansX >= dnsP->dn_ansC )
	return ( FALSE );		/* No more results, return false */

    ++dnsP->dn_ansX;			/* Step to the next */

    /* Setup the answer and return the status */
    return ( mdns_ans_set( dnsP ) );
}
/*

*//* mdns_free( dnsP )

	Close/free a DNS handle

Accepts :

	dnsP		Ptr to DNS handle

Returns :

	<nothing>

Notes :


*/

void
mdns_free ARGLIST( (dnsP) )
    FARG( MML_DNS	*dnsP )		/* Ptr to dns access block */
{
    /* Close the resolver state block */
    res_nclose( dnsP->dn_statP );

    /* Free the response buffer */
    dealloc( dnsP->dn_rbufP );

    /* Note that we co-allocated the resolver state block with our
       MML_DNS block, so there's no need to free it specifically
    */

    /* Free the DNS handle */
    dealloc( dnsP );
}
/*

*//* mdns_ip4_batoip( oP )

	Convert byte (octet) array to IP address

Accepts :

	oP		A 4-byte array containing bytes of IP address,
			  high byte in [0].

Returns :

	<value>		IPv4 address, in host format

Notes :

	As opposed to using a union and net-to-host conversions.

	and as opposed to simply being a macro.

*/

in_addr_t
mdns_ip4_batoip ARGLIST( (oP) )
    FARG( UBYTE		*oP )		/* Ptr to the bytes */
{
    return( ( oP[0] << 24 ) |
	    ( oP[1] << 16 ) |
	    ( oP[2] <<  8 ) |
	    ( oP[3]       ) );
}
/*

*//* mdns_ip4_iptoba( addr, oP )

	Convert IPv4 address to byte (octet) array

Accepts :

	addr		The IP address to convert, in host format
	oP		Pointer to 4-byte array

Returns :

	<nothing>

Notes :

	Puts first IP octet in 0th array byte, etc.

*/

void
mdns_ip4_iptoba ARGLIST( (addr, oP) )
   NFARG( in_addr_t	addr )		/* The address */
    FARG( UBYTE		*oP )		/* Where to put the bytes */
{
    /* This could be done in a loop but just unroll it by hand */
    oP += 3;
    *oP-- = addr & 0xff;  addr >>= 8;
    *oP-- = addr & 0xff;  addr >>= 8;
    *oP-- = addr & 0xff;  addr >>= 8;
    *oP = addr & 0xff;
}
/*

*//* mdns_lock(dnsP)

	Gain a lock on the DNS handle

Accepts :

	dnsP		PTR to dns handle

Returns :

	<nothing>

Notes :

	Is a no-op if pthread support is not enabled;

	Also locks the entire module if res_ninit() et al are not
	available, since all MML_DNS structs are interdependent (on
	global static context maintained by the resolver library).

*/

void
mdns_lock ARGLIST( (dnsP) )
    FARG( MML_DNS	*dnsP )		/* Ptr to dns access block */
{
#ifndef MML_HAVE_RES_NINIT
    if ( mml_mutex_lock( &Mdns_lock ) != 0 )
	warning( "mdns_lock: can't get Mdns_lock" );
#endif

    if ( mml_mutex_lock( &dnsP->dn_applock ) != 0 )
	warning( "mdns_lock: can't get applock" );

    /* done */
}
/*

*//* mdns_new()

	Allocate and open a new DNS handle

Accepts :



Returns :

	<value>		Ptr to MML_DNS block if successful
			NULL if not

Notes :


*/

MML_DNS *
mdns_new NOARGLIST
{
	MML_DNS		*dnsP;		/* Ptr to block */

    /* Co-allocate a resolver query block with our block */
    dnsP = (MML_DNS *)emalloc( "mdns_new", "DNS block",
		        sizeof( MML_DNS ) + sizeof( struct __res_state) );

    if ( dnsP != NULL ) {
	/* Allocate response buffer */
	dnsP->dn_rbufL = NS_PACKETSZ;
	dnsP->dn_rbufP = emalloc( "mdns_new", "DNS response buffer",
				   dnsP->dn_rbufL );
	if ( dnsP->dn_rbufP == NULL ) {
	    dealloc( dnsP );
	    dnsP = NULL;
	}
	dnsP->dn_rhdrP = (HEADER *)dnsP->dn_rbufP;
    }

    if ( dnsP != NULL ) {
	/* Point to the resolver query state block */
	dnsP->dn_statP = (res_state)(&dnsP[1]);

	/* Initialize the resolver block */
	memset( dnsP->dn_statP, 0, sizeof(struct __res_state) );
	if ( res_ninit( dnsP->dn_statP ) < 0 ) {
	    /* Failed... */
	    dealloc( dnsP->dn_rbufP );
	    dealloc( dnsP );
	    dnsP = NULL;
	}
	else {
	    /* Initialize the rest of the block */
	    dnsP->dn_ansC = 0;
	}
    }

    /* Init the lock that applications can use */
    if ( mml_mutex_init( &dnsP->dn_applock ) != 0 )
	warning( "mdns_new: can't init applock" );

    return ( dnsP );
}
/*

*//* mdns_query( dnsP, queryP, qclass, qtype )

	Make a query 

Accepts :

	dnsP		Ptr to dns query handle
	queryP		Ptr to string for the query
	qclass		query class (probably ns_c_in)
	qtype		query type

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :


*/

BOOL
mdns_query ARGLIST( (dnsP, queryP, qclass, qtype) )
   NFARG( MML_DNS	*dnsP )		/* DNS control block */
   NFARG( char		*queryP )	/* Query to make */
   NFARG( ns_class	qclass )	/* Query class */
    FARG( ns_type	qtype )		/* Query type */
{
	int		i;
	int		len;
	int		ansX, ansC;
	UBYTE		*newbufP;

    /* Try to make the query, increasing buffer size as necessary.
       we'll iterate a small number of times to try to catch timing
       error variations in the response size.  But not forever.
    */
    for ( i = 0; i < 5; ++i ) {
	len = res_nquery( dnsP->dn_statP, queryP, qclass, qtype,
			  dnsP->dn_rbufP, dnsP->dn_rbufL );

	if ( len <= dnsP->dn_rbufL )
	    break;

	/* Response is larger than our buffer.  Make a new buffer.
	   Not using realloc here, so that we can just keep the old
	   buffer if the new allocation fails.  (Not that we'll
	   necessarily get a return if emalloc just exits,
	   but be prepared, ya know.)
	*/
	len += 100;			/* Add some slop. */
	newbufP = (UBYTE *)emalloc( "mdns_query", "new response buffer",
				    len );
	if ( newbufP == NULL )
	    return ( FALSE );
	dealloc( dnsP->dn_rbufP );
	dnsP->dn_rbufP = newbufP;
	dnsP->dn_rbufL = len;
	dnsP->dn_rhdrP = (HEADER *)dnsP->dn_rbufP;
    }

    if ( len < 0 )		/* Maybe this should be < sizeof(HEADER) ? */
	return ( FALSE );

    /* Skip past the query that's returned in the result buffer */
    dnsP->dn_rP = dnsP->dn_rbufP + sizeof(HEADER);
    dnsP->dn_reP = dnsP->dn_rbufP + len;

    ansC = ntohs( dnsP->dn_rhdrP->qdcount );
    for ( ansX = 0; ansX < ansC; ++ansX ) {
	len = dn_expand( dnsP->dn_rbufP,
			 dnsP->dn_reP, dnsP->dn_rP,
			 (BYTE *)dnsP->dn_nbuf, sizeof(dnsP->dn_nbuf) );
	if ( len < 0 )
	    break;
	dnsP->dn_rP += (len + NS_QFIXEDSZ);
    }

    /* Result ptr points to first result header.. set it up */
    dnsP->dn_rfP = dnsP->dn_rP;
    dnsP->dn_ansC = ntohs( dnsP->dn_rhdrP->ancount );

    /* Answer always comes here.. */
    dnsP->dn_ansP = &dnsP->dn_nbuf[0];

    return ( mdns_ans_first( dnsP ) );
}
/*

*//* mdns_query_type_str( qclass, nameP )

	Get query type from string representation

Accepts :

	qclass		query class (probably ns_c_in)
	nameP		Query type name to translate

Returns :

	<value>		query type value, -1 if unknown

Notes :

	The query type is returned as an int, but is really whatever
	type corresponds to the class.

	Only qclass ns_c_in is supported, and it may ever be so.

	Seems like there must already be one of these somewhere,
	(other than things like dns_datatype_fromtext() buried in
	bind), but what do I know.

*/

int
mdns_query_type_str ARGLIST( (qclass, nameP) )
   NFARG( ns_class	qclass )	/* Query class */
    FARG( char		*nameP )	/* Query type name */
{
	QTNT		*qtntP;		/* name/value ptr */

    if ( qclass != ns_c_in ) {
	return ( -1 );
    }

    /* Look up the name */
    for ( qtntP = &Qtnt_in[0]; qtntP->qt_nameP != NULL; ++qtntP )
	if ( stricmp( nameP, qtntP->qt_nameP ) == 0 )
	    break;

    return ( qtntP->qt_type );
}
/*

*//* mdns_unlock(dnsP)

	Release a lock on the DNS handle

Accepts :

	dnsP		PTR to dns handle

Returns :

	<nothing>

Notes :

	Is a no-op if pthread support is not enabled;

	Also unlocks the entire module if res_ninit() et al are not
	available, since all MML_DNS structs are interdependent (on
	global static context maintained by the resolver library).

*/

void
mdns_unlock ARGLIST( (dnsP) )
    FARG( MML_DNS	*dnsP )		/* Ptr to dns access block */
{
    if ( mml_mutex_lock( &dnsP->dn_applock ) != 0 )
	warning( "mdns_unlock: can't release applock" );

#ifndef MML_HAVE_RES_NINIT
    if ( mml_mutex_unlock( &Mdns_lock ) != 0 )
	warning( "mdns_unlock: can't release Mdns_lock" );
#endif

}

/***********************************************************************
 *                                                                     *
 *                     Internal support routines                       *
 *                                                                     *
 ***********************************************************************/



/*

*//* mdns_ans_set(dnsP) 

	Setup next answer

Accepts :

	dnsP		Ptr to DNS block set up to point to next
			answer's result header.

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	Answer point is set past the result header, with
	result type and length filled in.

*/

static BOOL
mdns_ans_set ARGLIST( (dnsP) )
    FARG( MML_DNS	*dnsP )		/* DNS block */
{
	BOOL		status;
	int		len;

    len = dn_expand( dnsP->dn_rbufP,
		     dnsP->dn_reP, dnsP->dn_rP,
		     (BYTE *)&dnsP->dn_nbuf[0], sizeof(dnsP->dn_nbuf) );

    if ( len < 0 )
	return ( FALSE );

    status = TRUE;			/* Let's assume success now */

    dnsP->dn_rP += len;

    dnsP->dn_anstype = mdns_val_word( dnsP->dn_rP );
    dnsP->dn_ansL = mdns_val_word( dnsP->dn_rP + 8 );

    /* Skip header, get the real result */
    dnsP->dn_rP += NS_RRFIXEDSZ;

    /* How we decode the result depends on what it is. */
    switch( dnsP->dn_anstype ) {
	case	ns_t_a:			/* IPv4 address */
	    dnsP->dn_ipaddr = htonl( mdns_ip4_batoip( dnsP->dn_rP ) );
	    break;

	case	ns_t_txt:		/* Byte-counted text */
	    len = dnsP->dn_rP[0];
	    if ( len >= sizeof(dnsP->dn_nbuf) )
		status = FALSE;
	    else {
		memcpy( &dnsP->dn_nbuf[0], &dnsP->dn_rP[1], len );
		dnsP->dn_nbuf[len] = NUL;
	    }
	    break;

	case	ns_t_cname:
	case	ns_t_ptr:
	    /* Some compressed domain name value-- expand it. */
	    len = dn_expand( dnsP->dn_rbufP,
			     dnsP->dn_reP, dnsP->dn_rP,
			     (BYTE *)&dnsP->dn_nbuf[0], sizeof(dnsP->dn_nbuf) );
	    if ( len < 0 )
		status = FALSE;
	    break;

	case	ns_t_mx:		/* MX value */
	    /* MX value has 2-byte precedence plus name.  We store the
	       value into the dn_val1 location, and the name into the
	       answer string location.
	    */
	    dnsP->dn_val1 = mdns_val_word( &dnsP->dn_rP[0] );
	    len = dn_expand( dnsP->dn_rbufP,
			     dnsP->dn_reP, dnsP->dn_rP +2,
			     (BYTE *)&dnsP->dn_nbuf[0], sizeof(dnsP->dn_nbuf));
	    if ( len < 0 )
		status = FALSE;
	    break;

	default:
	    status = FALSE;
	    break;
    }

    if ( status )
	dnsP->dn_rP += dnsP->dn_ansL;

    return ( status );
}
/*

*//* mdns_val_word(bP)

	Get a 16-bit (word) value from a buffer

Accepts :

	bP		Ptr to to the 2-byte value

Returns :

	<value>		the word value

Notes :


*/

static int
mdns_val_word ARGLIST( (bP) )
    FARG( UBYTE		*bP )		/* Ptr to 2 bytes */
{
	int		val;		/* Value to return */

    val = *bP++;
    val = (val << 8) + *bP;

    return ( val );
}
