/* mml-bio.c -- Basic buffered I/O functions for 'mml' package stuff.

	Copyright 2003,2004,2005,2006
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Contains a number of functions to accomplish basic buffered I/O.

*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/select.h>
#include <ctype.h>
#include <sys/stat.h>
#include <errno.h>

#include <mml/mml.h>

#include <mml/mml-alloc.h>
#include <mml/mml-in.h>
#include <mml/mml-bio.h>

/* Local definitions */



/* External data referenced */



/* External routines used */



/* Local data publicly available */



/* Local routines and forward references */



/* Private data */


/*

*//* bio_bufsiz_set( bioP, bufL )

	Specify the buffer size for an bio

Accepts :

	bioP		Ptr to BIO struct
	bufL		Desired buffer size

Returns :

	<nothing>

Notes :

	Buffer size may not be set once the buffer has been
	instantiated.  I guess we could make this work if we need to,
	but no need to yet.

	bufL is silently constrained to lie within BIO_BUFMIN
	and BIO_BUFMAX

*/

void
bio_bufsiz_set ARGLIST( (bioP, bufL) )
   NFARG( BIO		*bioP )		/* io struct */
    FARG( int		bufL )		/* Desired buffer size */
{
    if ( bioP->bio_bufP != NULL ) {
	return;				/* Buffer is already there. */
    }

    if ( bufL < BIO_BUFMIN )
	bufL = BIO_BUFMIN;
    else if ( bufL > BIO_BUFMAX )
	bufL = BIO_BUFMAX;

    bioP->bio_bufL = bufL;
}
/*

*//* bio_drain( bioP )

	Write any buffered data to BIO

Accepts :

	bioP		Pointer to BIO struct

Returns :

	<value>		TRUE if OK
			FALSE otherwise

Notes :


*/

BOOL
bio_drain ARGLIST( (bioP) )
    FARG( BIO	*bioP )		/* io struct */
{
	int		len;
	UBYTE		*bufP;

    if ( ( ( bufP = bioP->bio_bufP ) != NULL ) &&
         ( bioP->bio_bufC > 0 ) ) {

	/* Loop as necessary */
	while ( bioP->bio_bufC > 0 ) {
	    len = write( bioP->bio_fd, bufP, bioP->bio_bufC );
	    if ( len < 0 )
		return ( FALSE );
	    bufP += len;
	    bioP->bio_bufC -= len;
	}
    }

    return ( TRUE );
}
/*

*//* bio_fill( bioP )

	Read into underlying BIO input buffer

Accepts :

	bioP		Pointer to BIO struct

Returns :

	<value>		TRUE if OK
			FALSE otherwise

Notes :

	On FALSE, sets bio_result to indicate the cause
	  (and potentially bio_errno)

*/

BOOL
bio_fill ARGLIST( (bioP) )
    FARG( BIO	*bioP )		/* io struct */
{
	int		len;
	UBYTE		*bufP;
	fd_set		readfds;	/* fd set for select() */
	struct timeval	totv;		/* for select() timeout */

    if ( ( bufP = bioP->bio_bufP ) == NULL ) {
	if ( !bio_initbuf( bioP ) )
	    return ( FALSE );
	bufP = bioP->bio_bufP;
    }
    else {
	bioP->bio_bufX = 0;
    }

    /* If timeout wanted, go through select() first. */
    if ( bioP->bio_tosecs > 0 ) {
	FD_ZERO( &readfds );
	FD_SET( bioP->bio_fd, &readfds );
	totv.tv_sec = bioP->bio_tosecs;
	totv.tv_usec = 0;

	if ( select( bioP->bio_fd +1, &readfds, NULL, NULL, &totv ) < 0 ) {
	    /* Same as read error */
	    bioP->bio_bufC = 0;
	    bioP->bio_result = IOR_ERRNO;
	    bioP->bio_errno = errno;
	    return ( FALSE );
	}

	if ( !FD_ISSET( bioP->bio_fd, &readfds ) ) {
	    /* Timed out. */
	    bioP->bio_bufC = 0;
	    bioP->bio_result = IOR_TIMEOUT;
	    return ( FALSE );
	}
    }

    /* OK to try to read now. */
    len = read( bioP->bio_fd, bufP, bioP->bio_bufL );
    if ( len > 0 ) {
	bioP->bio_bufC = len;
	bioP->bio_result = IOR_NONE;
	return ( TRUE );
    }
    
    if ( len == 0 ) {
	bioP->bio_bufC = 0;
	bioP->bio_result = IOR_EOF;
    }
    else if ( len < 0 ) {
	bioP->bio_bufC = 0;
	bioP->bio_result = IOR_ERRNO;
	bioP->bio_errno = errno;
    }

    return ( FALSE );
}
/*

*//* bio_finit( bioP )

	Finalize a BIO structure

Accepts :

	bioP		Pointer to BIO struct

Returns :

	<nothing>

Notes :

	Cleans up any allocated structures for this BIO.  Does NOT
	  attempt to remove the BIO structure itself.

*/

void
bio_finit ARGLIST( (bioP) )
    FARG( BIO	*bioP )		/* io struct */
{
    /* Free any allocated buffer */
    if ( bioP->bio_bufP != NULL ) {
	dealloc( bioP->bio_bufP );
	bioP->bio_bufP = NULL;
    }
}
/*

*//* bio_getb( bioP )

	Input a byte via BIO

Accepts :

	bioP		Ptr to BIO struct

Returns :

	<value>		>= 0 : the byte;
			if error or other failure to get a byte.
			  the bioP->errret value or EOF.  See note.

Notes :

	Uses buffered I/O.

	On error, if bioP->errret is a negative value, that value is
	returned, otherwise CHV_EOD.  This helps this function
	to be used in an mml-in stream without further layering.

	On error return, the cause can be deduced by looking at
	the bio_result et al values.

*/

int
bio_getb ARGLIST( (bioP) )
    FARG( BIO		*bioP )		/* io struct */
{
	UBYTE		*bufP;

    if ( ( bufP = bioP->bio_bufP ) == NULL ) {
	if ( !bio_initbuf( bioP ) )
	    return ( ( bioP->bio_errret < 0 ) ?
		    bioP->bio_errret : CHV_EOF );
	bufP = bioP->bio_bufP;
    }

    /* Fill as needed */
    if ( bioP->bio_bufX == bioP->bio_bufC )
	if ( !bio_fill( bioP ) )
	    return ( ( bioP->bio_errret < 0 ) ?
		    bioP->bio_errret : CHV_EOF );

    return ( bufP[ bioP->bio_bufX++ ] );
}
/*

*//* bio_init( bioP, fd )

	Init(clear) an BIO struct.

Accepts :

	bioP		Pointer to BIO struct
	fd		File descriptor to associate

Returns :

	<nothing>

Notes :

	This function just sets the fd, and clears the rest of
	the struct for later use.

*/

void
bio_init ARGLIST( (bioP, fd) )
   NFARG( BIO		*bioP )		/* io struct */
    FARG( int		fd )		/* File descriptor */
{
    bioP->bio_fd = fd;		/* File descriptor */
    bioP->bio_bufP = NULL;	/* I/O buffer */
    bioP->bio_tosecs = 0;	/* Timeout in seconds */
    bioP->bio_errret = CHV_EOD; /* Return this on EOF or error */
    bioP->bio_bufL = 0;		/* Size of buffer (0 means use a default) */
    bioP->bio_bufC = 0;		/* Number of bytes in buffer */
    bioP->bio_bufX = 0;		/* Access point in buffer */
    bioP->bio_result = IOR_NONE;	/* Extended result info */
    bioP->bio_errno = 0;	/* Errno */
}
/*

*//* bio_initbuf( bioP )

	Initialize BIO buffer

Accepts :

	bioP		Pointer to BIO struct

Returns :

	<value>		TRUE if successful
			FALSE if not

Notes :


*/

BOOL
bio_initbuf ARGLIST( (bioP) )
    FARG( BIO	*bioP )		/* io struct */
{
	int		len;

    /* Only do this if there's no buffer already */
    if ( bioP->bio_bufP == NULL ) {
	len = bioP->bio_bufL;		/* Desired buffer size */
	if ( len == 0 )
	    len = BIO_BUFLEN;
	else if ( len < BIO_BUFMIN )	/* In case somebody set it badly */
	    len = BIO_BUFMIN;
	else if ( len > BIO_BUFMAX )
	    len = BIO_BUFMAX;

	bioP->bio_bufP = emalloc( "bio_initbuf", "buffer", len );
	if ( bioP->bio_bufP == NULL ) {
	    bioP->bio_result = IOR_FAIL;
	    return ( FALSE );
	}

	/* Set up buffering state */
	bioP->bio_bufL = len;
	bioP->bio_bufC = 0;
	bioP->bio_bufX = 0;
    }

    return ( TRUE );
}
/*

*//* bio_putb( bioP, bval )

	Output a byte via bio

Accepts :

	bioP		Ptr to BIO struct
	bval		Byte value to output

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	Uses buffered I/O.

*/

BOOL
bio_putb ARGLIST( (bioP, bval) )
   NFARG( BIO		*bioP )		/* io struct */
    FARG( UBYTE		bval )		/* byte value */
{
	UBYTE		*bufP;

    if ( ( bufP = bioP->bio_bufP ) == NULL ) {
	if ( !bio_initbuf( bioP ) )
	    return ( FALSE );
	bufP = bioP->bio_bufP;
    }

    if ( bioP->bio_bufC == bioP->bio_bufL )
	if ( !bio_drain( bioP ) )
	    return ( FALSE );

    bufP[ bioP->bio_bufC++ ] = bval;

    return ( TRUE );
}
/*

*//* bio_putn( bioP, bP, bC )

	Output a number of bytes via BIO

Accepts :

	bioP		Ptr to BIO struct
	bP		Ptr to bytes to output
	bC		Number of bytes

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	Uses buffered I/O.

*/

BOOL
bio_putn ARGLIST( (bioP, bP, bC) )
   NFARG( BIO		*bioP )		/* io struct */
   NFARG( UBYTE		*bP )		/* Bytes to write */
    FARG( int		bC )		/* Byte count */
{
    /* Could probably optimize this, but dumb implementation is
       just to loop and output via bio_putb.
    */
    while ( bC-- > 0 )
	if ( ! bio_putb( bioP, *bP++ ) )
	    return ( FALSE );

    return ( TRUE );
}
/*

*//* bio_putstr( bioP, bP )

	Output from a nul-terminated string

Accepts :

	bioP		Ptr to BIO struct
	bP		Ptr to bytes to output

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	Uses buffered I/O.

*/

BOOL
bio_putstr ARGLIST( (bioP, bP) )
   NFARG( BIO		*bioP )		/* io struct */
    FARG( UBYTE		*bP )		/* Bytes to write */
{
    /* As with bio_putn, this could probably be optimized */
    while ( *bP != NUL )
	if ( ! bio_putb( bioP, *bP++ ) )
	    return ( FALSE );

    return ( TRUE );
}
/*

*//* bio_timeout_set( bioP, tosecs )

	Specify a timeout for further I/O 

Accepts :

	bioP		Ptr to BIO struct
	tosecs		Timeout in seconds

Returns :

	<nothing>

Notes :

	Timeout is silently constrained to be non-negative.

*/

void
bio_timeout_set ARGLIST( (bioP, tosecs) )
   NFARG( BIO		*bioP )		/* io struct */
    FARG( int		tosecs )	/* Desired timeout, in seconds */
{
    bioP->bio_tosecs = ( tosecs >= 0 ) ? tosecs : 0;
}
/*

*//* bio_ungetb( bioP, bval )

	Push back a byte after fetching one e.g. via bio_getb()

Accepts :

	bioP		Ptr to BIO struct
	bval		Byte value to put back

Returns :

	<nothing>

Notes :

	This is only intended to work once, immediately after a byte has
	been fetched out of the BIO buffer.  It's possible for a caller to
	violate that assumption, and for the BIO to be in a state where a
	byte can't be pushed back-- too bad for them, we don't complain,
	but we may not do anything either.

*/

void
bio_ungetb ARGLIST( (bioP, bval) )
   NFARG( BIO		*bioP )		/* io struct */
    FARG( UBYTE		bval )		/* byte value */
{
	UBYTE		*bufP;

    if ( ( ( bufP = bioP->bio_bufP ) != NULL ) &&
	 ( bioP->bio_bufX > 0 ) )
	    bufP[ --bioP->bio_bufX ] = bval;
}
/*

*//* in_h_bio_getc( instrP )

	Stream handler for "bio": get char

Accepts :

	instrP		Ptr to the input stream

Returns :

	<value>		character input,
			EOF if end of string

Notes :

	This is a routine that is specified as a "get character"
	routine for a BIO stream (i.e., in in_stream_push).

	A BIO stream reads from BIO input, setting up the
	proper MML_IN result values on EOF.

*/

int
in_h_bio_getc ARGLIST( ( instrP ) )
    FARG( MML_INS	*instrP )	/* The input stream */
{
	int		ch;		/* Char */

    /* Get byte */
    ch = bio_getb( (BIO *)instrP->is_handleP );

    /* on EOF or other error, set up the input stack's return info */
    if ( ch < 0 ) {
	instrP->is_inP->in_result = ((BIO *)instrP->is_handleP)->bio_result;
	instrP->is_inP->in_errno = ((BIO *)instrP->is_handleP)->bio_errno;
    }

    return( ch );
}
/*

*//* in_stream_push_bio( inP, bioP, nameP )

	Push a BIO into the input stream

Accepts :

	inP		Ptr to input handle
	bioP		Ptr to BIO handle
	nameP		Name for this stream, or NULL if none

Returns :

	<value>		TRUE if OK
			FALSE if not.
			May take error exit on allocation failure.

Notes :

	Pushes the string onto the stream stack using standard BIO
	input function.

	nameP's string is copied to make a persistent copy.

	By default the BIO is initialized to return CHV_EOD for EOF,
	meaning that the stream won't pop itself on EOF.  Application
	can change the EOF character to EOF to change this.

*/

BOOL
in_stream_push_bio ARGLIST( (inP, bioP, nameP) )
   NFARG( MML_IN	*inP )		/* Input handle */
   NFARG( BIO		*bioP )		/* BIO handle */
    FARG( char		*nameP )	/* Name for the stream */
{
    /* Just push the BIO */
    return ( in_stream_push( inP, in_h_bio_getc, NULL, (void *)bioP, nameP ) );
}
