/* mml-alloc.c -- memory allocation routines for "mml" library

	Copyright 2003,2004,2005,2006
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Contains memory allocation routines.

Basically wrappers in front of system-level routines.

Mainly lifted from older mem library circa 1989

*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>

    /* Hack for data declared in this module */
#define	MML_ALLOC

#include <mml/mml.h>

#include <mml/mml-err.h>
#include <mml/mml-alloc.h>


/* Local definitions */

    /* Handles a re-use list (e.g. a bin) for a particular reusable size. */
typedef struct {
    void	*ru_listP;		/* The re-use list */
    int		ru_allocC;		/* # of times allocated from */
    int		ru_ruC;			/* # of times released */

#ifdef	MML_PTHREADS
    pthread_mutex_t ru_lock;		/* Access lock */
#endif	/* MML_PTHREADS */
}  RU_BIN;


#ifdef	MML_MALLOC_DEBUGGING
    /* Debugging header to add to memory chunk */
typedef struct alloc_dbg {
    struct alloc_dbg *mad_nextP;	/* Ptr to next one */
    struct alloc_dbg *mad_prevP;	/* Ptr to previous one */
    long	mad_seq;		/* Sequence number */
    char	*mad_facP;		/* Facility description */
    char	*mad_noteP;		/* Note/description */
    size_t	mad_size;		/* Size */
}  ALLOC_DBG;
#endif	/* MML_MALLOC_DEBUGGING */


/* External data referenced */



/* External routines used */



/* Local data publicly available */

	BOOL	Mml_malloc_verboseF;

/* Local routines and forward references */



/* Private data */


#ifdef	MML_PTHREADS
    /* For any random critsec */
static	pthread_mutex_t	Alloc_lock = PTHREAD_MUTEX_INITIALIZER;
#endif	/* MML_PTHREAD */

static	BOOL	Mml_alloc_initF;	/* TRUE if initialized */

    /* The memory bins for rualloc, indexed by chunk size factor (i.e.
       index 0 is for size 1*MML_ALLOC_ALIGN, index 2 for 2*, etc.)
       Indexes up to MML_ALLOC_MIN are wasted.
    */
static	RU_BIN	Ru_bins[ MML_ALLOC_RUMAX / MML_ALLOC_ALIGN ];


#ifdef	MML_MALLOC_DEBUGGING
static	ALLOC_DBG	*Dbg_headP;	/* Ptr to head of list */
static	ALLOC_DBG	*Dbg_tailP;	/* Ptr to tail */
#endif	/* MML_MALLOC_DEBUGGING */

/*

*//* dealloc( memP )

	deallocate previously allocated memory

Accepts :

	memP		Ptr to memory to free

Returns :

	< nothing >

Notes :

*/

void
dealloc ARGLIST( ( memP ) )
    FARG( void		*memP )		/* Ptr to memory to free */
{
#ifdef	MML_MALLOC_DEBUGGING
	ALLOC_DBG	*dbgP;
	ALLOC_DBG	*prevP, *nextP;
#endif	/* MML_MALLOC_DEBUGGING */

    if ( Mml_malloc_verboseF )
	printf( "dealloc %lx\n", memP );

#ifndef	MML_MALLOC_DEBUGGING
    free( memP );
#else
    if ( mml_mutex_lock( &Alloc_lock ) != 0 )
	warning( "dealloc: can't get Alloc_lock" );
    dbgP = (ALLOC_DBG *)((char *)memP - sizeof(ALLOC_DBG) );

    /* Remove from list */
    prevP = dbgP->mad_prevP;
    nextP = dbgP->mad_nextP;

    if ( prevP == NULL )
	/* Removing first node, update head */
	Dbg_headP = nextP;
    else
	/* Not removing first node, update previous node */
	prevP->mad_nextP = nextP;

    if ( nextP == NULL )
	/* Removing last node; update tail */
	Dbg_tailP = prevP;
    else
	/* There's a node following, update it to point back. */
	nextP->mad_prevP = prevP;

    free( dbgP );			/* Free it */
    if ( mml_mutex_unlock( &Alloc_lock ) != 0 )
	warning( "dealloc: can't release Alloc_lock" );
#endif	/* MML_MALLOC_DEBUGGING */

}
/*

*//* derualloc( memP, size )

	Release previously allocated memory for re-use

Accepts :

	memP		Ptr to memory to free
	size		The size of the memory

Returns :

	< nothing >

Notes :

	See erualloc() for other notes

	Unfortunately there is no portable way (that I know of) to obtain
	  the size of an allocated memory chunk, so caller has to pass it
	  when releasing it.  This should not be much of a burden since
	  the point is to re-use known-size objects.  But it would be
	  nice not to have to depend on the caller.

*/

void
derualloc ARGLIST( ( memP, size ) )
   NFARG( void		*memP )		/* Ptr to memory to free */
    FARG( size_t	size )		/* Size */
{
	int		binX;		/* Which bin (index) */
	RU_BIN		*binP;		/* Ptr to the bin */

    if ( size <= 0 ) {
	warning( "derualloc: bad size %d", size );
	return;
    }

    /* Just release it if it's larger than our re-use max */
    if ( size > MML_ALLOC_RUMAX ) {
	dealloc( memP );
	return;
    }

    /* Make sure inited */
    if ( ! ( Mml_alloc_initF || mml_alloc_init() ) )
	return;

    if ( size < MML_ALLOC_MIN ) {
	/* I'd like to do something special with small chunks, like allocate
	   a bigger section and parcel it up.  Some other time.
	*/
	size = MML_ALLOC_MIN;
    }

    /* Normalize size (round up to next alignment boundary) */
    size = ( size + MML_ALLOC_ALIGN -1 ) & ~(MML_ALLOC_ALIGN_MASK);

    /* Access the specific re-use bin */
    binX = size >> MML_ALLOC_ALIGN_SHIFT;
    if ( binX < 0 )
	return;
    binP = &Ru_bins[binX];

    /* Link the new chunk in at the front of the list */
    if ( mml_mutex_lock( &binP->ru_lock ) != 0 )
	warning( "derualloc: can't get lock" );
    *(void **)memP = binP->ru_listP;
    binP->ru_listP = memP;
    ++binP->ru_ruC;
    if ( mml_mutex_unlock( &binP->ru_lock ) != 0 )
	warning( "derualloc: can't release lock" );

    /* done */
}
/*

*//* emalloc( facP, noteP, size )

	Allocate memory, give error if failure.

Accepts :

	facP		Name of facility (e.g., subroutine name)
	noteP		A descriptive note about the thing being allocated
	size		Desired size

Returns :

	< value >	Ptr to thing.

Notes :

	error return is taken if allocation fails.

	size will be forced to a minimum.

*/

void *
emalloc ARGLIST( ( facP, noteP, size ) )
    NFARG( char		*facP )		/* Ptr to facility name */
    NFARG( char		*noteP )	/* Ptr to note for thing */
    FARG( size_t	size )		/* Number of bytes */
{
	void		*newP;		/* Ptr to memory */

#ifdef	MML_MALLOC_DEBUGGING
	ALLOC_DBG	*dbgP;
#endif	/* MML_MALLOC_DEBUGGING */

    /* Make sure inited */
    if ( ! ( Mml_alloc_initF || mml_alloc_init() ) )
	return ( NULL );

    if ( size < MML_ALLOC_MIN ) {
	/* I'd like to do something special with small chunks, like allocate
	   a bigger section and parcel it up.  Same comment as elsewhere.
	*/
	size = MML_ALLOC_MIN;
    }

    /* Normalize size (round up to next alignment boundary) */
    size = ( size + MML_ALLOC_ALIGN -1 ) & ~(MML_ALLOC_ALIGN_MASK);

#ifdef	MML_MALLOC_DEBUGGING
    size += sizeof(ALLOC_DBG);		/* May play havoc with chunks */
#endif	/* MML_MALLOC_DEBUGGING */

    if ( ( newP = (void *)malloc( size ) ) == NULL ) {
	error( "%s: Error allocating %d bytes for %s",
		 facP, size, noteP );
	return ( NULL );
    }

    if ( mml_mutex_lock( &Alloc_lock ) != 0 )
	warning( "emalloc: can't get Alloc_lock" );
    ++Mml_emallocC;
    if ( Mml_malloc_verboseF ) {
	printf( "emalloc: got %lx fac=\"%s\" note=\"%s\" size=%d\n",
	         newP, facP, noteP, size );
    }

#ifdef	MML_MALLOC_DEBUGGING
    /* Set up the debugging header and link it into the list */
    dbgP = (ALLOC_DBG *)newP;
    newP = (void *)((char *)newP + sizeof(ALLOC_DBG));

    dbgP->mad_seq = Mml_emallocC;
    dbgP->mad_facP = facP;
    dbgP->mad_noteP = noteP;
    dbgP->mad_size = size;

    if ( ( dbgP->mad_prevP = Dbg_tailP ) != NULL )
	/* Link old last node to this new last node */
	(dbgP->mad_prevP)->mad_nextP = dbgP;
    else
	/* List was empty, so new node is also head node */
	Dbg_headP = dbgP;

    Dbg_tailP = dbgP;
    dbgP->mad_nextP = NULL;

#endif	/* MML_MALLOC_DEBUGGING */

    if ( mml_mutex_unlock( &Alloc_lock ) != 0 )
	warning( "emalloc: can't release Alloc_lock" );

    return( newP );
}
/*

*//* erealloc( facP, noteP, oldP, size )

	Reallocate memory, give error if failure.

Accepts :

	facP		Name of facility (e.g., subroutine name)
	noteP		A descriptive note about the thing being allocated
	oldP		Ptr to old thing (may be NULL)
	size		Desired size

Returns :

	< value >	Ptr to thing.

Notes :

	error return is taken if allocation fails.
*/

void *
erealloc ARGLIST( ( facP, noteP, oldP, size ) )
    NFARG( char		*facP )		/* Ptr to facility name */
    NFARG( char		*noteP )	/* Ptr to name of thing */
    NFARG( void		*oldP )		/* Ptr to existing thing */
    FARG( size_t	size )		/* Number of bytes */
{
	void		*newP;		/* Ptr to memory */

#ifdef	MML_MALLOC_DEBUGGING
	ALLOC_DBG	*dbgP;
#endif	/* MML_MALLOC_DEBUGGING */

    /* Make sure inited */
    if ( ! ( Mml_alloc_initF || mml_alloc_init() ) )
	return ( NULL );

    if ( mml_mutex_lock( &Alloc_lock ) != 0 )
	warning( "erealloc: can't get Alloc_lock" );
    ++Mml_ereallocC;
    if ( mml_mutex_unlock( &Alloc_lock ) != 0 )
	warning( "emalloc: can't release Alloc_lock" );

    if ( oldP == NULL ) {
	if ( Mml_malloc_verboseF )
	    printf( "erealloc->emalloc\n" );
	newP = (void *)emalloc( facP, noteP, size );
    }
    else {
	if ( size < MML_ALLOC_MIN ) {
	    /* I'd like to do something special with small chunks, like
	       allocate a bigger section and parcel it up.  Have we heard
	       this before?
	    */
	    size = MML_ALLOC_MIN;
	}

	/* Normalize size (round up to next alignment boundary) */
	size = ( size + MML_ALLOC_ALIGN -1 ) & ~(MML_ALLOC_ALIGN_MASK);

#ifndef	MML_MALLOC_DEBUGGING
	newP = (void *)realloc( oldP, size );

#else	/* MML_MALLOC_DEBUGGING */
	/* Must reallocate the actual chunk, and adjust the list pointers */
	if ( mml_mutex_lock( &Alloc_lock ) != 0 )
	    warning( "erealloc: can't get Alloc_lock" );

	dbgP = (ALLOC_DBG *)((char *)oldP - sizeof(ALLOC_DBG) );
	size += sizeof(ALLOC_DBG);
	dbgP = (ALLOC_DBG *)realloc( dbgP, size );

	if ( dbgP == NULL )
	    newP = NULL;
	else {
	    /* Adjust the neighbor's links to the new node */
	    if ( dbgP->mad_nextP != NULL )
		(dbgP->mad_nextP)->mad_prevP = dbgP;
	    else
		Dbg_tailP = dbgP;

	    if ( dbgP->mad_prevP != NULL )
		(dbgP->mad_prevP)->mad_nextP = dbgP;
	    else
		Dbg_headP = dbgP;

	    newP = (void *)((char *)dbgP + sizeof(ALLOC_DBG));
	}

	if ( mml_mutex_unlock( &Alloc_lock ) != 0 )
	    warning( "erealloc: can't release Alloc_lock" );
#endif	/* MML_MALLOC_DEBUGGING */

	if ( Mml_malloc_verboseF ) {
	    printf( "erealloc: got %lx fac=\"%s\" note=\"%s\" size=%d\n",
	         newP, facP, noteP, size );
	}
    }


    if ( newP == NULL )
	error( "%s: Error re-allocating %d bytes for %s",
		 facP, size, noteP );

    return( newP );
}
/*

*//* erualloc( facP, noteP, size )

	Allocate memory, first trying re-used chunks; give error if failure.

Accepts :

	facP		Name of facility (e.g., subroutine name)
	noteP		A descriptive note about the thing being allocated
	size		Desired size

Returns :

	< value >	Ptr to thing.

Notes :

	error return is taken if allocation fails.

	size will be forced to a minimum and to a chunk size boundary.

	This function will reuse previously released (via derualloc())
	  memory if possible.  erualloc/derualloc should be used
	  for items that are known to be recycled a lot (e.g. linked
	  list nodes).

	erualloc and emalloc allocations are completely compatible.
	  An allocation obtained via emalloc may be released via
	  derualloc for later re-use.  It's very tempting to seed the
	  erualloc bins with larger allocations divvied up, but then
	  that compatibility would be broken.

*/

void *
erualloc ARGLIST( ( facP, noteP, size ) )
    NFARG( char		*facP )		/* Ptr to facility name */
    NFARG( char		*noteP )	/* Ptr to note for thing */
     FARG( size_t	size )		/* Number of bytes */
{
	int		binX;		/* Which bin (index) */
	void		*newP;		/* Ptr to memory */
	RU_BIN		*binP;		/* Ptr to the bin */

    /* Make sure inited */
    if ( ! ( Mml_alloc_initF || mml_alloc_init() ) )
	return ( NULL );

    if ( size < MML_ALLOC_MIN ) {
	/* I'd like to do something special with small chunks, like allocate
	   a bigger section and parcel it up.  Same comment as elsewhere.
	*/
	size = MML_ALLOC_MIN;
    }

    /* Normalize size (round up to next alignment boundary) */
    size = ( size + MML_ALLOC_ALIGN -1 ) & ~(MML_ALLOC_ALIGN_MASK);

    /* Access the specific re-use bin */
    binX = size >> MML_ALLOC_ALIGN_SHIFT;
    if ( binX < 0 )
	return;
    binP = &Ru_bins[binX];

    /* Check the list */
    if ( mml_mutex_lock( &binP->ru_lock ) != 0 )
	warning( "erualloc: can't get bin lock" );
    newP = binP->ru_listP;
    if ( newP != NULL ) {
	binP->ru_listP = *(void **)newP;
	++binP->ru_allocC;
    }
    if ( mml_mutex_unlock( &binP->ru_lock ) != 0 )
	warning( "erualloc: can't release bin lock" );

    /* If nothing gained there, use regular emalloc */
    if ( newP == NULL )
	newP = emalloc( facP, noteP, size );

    return( newP );
}
/*

*//* mml_alloc_dump(ofP)

	Dump the malloc list to a file

Accepts :

	ofP		FILE ptr to dump the list to

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :


*/

#ifdef	MML_MALLOC_DEBUGGING

BOOL
mml_malloc_dump ARGLIST( (ofP) )
    FARG( FILE		*ofP )		/* File to output to */
{
	BOOL		okF;
	ALLOC_DBG	*dbgP;

    for ( okF = TRUE, dbgP = Dbg_headP;
		okF && ( dbgP != NULL );
			dbgP = dbgP->mad_nextP ) {

	okF = okF && ( fprintf( ofP, "%lx seq=%ld facP=\"%s\" note=\"%s\" size=%ld\n",
			       dbgP,
			       dbgP->mad_seq,
			       dbgP->mad_facP,
			       dbgP->mad_noteP,
			       dbgP->mad_size ) > 0 );
    }

    return ( okF );
}

#endif	/* MML_MALLOC_DEBUGGING */

/*

*//* mml_alloc_init()

	Initialize the mml-alloc module

Accepts :

	<nothing>

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	Application can call this at its start; however, mml-alloc
	  will self-init as needed.

	Will simply return TRUE if already initialized.

*/

BOOL
mml_alloc_init NOARGLIST
{
	int		i;

    if ( mml_mutex_lock( &Alloc_lock ) != 0 )
	warning( "mml_alloc_init: can't get Alloc_lock" );

    if ( ! Mml_alloc_initF ) {
	/* Initialize the reuse bins */
	for ( i = 0; i < ( MML_ALLOC_RUMAX / MML_ALLOC_ALIGN ); ++i ) {
	    Ru_bins[i].ru_listP = NULL;
	    Ru_bins[i].ru_allocC = 0;
	    Ru_bins[i].ru_ruC = 0;
	    if ( mml_mutex_init( &Ru_bins[i].ru_lock ) != 0 )
		warning( "mml_alloc_init: can't init bin %d lock", i );
	}

	Mml_alloc_initF = TRUE;
    }
    if ( mml_mutex_unlock( &Alloc_lock ) != 0 )
	warning( "mml_alloc_init: can't release Alloc_lock" );

    return ( TRUE );
}
/*

*//* newstr( facP, nameP, oldP )

	Allocate a new copy of a string

Accepts :

	facP		Name of facility (e.g., subroutine name)
	nameP		Name of item being allocated
	oldP		Ptr to old string to duplicate

Returns :

	< value >	Ptr to new string

Notes :

	Hands off to newstr_n, see that.

*/

char *
newstr ARGLIST( ( facP, nameP, oldP ) )
   NFARG( char		*facP )		/* Ptr to facility name */
   NFARG( char		*nameP )	/* Ptr to name of thing */
    FARG( char		*oldP )		/* Ptr to existing string */
{
    return ( newstr_n( facP, nameP, oldP, strlen( oldP ) ) );
}
/*

*//* newstr_n( facP, nameP, oldP, len )

	Allocate a new nul-terminated string from a counted string

Accepts :

	facP		Name of facility (e.g., subroutine name)
	nameP		Name of item being allocated
	oldP		Ptr to old string to duplicate
	len		Number of bytes in the old string

Returns :

	< value >	Ptr to new string

Notes :

	Uses emalloc, which implies that error() is called is taken if
	allocation fails.

	I suppose this might belong in the "mml-str" module.
	Hard to know.

*/

char *
newstr_n ARGLIST( ( facP, nameP, oldP, len ) )
   NFARG( char		*facP )		/* Ptr to facility name */
   NFARG( char		*nameP )	/* Ptr to name of thing */
   NFARG( char		*oldP )		/* Ptr to existing string */
    FARG( int		len )		/* Length of existing string */
{
	char		*newP;		/* Ptr to new string */

    newP = (char *)emalloc( facP, nameP, len +1 );
    memcpy( newP, oldP, len );
    newP[len] = NUL;

    return( newP );
}
