/* mml-cgi.c -- functions to help with Apache CGI web interface

	Copyright 1994,2003,2004,2005,2006
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Contains some functions for a web server program to use with an Apache-style
CGI interface.

*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <string.h>

#include <mml/mml.h>

#include <mml/mml-alloc.h>
#include <mml/mml-cgi.h>

/* Local Definitions */

typedef					/* Query argument block */
  struct qarg {
    struct qarg		*q_nextP;	/* Ptr to next one */
    char		*q_tagP;	/* Query argument tag */
    char		*q_valP;	/* Query argument value */
  }  QARG;


/* External data referenced */


/* External routines used */

extern	long		atol();

/* Local data publicly available */


/* Local routines and forward references */

static	int	hexch_val PROTO( (int ch) );
static	char	*get_qsubstring PROTO( (char **bufPP, int delim) );
static	QARG	*qarg_enter PROTO( (char *catP, char *grpP) );
static	QARG	*qarg_find PROTO( (char *catP) );
static	char	*qargs_in_buf PROTO( (char *bufP) );
static	char	*qargs_in_get PROTO( (void) );
static	char	*qargs_in_post PROTO( (void) );

/* Private data */


static	char	Errmsg[100];		/* Error message buffer */
static	QARG	*QargP;			/* List of qargs */


/*

*//* qargs_in( method )

	Get query arguments (that is, incoming arguments)

Accepts :

	method		"GET" to accept get method;
			"POST" to accept post method;
			NULL to accept any supported method


Returns :

	< value >	NULL if OK
			ptr to error message string if not

*/

char *
qargs_in ARGLIST( (methodP) )
    FARG( char		*methodP )	/* Acceptable method(s) */
{
	char		*rq_methodP;	/* Actual request method */

    if ( ( ( rq_methodP = getenv( "REQUEST_METHOD" ) ) == NULL ) ||
	 ( *rq_methodP == NUL ) )
	rq_methodP = "<unspecified>";

    if ( stricmp( rq_methodP, "GET" ) == 0 ) {
	if ( ( methodP != NULL ) && ( stricmp( methodP, "GET" ) != 0 ) ) {
	    if ( snprintf( &Errmsg[0], sizeof( Errmsg ),
			 "Method is \"%s\", but only \"%s\" allowed.",
			 rq_methodP, methodP ) >= sizeof(Errmsg) ) {
		snprintf( &Errmsg[0], sizeof( Errmsg ),
			 "Method must be \"GET\" (but is not)." );
	    }
	    return ( &Errmsg[0] );
	}
	return ( qargs_in_get() );
    }
    else if ( stricmp( rq_methodP, "POST" ) == 0 ) {
	if ( ( methodP != NULL ) && ( stricmp( methodP, "GET" ) != 0 ) ) {
	    if ( snprintf( &Errmsg[0], sizeof( Errmsg ),
			   "Method is \"%s\", but only \"%s\" allowed.",
			   rq_methodP, methodP ) >= sizeof(Errmsg) ) {
		snprintf( &Errmsg[0], sizeof( Errmsg ),
			 "Method must be \"POST\" (but is not)." );
	    }
	    return ( &Errmsg[0] );
	}
	return ( qargs_in_post() );
    }
    else {
	if ( snprintf( &Errmsg[0], sizeof( Errmsg ),
			"qargs_in: request method \"%s\" not supported.",
			rq_methodP ) >= sizeof(Errmsg) ) {
	    snprintf( &Errmsg[0], sizeof( Errmsg ),
		     "qargs_in: unknown request method." );
	}
	return ( &Errmsg[0] );
    }
}
/*

*//* qarg_find_tag( tagP )

	Find handle for specific tag

Accepts :

	tagP		Tag string


Returns :

	< value >	Ptr to handle for first matching one
			or NULL if none found

*/

void *
qarg_find_tag ARGLIST( (tagP) )
    FARG( char		*tagP )		/* The tag */
{
    return ((void *)qarg_find( tagP ));
}
/*

*//* qarg_first()

	Return first tag registered

Accepts :


Returns :

	< value >	Handle for first thing
			NULL if none

*/

void *
qarg_first NOARGLIST
{
    return (( void *)QargP );
}
/*

*//* qarg_next( qP )

	Get tag following the given one

Accepts :

	qP		Ptr to handle

Returns :

	< value >	Handle of next one
			NULL if not found

*/

void *
qarg_next ARGLIST( (qP) )
    FARG( void		*qP )		/* The handle */
{
    return ( (void *)((QARG *)qP)->q_nextP );
}
/*

*//* qarg_put( tagP, valP )

	Access routine: put a tag/value pair into the taglist

Accepts :

	tagP		Ptr to tag string to enter
	valP		Ptr to value string to enter

Returns :

	< value >	TRUE if OK
			FALSE if not

Notes :

	Copies of the tag and the value are made.

	Caller should ensure that tag doesn't already exist.

*/

BOOL
qarg_put ARGLIST( (tagP, valP) )
   NFARG( char		*tagP )		/* Tag to enter */
    FARG( char		*valP )		/* Value to enter */
{
    /* Make copies of the input strings */
    tagP = newstr( "qarg_put", "tag", tagP );
    if ( tagP == NULL )
	return ( FALSE );

    valP = newstr( "qarg_put", "value", valP );
    if ( valP == NULL ) {
	dealloc( tagP );
	return ( FALSE );
    }
    
    return ( qarg_enter( tagP, valP ) != NULL );
}
/*

*//* qarg_tag and qarg_value

	Access routines for tag and value of handle

*/

char *
qarg_tag ARGLIST( (qP) )
    FARG( void		*qP )		/* Handle */
{
    return ( ((QARG *)qP)->q_tagP );
}


char *
qarg_value ARGLIST( (qP) )
    FARG( void		*qP )		/* Handle */
{
    return ( ((QARG *)qP)->q_valP );
}
/*

*//* qarg_value_set( qP, valP )

	Replace the value for tag handle/value pair

Accepts :

	qP		The semi-opaque handle
	valP		Ptr to new value

Returns :

	<value>		pointer to the copy of the value, if successful
			NULL if not

Notes :

	A copy of the value is made herein.

*/

char *
qarg_value_set ARGLIST( (qP, valP) )
   NFARG( void		*qP )		/* Handle */
    FARG( char		*valP )		/* Value */
{
    /* Make a copy of the value */
    valP = newstr( "qarg_value_set", "value", valP );
    if ( valP == NULL )
	return ( valP );

    /* Deallocate any old copy, and replace with the new one */
    if ( ((QARG *)qP)->q_valP != NULL )
	dealloc( ((QARG *)qP)->q_valP );
    ((QARG *)qP)->q_valP = valP;

    return ( valP );
}

/***********************************************************************
 *                                                                     *
 *                     Internal support routines                       *
 *                                                                     *
 ***********************************************************************/

/*

*//* qargs_in_get()

	Get args via "GET" method

Accepts :


Returns :

	< value >	NULL if OK
			ptr to error message string if not

*/

static char *
qargs_in_get NOARGLIST
{
	char		*qsP;		/* Query string */

    qsP = getenv( "QUERY_STRING" );
    if ( ( qsP == NULL ) || ( *qsP == NUL ) )
#if 0
	return ( "No query string provided!" );
#else
	/* We're going to allow this: caller can trap the abscense of
	   required query tags.
	*/
	return ( NULL );
#endif

    /* Interpret the query */
    return ( qargs_in_buf( qsP ) );
}
/*

*//* qargs_in_post()

	Get args via "POST" method

Accepts :


Returns :

	< value >	NULL if OK
			ptr to error message string if not

*/

static char *
qargs_in_post NOARGLIST
{
	int		ch;
	long		byteC;		/* # of bytes in content */
	long		byteX;		/* Current byte # */
	char		*clP;		/* Content length value string */
	char		*cvP;		/* Content value string */
	char		*respP;		/* Response */

    /* Get length of input data */
    clP = getenv( "CONTENT_LENGTH" );
    if ( ( clP == NULL ) || ( !isdigit( *clP ) ) )
	return ( "POST method: no CONTENT_LENGTH given." );

    byteC = atol( clP );
    cvP = emalloc( "qargs_in_post", "cvp", byteC + 1 );
    if ( cvP == NULL )
	return ( "POST method: can't allocate content buffer." );

    /* Accept the content into the buffer */
    for ( byteX = 0; byteX < byteC; ++byteX ) {
	if ( ( ch = getchar() ) == EOF ) {
	    /* Short input-- can't really complain anywhere, just take it. */
	    break;
	}
	cvP[byteX] = ch;
    }

    cvP[byteX] = NUL;

#if 0
printf( "<p>Text received:<p>\n<pre>\n%s\n</pre>\n", cvP );
#endif

    /* Interpret the arg string. */
    respP = qargs_in_buf( cvP );

    /* Lose the allocated buffer, and return. */
    dealloc( cvP );

    return ( respP );
}
/*

*//* qargs_in_buf( bufP )

	Get args from a buffer

Accepts :

	bufP		Ptr to the buffer

Returns :

	< value >	NULL if OK
			ptr to error message string if not

*/

static char *
qargs_in_buf ARGLIST( (char *bufP) )
    FARG( char		*bufP )		/* Buffer to get args from */
{
	char		ch;
	char		*tagP;
	char		*valP;

    /* Loop for argument pairs */
    for ( ; ; ) {
	/* Skip leading whitespace and cr/lfs */
	for ( ; ( ch = *bufP ) != NUL; ++bufP )
	    if ( ( ch != ' ' ) &&
	         ( ch != '\t' ) &&
		 ( ch != '\r' ) &&
		 ( ch != '\n' ) )
		break;

	/* be done if done */
	if ( *bufP == NUL )
	    break;

	/* Collect the tag substring */
	tagP = get_qsubstring( &bufP, '=' );

	/* Collect the arg substring */
	valP = get_qsubstring( &bufP, '&' );

	/* enter this tag/value pair */
	if ( qarg_enter( tagP, valP ) == NULL )
	    return ( "qarg_enter failure." );
    }
	
    return ( NULL );

}
/*

*//* get_qsubstring( bufPP, delim )

	Get optionally quoted substring

Accepts :

	bufPP		Ptr to the buffer pointer
	delim		Character that terminates substring

Returns :

	< value >	Safe copy of the substring extracted
	*bufPP		Updated to one past the delimiter found

Notes :

	If no delimiter found, one is imagined right before the
	terminating NUL character.

*/

static char *
get_qsubstring ARGLIST( (bufPP, delim) )
   NFARG( char		**bufPP )	/* Buffer ptr ptr */
    FARG( int		delim )		/* Delimiter */
{
	char		ch;
	char		qch;
	char		state;
	int		byteC;		/* # of bytes in substring */
	int		ssX;
	int		bufX;
	char		*bufP;		/* Actual buffer ptr */
	char		*tailP;		/* Ptr to tail */
	char		*ssP;		/* Substring ptr */

    /* Text states */
#define	QS_TEXT		0
#define	QS_QUOTE1	1		/* First char of quoted hex char */
#define	QS_QUOTE2	2		/* Second char of quoted hex char */

    bufP = *bufPP;

    /* Scope out the substring */
    tailP = strchr( bufP, delim );
    if ( tailP != NULL )
	byteC = tailP++ - bufP;		/* Point tail beyond delimiter */
    else {
	byteC = strlen( bufP );
	tailP = bufP + byteC;
    }

    /* Allocate a buffer large enough to hold this substring, on the
       theory that any quoting will compress rather than expand the
       actual text.
    */
    ssP = emalloc( "get_qsubstring", "substring", byteC +1 );
    if ( ssP == NULL )
	exit( 1 );			/* What else to do.. */

    state = QS_TEXT;
    for ( bufX = ssX = 0; bufX < byteC; ++bufX ) {
	ch = bufP[bufX];
	if ( ( ch == '\r' ) || ( ch == '\n' ) )
	    continue;
	switch ( state ) {
	case QS_TEXT:
	    if ( ch == '%' )
		state = QS_QUOTE1;
	    else if ( ch == '+' )
		ssP[ssX++] = ' ';
	    else
		ssP[ssX++] = ch;
	    break;

	case QS_QUOTE1:
	    qch = ( hexch_val( ch ) ) << 4;
	    state = QS_QUOTE2;
	    break;

	case QS_QUOTE2:
	    qch |= hexch_val( ch );
	    ssP[ssX++] = qch;
	    state = QS_TEXT;
	    break;
	}
    }

    ssP[ssX] = NUL;

    *bufPP = tailP;

    return ( ssP );
}
/*

*//* qarg_enter( tagP, valP )

	Enter a qarg block

Accepts :

	tagP		Tag name
	valP		Value string

Returns :

	<value>		Ptr to the QARG block entered.

Notes :

	It's assumed that the passed strings are "safe" -- that is,
	this routine doesn't make a copy of them.

*/

static QARG *
qarg_enter ARGLIST( (tagP, valP) )
   NFARG( char		*tagP )		/* Tag name */
    FARG( char		*valP )		/* Value string */
{
	int		match;
	QARG		*qargP;		/* Block ptr */
	QARG		**qargPP;	/* Link ptr */

    for ( qargPP = &QargP; (qargP = *qargPP) != NULL;
		qargPP = &qargP->q_nextP ) {
	match = stricmp( tagP, qargP->q_tagP );
#if 0  /* duplicates ok */
	if ( match == 0 ) {
	    /* Found match: give warning and return */
	    fprintf( stderr, "qarg_enter: tag \"%s\" already entered.",
			tagP );
	    return( qargP );
	}
#endif
	if ( match < 0 )
	    break;
    }

    /* Make new qarg block and link it in */
    qargP = (QARG *)emalloc( "qarg_enter", "qarg block",
    			sizeof(QARG) );
    qargP->q_tagP = tagP;
    qargP->q_valP = valP;

    qargP->q_nextP = *qargPP;
    *qargPP = qargP;

    return ( qargP );
}
/*

*//* qarg_find( tagP )

	Find a qarg block for a given tag

Accepts :

	tagP		tag to look for

Returns :

	< value >	Ptr to QARG if found
			NULL if not

*/

static QARG *
qarg_find ARGLIST( (tagP) )
    FARG( char		*tagP )		/* Tag looking for */
{
	QARG		*qargP;

    /* Look for the block */
    for ( qargP = QargP; qargP != NULL; qargP = qargP->q_nextP )
	if ( stricmp( tagP, qargP->q_tagP ) == 0 )
	    return ( qargP );

    return ( NULL );
}
/*

*//* hexch_val( ch )

	Return the value described by a hex ASCII character

Accepts :

	ch		A char with a hex representative value

Returns :

	< value >	The described value
			or -1 if invalid hex ASCII char.

*/

static int
hexch_val ARGLIST( (ch) )
    FARG( int		ch )	/* The char */
{
    if ( ( ch >= '0' ) && ( ch <= '9' ) )
	return ( ch - '0' );
    if ( ( ch >= 'A' ) && ( ch <= 'F' ) )
	return ( ch + 10 - 'A' );
    if ( ( ch >= 'a' ) && ( ch <= 'f' ) )
	return ( ch + 10 - 'a' );
    return ( -1 );
}
