/* mml-in.c -- mml library (mml) input stack stuff

	Copyright 2003,2004,2005,2006
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

mml-in contains a set of functions providing for layered input,
as for a command flow that can include nested inputs (from files
or other sources).

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include <mml/mml.h>

#include <mml/mml-err.h>
#include <mml/mml-alloc.h>
#include <mml/mml-in.h>

/* Local definitions */



/* External data referenced */



/* External routines used */



/* Local data publicly available */



/* Local routines and forward references */

    /* Allocate a cfile (counted file) stream handle */
static	IN_CFILE *cfile_alloc PROTO( (void) );

    /* Release a cfile stream handle */
static	void	cfile_free PROTO( (IN_CFILE *cfileP) );

    /* Derive input stack results from an input file's status */
static	void	in_file_results PROTO( (MML_IN *inP, FILE *fP) );

    /* Allocate a stream handle */
static	MML_INS	*instr_alloc PROTO( (void) );

    /* Release a stream handle */
static	void	instr_free PROTO( (MML_INS *instrP) );

    /* Allocate a string stream handle */
static	IN_SSTR	*sstr_alloc PROTO( (void) );

    /* Release a string stream handle */
static	void	sstr_free PROTO( (IN_SSTR *sstrP) );


/* Private data */


/*

*//* in_char( inP )

	Inputs the next character from an input stack

Accepts :

	inP		Ptr to input handle

Returns :

	<value>		the character
			may be EOF

Notes :

	The underlying input stream shall return the non-negative
	  character code for a valid character read.

	For an invalid input, the underlying input stream must set up the
	  extended result information (is_result etc) in the input stream
	  structure, indicating the reason that there is no more data.
	  Any non-EOF condition will remain sticky unless the application
	  resets it.  i.e., if an error is indicated, all further calls
	  to in_char() will return EOF until the extended result
	  indicator is cleared.  In the end-of-data case, the underlying
	  input stream must return one of the following character codes:

	    CHV_EOF (or EOF); results in popping the stream.

	    CHV_EOD; same as EOF, but does not result in popping
	      the stream.

	  In all error cases, this function will return EOF.
	    
*/

int
in_char ARGLIST( (inP) )
    FARG( MML_IN	*inP )		/* Input handle */
{
	int		ch;		/* Character */
	MML_INS		*instrP;	/* Ptr to input stream */

    /* Clear the "last was escaped" flag, as we are going to return
       a raw character here.
    */
    inP->in_lescF = FALSE;

    /* Loop trying.. */
    for ( ; ; ) {
	/* If the saved result status indicates something abnormal,
	   return EOF (errors are sticky until cleared).
	*/
	switch ( inP->in_result ) {
		/* These are all OK, meaning that we can proceed */
	    case IOR_NONE:		/* Nothing in particular */
	    case IOR_TIMEOUT:		/* Operation timed out */
	    case IOR_EOF:		/* End of file */
		break;

		/* These indicate a previous failure, so we fail now too */
	    case IOR_FAIL:		/* Some internal failure */
	    case IOR_ERRNO:		/* As indicated by errno */
		return ( EOF );

	    default:
		warning( "in_char, in_result has bad value %d", inP->in_result );
		return ( EOF );
	}

	/* If no more streams: return EOF */
	if ( ( instrP = inP->in_streamP ) == NULL ) {
	    inP->in_result = IOR_EOF;
	    return ( EOF );
	}

	/* Check for pushed-back character */
	if ( ( ch = instrP->is_pbc ) >= 0 ) {
	    instrP->is_pbc = -1;	/* Clear for next time */
	}
	else {
	    /* Fetch next character from this stream */
	    ch = (*instrP->is_getcP)( instrP );
	}

	if ( ch >= 0 ) {		/* Valid character? */
	    /* Iff last char was end of line, this is beginning of line */
	    inP->in_bolF = inP->in_eolF;

	    /* Check for end of line */
	    if ( ch == '\n' ) {
		inP->in_eolF = TRUE;	/* Set end of line flag */

		/* Bump line number, after caching prior value. */
		inP->in_clineN = instrP->is_lineN++;
	    }
	    else
		inP->in_eolF = FALSE;
	    break;
	}

	/* Got a special character- proceed accordingly. */
	if ( ch == CHV_EOF ) {
	    /* Standard EOF.  POP the stream and try the next one. */
	    in_stream_pop( inP );
	}
	else if ( ch == CHV_EOD ) {
	    /* EOF but don't pop the stream */
	    ch = EOF;
	    break;
	}
	else {
	    /* Dunno what this is-- pop the stream, since it will probably
	       keep happening otherwise, and give EOF.
	    */
	    warning( "in_char got bad code %d from stream.", ch );
	    in_stream_pop( inP );
	    ch = EOF;
	    break;
	}
    }

    /* Got here with a character to return. */
    return ( ch );
}
/*

*//* in_charback( inP, ch )

	Push a character back onto the stream

Accepts :

	inP		Ptr to input handle
	ch		The character to push back

Returns :

	<value>		the character

Notes :

	The pushed back character is associated with the current level.
	Thus if you push another stream after pushing back a char, the
	pushed-back char won't be seen again until the input streams have
	popped back to this level.

	Pushing back EOF is a no-op.  This allows caller to pushback
	any previously-gotten character without worrying about the
	stream being empty on EOF.  However it is otherwise an error
	to attempt to push back to an empty stream.

	There is only a single pushback level.  Multiple pushbacks will
	overwrite.  

*/

void
in_charback ARGLIST( (inP, ch) )
   NFARG( MML_IN	*inP )		/* Input handle */
    FARG( int		ch )		/* Character to push back */
{
    if ( ch != EOF ) {
	if ( inP->in_streamP == NULL )
	    error( "in_charback: pushback to empty stream." );
	else {
	    inP->in_streamP->is_pbc = ch;  /* Put char back on stream */

	    /* We're at end of line iff last char was at beginning of line */
	    inP->in_eolF = inP->in_bolF;

	    /* If we are pushing back an EOL character, un-count the line number */
	    if ( ch == '\n' )
		--inP->in_streamP->is_lineN;

	    /* Note: we can't really recreate the beginning of line
	       status, but that should not be an issue.  Nobody should
	       be pushing back a character and then checking that flag.
	    */
	}
    }
}
/*

*//* in_charesc( inP, rawP )

	Inputs the next character from an input stack, with escapes

Accepts :

	inP		Ptr to input handle
	rawP		Where to put the raw text if escaped
			NULL if none wanted

Returns :

	<value>		the character
			may be EOF

Notes :

	Like in_char, but interprets various escaped sequences

	Raw buffer doesn't have to be very big, but should be at
	least 5 bytes.

*/

int
in_charesc ARGLIST( (inP, rawP) )
   NFARG( MML_IN	*inP )		/* Input handle */
    FARG( char		*rawP )		/* Where to put raw input */
{
	int		ch;		/* Character */
	int		i;
	int		v;
	char		rawbuf[10];	/* Fake raw buffer */

    /* If no raw buf was wanted, use a local one and just lose it. */
    if ( rawP == NULL )
	rawP = &rawbuf[0];

    /* There's a little ambiguity here: if there was a pushedback
       character, was it already escaped?  (Especially if it was
       a backslash..)  We assume that the caller is operating
       in one mode or the other, so we use the in_lescF as a hint.

    */
    if ( inP->in_lescF ) {
	/* Last char was escaped- so assume that any pushed back
	   character was the escaped character. */
	if ( ( ch = inP->in_streamP->is_pbc ) >= 0 ) {
	    inP->in_streamP->is_pbc = -1;
	    inP->in_lescF = FALSE;

	    /* The only real problem is we've probably lost the raw
	       buffer.  Let's just simulate it and hope in this
	       case it's not an issue.  This is too weird to fix
	       unless it actually happens.
	    */
	    sprintf( rawP, "\\0%o", ch );
	    return ( ch );
	}
    }

    /* Get next char from stream.  Note that this clears in_lescF */
    ch = in_char( inP );
    *rawP++ = ch;

    if ( ch == '\\' ) {			/* Raw char.. */
	ch = in_char( inP );		/* Get next one */
	*rawP++ = ch;			/* We'll probably keep it here */

	switch( ch ) {			/* Process.. */

	    case '0':			/* Octal ASCII encoding */
		for( i = 0, v=0; i < 4; ++i ) {
		    if ( !isdigit(ch) ) {
			in_charback( inP, ch );
			--rawP;
			break;
		    }
		    v = ( v << 3 ) + ( ch - '0' );
		    ch = in_char( inP );
		    *rawP++ = ch;
		}
		ch = v;			/* Character assumes value */
		break;


	    case 'a':			/* Bell */
		ch = '\007';
		break;

	    case 'e':			/* ASCII Escape */
		ch = '\033';
		break;

	    case 'n':			/* Newline */
		ch = '\n';
		break;

	    case 'r':			/* Carriage return */
		ch = '\r';
		break;

	    case 't':			/* Tab */
		ch = '\011';
		break;

	    case '\r':			/* Carriage return */
		/* Flush any newline after it */
		if ( ( ch = in_char( inP ) ) != '\n' )
		    in_charback( inP, ch );
		else
		    *rawP++ = ch;

		/* Drop through to newline handling */

	    case '\n':			/* Newline */
		ch = ' ';		/* Turn it into a space */
		break;

	    default:			/* Anything else */
		break;			/*  stays the same */

	}

	inP->in_lescF = TRUE;		/* Remember this was escaped */
    }

    *rawP = NUL;			/* Terminate the raw buffer */
    return( ch );
}
/*

*//* in_destroy(inP)

	Destroy an input stack

Accepts :

	inP		Ptr to MML_IN input stack

Returns :

	<nothing>

Notes :

	Will call in_stream_pop to exhaust the stack, but
	otherwise does not delete the input stream(s).

*/

void
in_destroy ARGLIST( (inP) )
    FARG( MML_IN	*inP )		/* Ptr to input stack */
{
    /* Pop all the remaining streams */
    while ( inP->in_streamP != NULL )
	in_stream_pop( inP );

    /* Free up this stack */
    dealloc( inP );
}
/*

*//* in_h_cfile_finish( instrP )

	Stream handler for "counted file": finish the stream

Accepts :

	instrP		Ptr to the input stream

Returns :

	<nothing>

Notes :

	This is a routine that is specified as a "finish" routine for a
	counted file stream (i.e., in in_stream_push).

	This finish function gets rid of the intermediate control
	structure that is used for counted file parameters.  It does
	NOT close the file.

*/

void
in_h_cfile_finish ARGLIST( ( instrP ) )
    FARG( MML_INS	*instrP )	/* The input stream */
{
	int		ch;		/* Char */
	IN_CFILE	*cfileP;	/* String stream ptr */

    cfileP = (IN_CFILE *)instrP->is_handleP;

    /* Just release the object */
    cfile_free( cfileP );

    /* done */
}
/*

*//* in_h_cfile_getc( instrP )

	Stream handler for "counted file": get char

Accepts :

	instrP		Ptr to the input stream

Returns :

	<value>		character input,
			EOF if end of string

Notes :

	This is a routine that is specified as a "get character"
	routine for a counted file stream (i.e., in in_stream_push).
	A counted file stream inputs from a file but only up to a
	maximum number of bytes.

*/

int
in_h_cfile_getc ARGLIST( ( instrP ) )
    FARG( MML_INS	*instrP )	/* The input stream */
{
	int		ch;		/* Char */
	IN_CFILE	*cfileP;	/* String stream ptr */

    cfileP = (IN_CFILE *)instrP->is_handleP;

    /* Get character */
    if ( cfileP->cf_byteC-- <= 0 ) {
	/* Return EOF, seting status to EOF */
	instrP->is_inP->in_result = IOR_EOF;
	ch = EOF;
    }
    else {
	ch = getc( cfileP->cf_fP );
	if ( ch == EOF ) {
	    /* Set the input results */
	    in_file_results( instrP->is_inP, cfileP->cf_fP );
	}
    }

    return( ch );
}
/*

*//* in_h_file_finish( instrP )

	Stream handler for "file": finish the stream

Accepts :

	instrP		Ptr to the input stream

Returns :

	<nothing>

Notes :

	This is a routine that is specified as a "finish"
	function for a file stream (i.e., in in_stream_push).

	This finish function closes the file.

*/

void
in_h_file_finish ARGLIST( ( instrP ) )
    FARG( MML_INS	*instrP )	/* The input stream */
{
	int		ch;		/* Char */

    /* Close the file */
    fclose( (FILE *)instrP->is_handleP );

    /* done */
}
/*

*//* in_h_file_getc( instrP )

	Stream handler for "file": get char

Accepts :

	instrP		Ptr to the input stream

Returns :

	<value>		character input,
			EOF if end of file

Notes :

	This is a routine that is specified as a "get character"
	routine for a file stream (i.e., in in_stream_push).  A
	file stream inputs from a file through EOF.

*/

int
in_h_file_getc ARGLIST( ( instrP ) )
    FARG( MML_INS	*instrP )	/* The input stream */
{
	int		ch;		/* Char */
	FILE		*fP;		/* Ptr to input file */

    if ( ( ch = getc( (FILE *)instrP->is_handleP ) ) == EOF ) {
	/* End of input: set the input results */
	in_file_results( instrP->is_inP, (FILE *)instrP->is_handleP );
    }

    return ( ch );			/* Return what we got */
}
/*

*//* in_h_sstr_finish( instrP )

	Stream handler for "string": finish the stream

Accepts :

	instrP		Ptr to the input stream

Returns :

	<nothing>

Notes :

	This is a routine that is specified as a "finish" function
	for a string stream (i.e., in in_stream_push).

	This finish function gets rid of the IN_SSTR object, which
	may also deallocate the string buffer (depending on information
	in the string object).

*/

void
in_h_sstr_finish ARGLIST( ( instrP ) )
    FARG( MML_INS	*instrP )	/* The input stream */
{
    /* Release the string object */
    sstr_free( (IN_SSTR *)instrP->is_handleP);

    /* done */
}
/*

*//* in_h_sstr_getc( instrP )

	Stream handler for "string": get char

Accepts :

	instrP		Ptr to the input stream

Returns :

	<value>		character input,
			EOF if end of string

Notes :

	This is a routine that is specified as a "get character"
	routine for a string stream (i.e., in in_stream_push).

	A string stream inputs from a character array in memory,
	as described by an IN_SSTR object that is provided as the
	stream handle.

*/

int
in_h_sstr_getc ARGLIST( ( instrP ) )
    FARG( MML_INS	*instrP )	/* The input stream */
{
	int		ch;		/* Char */
	IN_SSTR		*sstrP;		/* String stream ptr */

    sstrP = (IN_SSTR *)instrP->is_handleP;

    /* Get character */
    if ( sstrP->cs_bufX < sstrP->cs_strL )
	ch = sstrP->cs_bufP[ sstrP->cs_bufX++ ];
    else {
	ch = EOF;			/* End of string */
	instrP->is_inP->in_result = IOR_EOF;	/* End result is EOF */
    }

    return( ch );
}
/*

*//* in_lineno( inP )

	Get the line number associated with the input

Accepts :

	inP		Ptr to MML_IN input stack

Returns :

	<value>		The line number if any stream is active,
			  0 otherwise

Notes :

	The line number may be out of sync with the data recently fetched,
	e.g. in the case where a stream has been automatically popped by the
	last input sequence.  The caller probably wants the line number of
	the popped stream, but will not get it in that case.  We attempt
	to cache the previously seen line number whenever it changes;
	this cached number will be returned if there is no active stream.

*/

int
in_lineno ARGLIST( (inP) )
    FARG( MML_IN	*inP )		/* Ptr to input stack */
{
    if ( inP->in_streamP != NULL )
	return ( inP->in_streamP->is_lineN );

    /* No active stream, return the cached line number. */
    return ( inP->in_clineN );
}
/*

*//* in_name( inP )

	Get the name associated with the current input stream

Accepts :

	inP		Ptr to MML_IN input stack

Returns :

	<value>		The name associated with the active stream,
			  some static string value otherwise.

Notes :

	As with in_lineno, the name may be out of sync with the data
	recently fetched, e.g. in the case where a stream has been
	automatically popped by the last input sequence.  There is
	no workaround for this.

	Caller should treat the returned string as very temporary;
	it may not survive the next input operation.

*/

char *
in_name ARGLIST( (inP) )
    FARG( MML_IN	*inP )		/* Ptr to input stack */
{
	char		*nameP;

    if ( ( inP->in_streamP == NULL ) ||
         ( inP->in_streamP->is_nameP == NULL ) )
	nameP = "<unnamed>";		/* No name. */
    else
	nameP = inP->in_streamP->is_nameP;

    return ( nameP );
}
/*

*//* in_new()

	Create a new input stack

Accepts :



Returns :

	<value>		Ptr to new MML_IN object if successful
			FALSE if not.
			(may take error exit)

Notes :


*/

MML_IN *
in_new NOARGLIST
{
	MML_IN		*inP;		/* Ptr to new thing */

    /* Allocate new structure */
    inP = (MML_IN *)emalloc( "in_new", "input stack object",
				sizeof(MML_IN) );

    /* Init the object */
    inP->in_lescF = FALSE;		/* Last char was not escaped */
    inP->in_eolF = TRUE;		/* Start of stream- sort of the
					   end of the zeroth line
					*/
    inP->in_bolF = FALSE;		/* N/A here really but set it */
    inP->in_streamP = NULL;		/* No current stream */
    inP->in_level = 0;			/* Level 0 */
    inP->in_clineN = 0;			/* Cached line number */
    inP->in_result = IOR_NONE;		/* IO result */
    inP->in_errno = 0;			/* errno */

    return ( inP );
}
/*

*//* in_nonblank( inP, eolF )

	Input the next non-blank (non-whitespace) character.

Accepts :

	inP		Ptr to MML_IN input handle
	eolF		TRUE if end-of-line is considered whitespace

Returns :

	<value>		character input,
			-1 if EOF or error

Notes :

	The notion of "non-blank" is hardwired herein , except for
	the eol flag.  An enhancement might be to associate the list
	of whitespace characters with the input pointer.

	Carriage-return is always considered to be blank.

*/

int
in_nonblank ARGLIST( ( inP, eolF ) )
   NFARG( MML_IN	*inP )		/* Input handle */
    FARG( BOOL		eolF )		/* TRUE if end of line is whitespace */
{
	int		ch;

    /* Just skip until non-whitespace, and return what we get. */
    for ( ; ; ) {
	ch = in_char( inP );
	if ( ( ch != ' ' ) &&
	     ( ch != '\t' ) &&
	     ( ch != '\r' ) &&
	     ( !eolF || ( ch != '\n' ) ) )
	    break;
    }

    return( ch );
}
/*

*//* in_stream_location( inP )

	Return a string representing the location in the stack

Accepts :

	inP		Ptr to MML_IN input handle

Returns :

	<value>		Address of a string describing the location

Notes :

	Returns a string like:
	    line 51 in file foo.command
	by searching for the topmost named stream in the stack.

	Returns some default like "[none]" if there is no named stream.

*/

char *
in_stream_location ARGLIST( ( inP ) )
    FARG( MML_IN	*inP )		/* Input handle */
{
	int		len;		/* Length of buffer required */
	char		*nameP;		/* Ptr to name of input */
	MML_INS		*instrP;	/* Stream */
static	char		*namebufP;	/* Buffer to use */
static	char		namebufL = 0;	/* Allocated size of buffer */

    /* Find the topmost stream that has a name */
    for ( instrP = inP->in_streamP; instrP != NULL;
	     instrP = instrP->is_prevP )
	if ( instrP->is_nameP != NULL )
	    break;

    if ( instrP == NULL )
	return ( "[none]" );
    if ( instrP != NULL )
	nameP = instrP->is_nameP;

    /* Calculate required buffer length.  This has to cover the
       format used..
    */
    len = 20 + strlen( nameP );

    /* Make sure our allocated buffer handles this.  This is very
       hackish.
    */
    if ( len > namebufL ) {
	if ( namebufL == 0 )
	    namebufP = emalloc( "in_stream_location", "buffer", len + 20 );
	else
	    namebufP = erealloc( "in_stream_location", "buffer", 
				    namebufP, len + 20 );
	if ( namebufP == NULL ) {
	    namebufL = 0;
	    return( "[error in in_stream_location]" );
	}
	namebufL = len + 20;
    }

    /* Use snprintf out of extreme paranoia */
    snprintf( namebufP, namebufL, "line %ld in %s",
		instrP->is_lineN, nameP );

    return ( namebufP );
}
/*

*//* in_stream_pop( inP )

	Remove input stream and make previous stream current.

Accepts :

	inP		Ptr to MML_IN input handle

Returns :

	<nothing>

Notes :

*/

void
in_stream_pop ARGLIST( ( inP ) )
    FARG( MML_IN	*inP )		/* Input handle */
{
	MML_INS		*instrP;	/* Input stream */

    instrP = inP->in_streamP;
    if ( instrP == NULL )
	error( "in_stream_pop: attempt to pop empty stream." );
    else {
	/* Pop to previous stream */
	inP->in_streamP = instrP->is_prevP;
	inP->in_lescF = FALSE;		/* Clear last-was-escaped */
	--inP->in_level;		/* Decrement level */

	/* If there's a stream finalizing function, call it */
	if ( instrP->is_finishP != NULL )
	    (*instrP->is_finishP)( instrP );

	/* Free up this stream but only if it's part of our pool. */
	if ( instrP->is_poolF )
	    instr_free( instrP );
    }
}
/*

*//* in_stream_push( inP, getcP, finishP, handleP, nameP )

	Push a new input stream onto an input


Accepts :

	inP		Ptr to input handle
	getcP		Routine to call to fetch a char from this stream
	finishP		Routine to call to finish the handle
	handleP		Handle associated with the stream (passed to
			the input routine)
	nameP		A descriptive name for the stream, for whatever
			purpose, or NULL if no name.

Returns :

	<value>		TRUE if OK
			FALSE if not.
			May take error exit on allocation failure.

Notes :

	nameP's string is copied to make a persistent copy.

	The finish function, if non-null, will be called when the
	  stream is popped.  The handle will be given as its only
	  argument.

	the character input routine is expected to close the input
	prior to returning EOF.

*/

BOOL
in_stream_push ARGLIST( (inP, getcP, finishP, handleP, nameP) )
   NFARG( MML_IN	*inP )		/* Input handle */
   NFARG( int		(*getcP)() )	/* Character input routine */
   NFARG( void		(*finishP)() )	/* Stream finish function */
   NFARG( void		*handleP )	/* Handle for the input stream */
    FARG( char		*nameP )	/* Name for the stream */
{
	MML_INS		*instrP;	/* Input stream thing */

    /* Create new input stream object */
    if ( ( instrP = instr_alloc() ) == NULL )
	return ( FALSE );

    /* Initialize.. */
    instrP->is_inP = inP;		/* Ptr to the stack object */
    instrP->is_pbc = -1;		/* No pushed back char */
    instrP->is_getcP = getcP;		/* Input routine */
    instrP->is_finishP = finishP;	/* Handle-finish function */
    instrP->is_handleP = handleP;	/* Handle */
    if ( nameP != NULL )		/* Copy the name.. */
	instrP->is_nameP = newstr( "in_stream_push", "name", nameP );
					/* Ignore error, not fatal
					   even if not great */
    else
	instrP->is_nameP = NULL;

    instrP->is_lineN = 1;		/* Line # 0 */
    inP->in_clineN = 1;			/* Cached value too */

    /* Hook in the new stream */
    instrP->is_prevP = inP->in_streamP;	/* Remember previous stream */
    inP->in_streamP = instrP;		/* Point to this one */
    ++inP->in_level;			/* Bump level count */

    inP->in_lescF = FALSE;		/* Last char not escaped */
    inP->in_result = IOR_NONE;		/* Pushing resets the result */
    inP->in_errno = 0;			/* And what the hey */

    return ( TRUE );
}
/*

*//* in_stream_push_cfile( inP, fP, byteC, nameP )

	Push a "counted file" into the input stream

Accepts :

	inP		Ptr to input handle
	fP		Ptr to input file
	byteC		Maximum number of bytes to read
	nameP		Name for this stream, or NULL if none

Returns :

	<value>		TRUE if OK
			FALSE if not.
			May take error exit on allocation failure.

Notes :

	Pushes the "counted file" onto the stream stack using the
	in_getc_cfile() input function.

	nameP's string is copied to make a persistent copy.

	File will be NOT be closed when EOF is reached.

	A "counted file" input stream is one that reads from a file up
	to a maximum number of bytes.  This is useful for reading from
	a specific range of a file (rather than through the EOF).
	Caller would seek to the start point and specify the number of
	bytes to read.

*/

BOOL
in_stream_push_cfile ARGLIST( (inP, fP, byteC, nameP) )
   NFARG( MML_IN	*inP )		/* Input handle */
   NFARG( FILE		*fP )		/* File handle */
   NFARG( off_t		byteC )		/* Maximum # of bytes to read */
    FARG( char		*nameP )	/* Name for the stream */
{
	IN_CFILE	*cfileP;	/* cfile stream handle */

    /* Make the new string stream object */
    if ( ( cfileP = cfile_alloc() ) == NULL )
	return ( FALSE );

    /* Fill in the handle */
    cfileP->cf_fP = fP;			/* The file ptr */
    cfileP->cf_byteC = byteC;		/* # of bytes remaining */

    return ( in_stream_push( inP, in_h_cfile_getc, in_h_cfile_finish,
				(void *)cfileP, nameP ) );
}
/*

*//* in_stream_push_file( inP, fP, nameP )

	Push a new file stream onto the input, close on EOF


Accepts :

	inP		Ptr to input handle
	fP		Ptr to open file handle
	nameP		Name for this stream, or NULL if none

Returns :

	<value>		TRUE if OK
			FALSE if not.
			May take error exit on allocation failure.

Notes :

	Pushes a file stream using standard input functions.

	nameP's string is copied to make a persistent copy.

	File will be closed when EOF is reached (or, more accurately,
	  when the stream is popped).

*/

BOOL
in_stream_push_file ARGLIST( (inP, fP, nameP) )
   NFARG( MML_IN	*inP )		/* Input handle */
   NFARG( FILE		*fP )		/* File pointer */
    FARG( char		*nameP )	/* Name for the stream */
{
    return ( in_stream_push( inP, in_h_file_getc, in_h_file_finish,
			      (void *)fP, nameP ) );
}
/*

*//* in_stream_push_file_nc( inP, fP, nameP )

	Push a new file stream onto the input, don't close on EOF


Accepts :

	inP		Ptr to input handle
	fP		Ptr to open file handle
	nameP		Name for this stream, or NULL if none

Returns :

	<value>		TRUE if OK
			FALSE if not.
			May take error exit on allocation failure.

Notes :

	Pushes a file stream using standard input functions.

	nameP's string is copied to make a persistent copy.

	File will be NOT closed when EOF is reached.  This is accomplished
	  by using not specifying a 'finish' handler for the stream.

*/

BOOL
in_stream_push_file_nc ARGLIST( (inP, fP, nameP) )
   NFARG( MML_IN	*inP )		/* Input handle */
   NFARG( FILE		*fP )		/* File pointer */
    FARG( char		*nameP )	/* Name for the stream */
{
    return ( in_stream_push( inP, in_h_file_getc, NULL,
			      (void *)fP, nameP ) );
}
/*

*//* in_stream_push_newstr( inP, strP, strL, nameP )

	Push a copy of a string into the input stream


Accepts :

	inP		Ptr to input handle
	strP		Ptr to string buffer
	strL		Length of string
	nameP		Name for this stream, or NULL if none

Returns :

	<value>		TRUE if OK
			FALSE if not.
			May take error exit on allocation failure.

Notes :

	Pushes the string onto the stream stack using standard string
	character input function.

	The string is copied to make a persistent copy.

	nameP's string is copied to make a persistent copy.

	File will be closed when EOF is reached.

*/

BOOL
in_stream_push_newstr ARGLIST( (inP, strP, strL, nameP) )
   NFARG( MML_IN	*inP )		/* Input handle */
   NFARG( char		*strP )		/* Ptr to string */
   NFARG( int		strL )		/* String length */
    FARG( char		*nameP )	/* Name for the stream */
{
	IN_SSTR		*sstrP;		/* String stream handle */

    /* Allocate the handle for the string */
    if ( ( sstrP = sstr_alloc() ) == NULL )
	return ( FALSE );

    /* Make a copy of the string */
    sstrP->cs_bufP = emalloc( "in_stream_push_newstr", "string", strL +1 );
    if ( sstrP->cs_bufP == NULL ) {
	sstr_free( sstrP );
	return ( FALSE );
    }

    /* Fill in the handle */
    sstrP->cs_bufX = 0;			/* Current fetch index */
    sstrP->cs_strL = strL;		/* Length of string */
    sstrP->cs_bufallocF = TRUE;		/* String must be freed */

    /* Copy the string */
    memcpy( sstrP->cs_bufP, strP, strL );

    return ( in_stream_push( inP, in_h_sstr_getc, in_h_sstr_finish,
			     (void *)sstrP, nameP ) );
}
/*

*//* in_stream_push_str( inP, strP, strL, nameP )

	Push a string into the input stream


Accepts :

	inP		Ptr to input handle
	strP		Ptr to string buffer
	strL		Length of string
	nameP		Name for this stream, or NULL if none

Returns :

	<value>		TRUE if OK
			FALSE if not.
			May take error exit on allocation failure.

Notes :

	Pushes the string onto the stream stack using standard string
	character input function.

	The string is used by reference, and thus must remain
	persistent throughout the life of the stream.

	nameP's string is copied to make a persistent copy.

*/

BOOL
in_stream_push_str ARGLIST( (inP, strP, strL, nameP) )
   NFARG( MML_IN	*inP )		/* Input handle */
   NFARG( char		*strP )		/* Ptr to string */
   NFARG( int		strL )		/* String length */
    FARG( char		*nameP )	/* Name for the stream */
{
	IN_SSTR		*sstrP;		/* String stream handle */

    /* Make the new string stream object */
    if ( ( sstrP = sstr_alloc() ) == NULL )
	return ( FALSE );

    /* Fill in the handle */
    sstrP->cs_bufX = 0;			/* Current fetch index */
    sstrP->cs_strL = strL;		/* Length of string */
    sstrP->cs_bufP = (UBYTE *)strP;	/* Where the string is */

    /* sstr_alloc clears cs_bufallocF */

    return ( in_stream_push( inP, in_h_sstr_getc, in_h_sstr_finish,
			     (void *)sstrP, nameP ) );
}
/*

*//* in_token_ws( inP, bufP, bufL, eolF )

	Get a token delimited by whitespace from an input stream.

Accepts :

	inP		Input stream
	bufP		Where to put the token
	bufL		Length of buffer for token
	eolF		If end of line chars are considered whitespace

Returns :

	<value>		Number of bytes in source token

Notes :

	Token grabber that only recognizes whitespace as token separators.

	The returned byte count may be larger than the buffer size
	provided; this would indicate that the returned token is
	truncated.

	In any case, the returned token is always NUL-terminated.
	Thus the maximum token length returned is one less than the
	buffer length specified.


*/

int
in_token_ws ARGLIST( (inP, bufP, bufL, eolF) )
   NFARG( MML_IN	*inP )		/* Input stream */
   NFARG( UBYTE		*bufP )		/* Where to put the token */
   NFARG( int		bufL )		/* Size of buffer */
    FARG( BOOL		eolF )		/* Whether eol chars are whitespace */
{
	int		ch;
	int		cC;

    --bufL;				/* Account for trailing NUL */

    /* Skip whitespace prior to token */
    ch = in_nonblank( inP, eolF );

    /* Grab characters until we see more whitespace */
    for ( cC = 0; ; ++cC ) {
	if ( ( ch == EOF ) ||
	     ( ch == ' ' ) ||
	     ( ch == '\t' ) ||
	     ( eolF && ( ch == '\r' ) || ( ch == '\n' ) ) )
	    break;

	if ( cC < bufL )
	    *bufP++ = ch;

	ch = in_char( inP );
    }

    *bufP = NUL;

    return ( cC );
}


/***********************************************************************
 *                                                                     *
 *                     Internal support routines                       *
 *                                                                     *
 ***********************************************************************/


/*

*//* in_file_results( inP, fP )

	Derive input results from an input file's status

Accepts :

	inP		Ptr to input stack
	fP		Ptr to input file

Returns :

	<nothing>

Notes :

	This function is to be called when an EOF is received on an
	  input file.  It determines the status of the file input
	  (truly EOF, or error) and sets up the input stack's
	  status information accordingly.

*/

static void
in_file_results ARGLIST( (inP, fP) )
   NFARG( MML_IN	*inP )		/* Ptr to input stack */
    FARG( FILE		*fP )		/* FILE handle */
{
    /* Check EOF */
    if ( feof( fP ) ) {
	inP->in_result = IOR_EOF;
    }

    /* Check error */
    else if ( ferror( fP ) ) {
	inP->in_result = IOR_ERRNO;
	inP->in_errno = errno;
    }

    else {
	/* Well, this shouldn't happen.  Warn and set EOF. */
	warning( "in_file_results: not EOF, not error.  setting EOF." );
	inP->in_result = IOR_EOF;
    }

    /* done */
}
/*

*//* instr_alloc()

	Allocate a stream structure

Accepts :



Returns :

	<value>		Ptr to MML_INS structure allocated
			NULL if failure (or error exit)

Notes :

	Uses erualloc in order to re-use another structure that
	  might have been returned (and not freed)

*/

static MML_INS *
instr_alloc NOARGLIST
{
	MML_INS		*instrP;		/* Ptr to new object */

    /* Get one off of a free list, or allocate */
    instrP = (MML_INS *)erualloc( "instr_alloc", "new stream",
					sizeof( MML_INS) );
    if ( instrP != NULL )
	instrP->is_poolF = TRUE;	/* Should be returned to the pool. */

    return ( instrP );
}
/*

*//* instr_free( instrP )

	Free a previously allocated stream handle

Accepts :

	instrP		Ptr to handle to free

Returns :

	<nothing>


Notes :

	Uses derualloc in order to maintain a free pool.

	Any name pointed to is freed here.

*/

static void
instr_free ARGLIST( (instrP) )
    FARG( MML_INS	*instrP )	/* stream handle to release */
{
    /* Release any name that was allocated */
    if ( instrP->is_nameP != NULL ) {
	dealloc( instrP->is_nameP );
	instrP->is_nameP = NULL;
    }

    /* Return this for re-use */
    derualloc( instrP, sizeof( MML_INS ) );

    /* Done */
}
/*

*//* cfile_alloc()

	Allocate a counted file stream structure

Accepts :


Returns :

	<value>		Ptr to IN_CFILE structure allocated
			NULL if failure (or error exit)

Notes :


*/

static IN_CFILE *
cfile_alloc NOARGLIST
{
    /* Return from the re-use list, or allocate */
    return ( (IN_CFILE *)erualloc( "cfile_alloc", "cfile handle",
					sizeof( IN_CFILE) ) );
}
/*

*//* cfile_free( cfileP )

	Free a previously allocated cfile handle

Accepts :

	cfileP		Ptr to handle to free

Returns :

	<nothing>


Notes :

	Keeps free handles on an internal list for quick re-allocation.

*/

static void
cfile_free ARGLIST( (cfileP) )
    FARG( IN_CFILE	*cfileP )	/* cfile handle to release */
{
    /* Release it for re-use */
    derualloc( cfileP, sizeof( IN_CFILE ) );
}
/*

*//* sstr_alloc()

	Allocate a string stream structure

Accepts :



Returns :

	<value>		Ptr to IN_SSTR structure allocated
			NULL if failure (or error exit)

Notes :

	The "buffer is allocated" flag (cs_bufallocF) is preset to false.

	Uses erualloc() in order to rapidly re-use memory that has
	  been returned (rather than freed)

*/

static IN_SSTR *
sstr_alloc NOARGLIST
{
	IN_SSTR		*sstrP;		/* Ptr to new object */

    /* Get new one */
    sstrP = (IN_SSTR *)erualloc( "sstr_alloc", "sstr handle",
					sizeof( IN_SSTR) );
    if ( sstrP != NULL )
	sstrP->cs_bufallocF = FALSE;	/* Caller assumes this is cleared */

    return ( sstrP );
}
/*

*//* sstr_free( sstrP )

	Free a previously allocated stream handle

Accepts :

	sstrP		Ptr to handle to free

Returns :

	<nothing>


Notes :

	Uses derualloc() to keeps free handles around for quick
	  re-allocation later via erualloc().

*/

static void
sstr_free ARGLIST( (sstrP) )
    FARG( IN_SSTR	*sstrP )	/* stream handle to release */
{
    /* If the buffer was allocated, free it now. */
    if ( sstrP->cs_bufallocF )
	dealloc( sstrP->cs_bufP );

    /* Release for re-use */
    derualloc( sstrP, sizeof( IN_SSTR) );

    /* done */
}
