/*	avl.c		AVL routines

	Copyright 2003,2004,2005,2006,2007
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Supersedes:
	Copyright 1988,1989,1990 Zinn Computer Company
			by Mark E. Mallett

	This is a general-purpose implementation of AVL trees in C.
It is derived from the description of AVL (Adelson-Velskii and Landis)
trees found in Knuth's "The Art of Computer Programming Volume 3:
Searching and Sorting" (Addison-Wesley, 1973) pgs 451-471.

	The routines in this file deal with tree maintenance only.
These routines are only concerned with the arrangement of the nodes
in the tree.  Composition of those nodes is left to outside knowledge.
The caller must construct an AVL tree header structure which specifies
routines that deal with nodes (comparison, construction, and deletion).
Please see the attached document for further information.

*/

#include <stdlib.h>
#include <stdio.h>

#include <mml/mml.h>

#include <mml/mml-alloc.h>
#include <mml/mml-avl.h>


/* Local definitions */

    /* Structure of node for key-only string trees */
typedef struct {
    AVLNODE	sn_avl;		/* AVL node header */
    char	*sn_strP;	/* Pointer to the string */
}  STRNODE;


/* External routines */


/* External data */


/* Local routines */

static	int	balance PROTO( (AVLNODE **branchPP) );
static	int	delete PROTO( (AVLTREE *treeP, AVLNODE **topPP, void *keyP) );
static	void	deltree PROTO( (AVLTREE *treeP, AVLNODE *nodeP) );
static	int	walk PROTO( (AVLTREE *treeP, AVLNODE *nodeP,
			     int (*rtc)(), int depth) );

/* Local data */

/*

*//* avlcount( treeP )

	Return the count of nodes in a tree

Accepts :

	treeP		Ptr to tree header

Returns :

	Number of nodes in the tree; -1 if error (treeP is null)

*/

int
avlcount ARGLIST( ( treeP ) )
    FARG( AVLTREE	*treeP )	/* Tree ptr */
{
    return( treeP == NULL ? -1 : treeP->t_nodeC );
}
/*

*//* avldelete( treeP, keyP )

	Delete a node from an AVL tree, locking the tree during deletion.

Accepts:

	treeP		Address of the tree definition structure.
	keyP		Address of the key for the node.  Interpretation
			 of the key is left to the key-compare routine
			 specified in the tree header.

Returns :

 	<value>		0 or 1 if deleted OK
			-1 if not found

Notes :

	A locking wrapper around avldelete_nolock().

	If the tree's lock can not be aquired, no attempt is made to
	  delete the node, and -1 is returned.

*/

int
avldelete ARGLIST ( ( treeP, keyP ) )
   NFARG( AVLTREE	*treeP )	/* Addr of tree block */
    FARG( void		*keyP )		/* Addr of key */
{
	int		r;

    if ( treeP == NULL )
	r = -1;

    else if ( mml_mutex_lock( &treeP->t_lock ) != 0 ) {
	warning( "avldelete: can't get lock" );
	r = -1;
    }
    else {
	r = avldelete_nolock( treeP, keyP );

	if ( mml_mutex_unlock( &treeP->t_lock ) != 0 )
	    warning( "avldelete: can't release lock" );
    }

    return( r );			/* done */
}
/*

*//* avldelete_nolock( treeP, keyP )

	Delete a node from an AVL tree, without locking the tree

Accepts:

	treeP		Address of the tree definition structure.
	keyP		Address of the key for the node.  Interpretation
			 of the key is left to the key-compare routine
			 specified in the tree header.

Returns :

 	<value>		0 or 1 if deleted OK
			-1 if not found

Notes :

	Does not lock the tree; useful when the caller wants to
	  maintain the lock, e.g. around multiple tree operations.

*/

int
avldelete_nolock ARGLIST ( ( treeP, keyP ) )
   NFARG( AVLTREE	*treeP )	/* Addr of tree block */
    FARG( void		*keyP )		/* Addr of key */
{
	int		r;

    if ( treeP == NULL )
	r = -1;

    else if ( ( r = delete( treeP, &treeP->t_rootP, keyP ) ) >= 0 )
	--treeP->t_nodeC;		/* Adjust the node count */

    return( r );			/* done */
}
/*

*//* avldelete_nounlock( treeP, keyP )

	Delete a node from an AVL tree, not releasing the tree's lock.

Accepts:

	treeP		Address of the tree definition structure.
	keyP		Address of the key for the node.  Interpretation
			 of the key is left to the key-compare routine
			 specified in the tree header.

Returns :

 	<value>		0 or 1 if deleted OK
			-1 if not found

Notes :

	A wrapper around avldelete_nolock(), locking the tree and
	  leaving it locked.

	If the tree's lock can not be aquired, no attempt is made to
	  delete the node, and -1 is returned.

*/

int
avldelete_nounlock ARGLIST ( ( treeP, keyP ) )
   NFARG( AVLTREE	*treeP )	/* Addr of tree block */
    FARG( void		*keyP )		/* Addr of key */
{
	int		r;

    if ( treeP == NULL )
	r = -1;

    else if ( mml_mutex_lock( &treeP->t_lock ) != 0 ) {
	warning( "avldelete_nounlock: can't get lock" );
	r = -1;
    }
    else {
	r = avldelete_nolock( treeP, keyP );
    }

    return( r );			/* done */
}
/*

*//* avldeltree( treeP )

	Delete all nodes from a tree

Accepts :

	treeP		Address of the tree header block

Returns :

	<nothing>

Notes :

	The header is not touched.  The tree is still available for
	  use after this deletion. 

*/

void
avldeltree ARGLIST( ( treeP ) )
    FARG( AVLTREE	*treeP )	/* Addr of tree header */
{
    if ( treeP == NULL )
	return;

    if ( mml_mutex_lock( &treeP->t_lock ) != 0 )
	warning( "avldeltree: can't get lock" );

    /* Call local routine to recursively delete nodes */
    deltree( treeP, treeP->t_rootP );

    /* Put header in correct state */
    treeP->t_rootP = NULL;
    treeP->t_nodeC = 0;

    if ( mml_mutex_unlock( &treeP->t_lock ) != 0 )
	warning( "avldeltree: can't release lock" );

    /* done */
}
/*

*//* avlfind( treeP, keyP )

	Lookup a value in an avl tree, with locking during the lookup.

Accepts :

	treeP		Address of the AVLTREE structure describing the tree.
	keyP		Address of a key for the node.  Interpretation of
			  the key is left to the "compare key" routine.

Returns :

	<value>		The address of the node if it is found,
			NULL if it is not.


Notes :

	Essentially a locking wrapper around avlfind_nolock()

	If the lock can not be acquired, NULL is returned.

*/

AVLNODE *
avlfind ARGLIST( ( treeP, keyP ) )
   NFARG( AVLTREE	*treeP )	/* Address of the tree struct */
    FARG( void		*keyP )		/* Address of the key info */

{
	AVLNODE		*nodeP;		/* The found node */

    if ( treeP == NULL )
	nodeP = NULL;

    else if ( mml_mutex_lock( &treeP->t_lock ) != 0 ) {
	warning( "avlfind: can't get lock" );
	nodeP = NULL;
    }
    else {
	nodeP = avlfind_nolock( treeP, keyP );

	if ( mml_mutex_unlock( &treeP->t_lock ) != 0 )
	    warning( "avlfind: can't release lock" );
    }

    return( nodeP );
}
/*

*//* avlfind_nolock( treeP, keyP )

	Lookup a value in an avl tree, without locking.

Accepts :

	treeP		Address of the AVLTREE structure describing the tree.
	keyP		Address of a key for the node.  Interpretation of
			  the key is left to the "compare key" routine.

Returns :

	<value>		The address of the node if it is found,
			NULL if it is not.

Notes :

	Doesn't lock the tree: handy when the caller wants to maintain
	  the lock around multiple tree manipulations.

*/

AVLNODE *
avlfind_nolock ARGLIST( ( treeP, keyP ) )
   NFARG( AVLTREE	*treeP )	/* Address of the tree struct */
    FARG( void		*keyP )		/* Address of the key info */

{
	AVLNODE		*nodeP;		/* To follow the tree with */

    if ( treeP == NULL )
	return ( NULL );

    /* Traverse the tree until the node is found or the tree runs out. */
    for( nodeP = treeP->t_rootP; nodeP != NULL; ) {
	switch( (*treeP->t_cmpkey)( keyP, nodeP ) ) {
	    case 0:			/* Node found! */
		return( nodeP );	/* Go ahead and return it */
		
	    case -1:			/* Take left branch */
		nodeP = nodeP->n_leftP;
		break;

	    case 1:			/* Take right branch */
	        nodeP = nodeP->n_rightP;
		break;

	    default:
		fprintf( stderr, "avlfind: cmpkey returned invalid result.\n" );
		break;
	}
    }

    /* Didn't find the node in the tree */
    return( NULL );
}
/*

*//* avlfind_nounlock( treeP, keyP )

	Lookup a value in an avl tree, acquiring and keeping the tree lock

Accepts :

	treeP		Address of the AVLTREE structure describing the tree.
	keyP		Address of a key for the node.  Interpretation of
			  the key is left to the "compare key" routine.

Returns :

	<value>		The address of the node if it is found,
			NULL if it is not.


Notes :

	Essentially a wrapper around avlfind_nolock(), locking the
	  tree and leaving it locked

	If the lock can not be acquired, NULL is returned.

*/

AVLNODE *
avlfind_nounlock ARGLIST( ( treeP, keyP ) )
   NFARG( AVLTREE	*treeP )	/* Address of the tree struct */
    FARG( void		*keyP )		/* Address of the key info */

{
	AVLNODE		*nodeP;		/* The found node */

    if ( treeP == NULL )
	nodeP = NULL;

    else if ( mml_mutex_lock( &treeP->t_lock ) != 0 ) {
	warning( "avlfind_nounlock: can't get lock" );
	nodeP = NULL;
    }
    else {
	nodeP = avlfind_nolock( treeP, keyP );
    }

    return( nodeP );
}
/*

*//* avlinsert( treeP, keyP, dataP )

	Insert a node in an avl tree, locking the tree during the insert.

Accepts :

	treeP		The address of the tree header structure.
	keyP		The address of the key for the node.
	dataP		The address of the data info for the tree.

Returns :

	<value>		-1 if failure of some kind
			 0 if successful insertion
			 1 if duplicate key

Notes:

	Essentially a locking wrapper around avlinsert_nolock();
	  see that for other notes.

	If the lock can not be acquired, -1 is returned and no attempt
	  is made to insert the node.

*/

int
avlinsert ARGLIST( ( treeP, keyP, dataP ) )
   NFARG( AVLTREE	*treeP )	/* Addr of the tree struct */
   NFARG( void		*keyP )		/* The key for insertion insert */
    FARG( void		*dataP )	/* The data for the insertion */
{
	int		r;

    if ( treeP == NULL )
	r = -1;

    else if ( mml_mutex_lock( &treeP->t_lock ) != 0 ) {
	warning( "avlinsert: can't get lock" );
	r = -1;
    }
    else {
	r = avlinsert_nolock( treeP, keyP, dataP );

	if ( mml_mutex_unlock( &treeP->t_lock ) != 0 )
	    warning( "avlinsert: can't release lock" );
    }

    return ( r );
}
/*

*//* avlinsert_nolock( treeP, keyP, dataP )

	Insert a node in an avl tree, without locking.

Accepts :

	treeP		The address of the tree header structure.
	keyP		The address of the key for the node.  Interpretation
			  of the key is by the compare and node-create
			  routines specified in the avl tree header.
	dataP		The address of the data info for the tree.  The
			 interpretation of this is left to the create-node
			 routine specified in the avl tree header.

Returns :

	<value>		-1 if failure of some kind
			 0 if successful insertion
			 1 if duplicate key

Notes:

	Does not lock the tree; useful when the caller wants to maintain
	  that locking.

	The tree header structure specifies a node construction routine
	that is responsible for allocating a node and putting the new
	key and data information into it.  It is called as follows:

	   nodeP = construct( treeP, keyP, dataP, enodeP )

	treeP, keyP, and dataP are as passed to this routine.  "enodeP"
	is NULL if a new node is required; otherwise it is the address
	of an already existing node that matches the specified key -
	in this case it is up to the constructor to decide whether to
	overwrite the existing node or to call it an error.  The routine
	is expected to return the address of the AVLNODE structure
	that is allocated (if enode==NULL ) or that exists, or to
	return NULL if the node is not made (or used).

*/

int
avlinsert_nolock ARGLIST( ( treeP, keyP, dataP ) )
   NFARG( AVLTREE	*treeP )	/* Addr of the tree struct */
   NFARG( void		*keyP )		/* The key for insertion insert */
    FARG( void		*dataP )	/* The data for the insertion */
{
	int		direction;	/* Direction we took from decision pt */
	int		result;
	AVLNODE		*nodeP;		/* Node that we're looking at */
	AVLNODE		*clearP;	/* For erasing tracks */
	AVLNODE		**nodePP;	/* Pointer to the next link */
	AVLNODE		**topPP;	/* Pointer to the top pointer */

    if ( treeP == NULL )
	return ( -1 );

    /* Traverse the tree to find an insertion point (or existing key).
       Along the way, we'll adjust the balance counts on the nodes as
       we pass by them.  And as we do this, we'll remember the potential
       tree rotation point (the lowest non-balanced treetop) as well as
       the direction we took from it (in case we have to fix it up when
       we discover a lower balance point). */

    nodePP = topPP = &(treeP->t_rootP);	/* Start at top of tree */
    direction = 0;			/* Haven't gone anywhwere yet */
    while( (nodeP = *nodePP) != NULL ) { /* Till we reach the end */

	/* See if we're at a potential balance point */
	if ( nodeP->n_balance != 0 ) {

	    /* New balance point.  Erase any trail we've made to here */
	    if ( direction != 0 )
		for( clearP = *topPP; clearP != nodeP;
				 direction = clearP->n_balance ) {
		    clearP->n_balance -= direction;
		    if ( direction < 0 )
			clearP = clearP->n_leftP;
		    else
			clearP = clearP->n_rightP;
		}
	     direction = 0;		/* So we make new balance point */
	     topPP = nodePP;		/* Remember new top */
	}

	/* Now follow the tree... */
	switch( (*treeP->t_cmpkey)( keyP, nodeP ) ) {
	    case 0:			/* Match */
		/* Here we have a duplicate node.  First erase the
		   trail that we left. */
		if ( direction != 0 )
		    for( clearP = *topPP; clearP != nodeP;
			    direction = clearP->n_balance ) {
			clearP->n_balance -= direction;
			if ( direction < 0 )
			    clearP = clearP->n_leftP;
			else
			    clearP = clearP->n_rightP;
		    }

		/* Give the node to the node constructor and
		   see what we get. */
		if ( (*treeP->t_mknode)( treeP, keyP, dataP, nodeP ) == NULL )
		    result = 1;		/* Duplicate key */
		else
		    result = 0;		/* Return success */
		return ( result );

	    case -1:			/* Go left */
		nodePP = &(nodeP->n_leftP);
		--nodeP->n_balance;
		if ( direction == 0 )	/* Remember balance point branch? */
		    direction = -1;
		break;

	    case 1:			/* Go right */
		nodePP = &(nodeP->n_rightP);
		++nodeP->n_balance;
		if ( direction == 0 )
		    direction = 1;
		break;

	    default:
		fprintf( stderr, "avlinsert: invalid return from cmpkey.\n" );
		break;
	}
    }

    /* Here we've gotten to the bottom, so make a new node */
    nodeP = (*treeP->t_mknode)( treeP, keyP, dataP, (AVLNODE *)NULL );
    if ( nodeP != NULL ) {		/* Successful node creation? */
	nodeP->n_balance = 0;		/* Fill in the nitty gritty */
	nodeP->n_leftP = nodeP->n_rightP = NULL;
	*nodePP = nodeP;		/* Link it in */
	balance( topPP );		/* May need reshaping now */
	++treeP->t_nodeC;		/* Update node count */
	result = 0;			/* Return success */
    }
    else {
	/* Error making node.  Erase our trail */
	if ( direction != 0 ) {
	    for( clearP = *topPP; clearP != NULL; ) {
		clearP->n_balance -= direction;
		if ( direction < 0 )
		    clearP = clearP->n_leftP;
		else
		    clearP = clearP->n_rightP;
		if ( clearP != NULL )
		    direction = clearP->n_balance;
	    }
	}
	result = -1;
    }

    return ( result );
}
/*

*//* avlinsert_nounlock( treeP, keyP, dataP )

	Insert a node in an avl tree, gaining and keeping the tree's lock.

Accepts :

	treeP		The address of the tree header structure.
	keyP		The address of the key for the node.
	dataP		The address of the data info for the tree.

Returns :

	<value>		-1 if failure of some kind
			 0 if successful insertion
			 1 if duplicate key

Notes:

	Essentially a wrapper around avlinsert_nolock(), locking the
	  tree and leaving it locked.

	If the lock can not be acquired, -1 is returned and no attempt
	  is made to insert the node.

*/

int
avlinsert_nounlock ARGLIST( ( treeP, keyP, dataP ) )
   NFARG( AVLTREE	*treeP )	/* Addr of the tree struct */
   NFARG( void		*keyP )		/* The key for insertion insert */
    FARG( void		*dataP )	/* The data for the insertion */
{
	int		r;

    if ( treeP == NULL )
	r = -1;

    else if ( mml_mutex_lock( &treeP->t_lock ) != 0 ) {
	warning( "avlinsert: can't get lock" );
	r = -1;
    }
    else {
	r = avlinsert_nolock( treeP, keyP, dataP );
    }

    return ( r );
}
/*

*//* avlnew( treeP, cmpkey, mknode, rmnode )

	Make a header for a new AVL tree.

Accepts :

	treeP		Address of the header structure, or NULL to
			  allocate one via malloc().
	cmpkey		Routine to compare keys
	mknode		Routine to make a node
	rmnode		Routine to destroy a node

Returns :

	<value>		The address of the header, or NULL if failure.

Notes :

	This routine is for convenience.  (aren't they all?)
*/

AVLTREE *
avlnew ARGLIST( ( treeP, cmpkey, mknode, rmnode ) )
   NFARG( AVLTREE	*treeP )	/* Addr of tree header block */
   NFARG( int		(*cmpkey)() )	/* Routine to compare keys */
   NFARG( AVLNODE	*(*mknode)() )	/* Routine to make a node */
    FARG( int		(*rmnode)() )	/* Routine to destroy a node */
{
    /* Allocate memory for header if needed */
    if ( treeP == NULL )
	if ( ( treeP = (AVLTREE *)emalloc( "avlnew", "new tree",
					  sizeof( AVLTREE) ) ) == NULL )
	    return( NULL );

    /* Fill in the header fields */
    treeP->t_rootP = NULL;
    treeP->t_nodeC = 0;
    treeP->t_infoP = NULL;
    treeP->t_walkP = NULL;
    if ( mml_mutex_init( &treeP->t_lock ) != 0 )
	warning( "avlnew: can't init lock" );
    treeP->t_cmpkey = cmpkey;
    treeP->t_mknode = mknode;
    treeP->t_rmnode = rmnode;

    return( treeP );
}
/*

*//* avlwalk( treeP, rtc )

	Walk a tree, calling a routine at each node, while locking the
	  tree for the duration of the walk.

Accepts :

	treeP		Address of the tree header structure
	rtc		Address of the routine to call (see Notes)

Returns :

	<nothing>

Notes :

	A locking wrapper around avlwalk_nolock; see that function's
	  intro for other detail and notes.

	If the lock can not be acquired, no attempt is made to walk
	  the tree.

*/

void
avlwalk ARGLIST( ( treeP, rtc ) )
   NFARG( AVLTREE	*treeP )	/* Ptr to tree header */
    FARG( int		(*rtc)() )	/* Routine to call per node */
{
    if ( treeP == NULL )
	return;

    if ( mml_mutex_lock( &treeP->t_lock ) != 0 )
	warning( "avlwalk: can't get lock" );
    else {
	/* Hand off to the non-locking walk function */
	avlwalk_nolock( treeP, rtc );

	if ( mml_mutex_unlock( &treeP->t_lock ) != 0 )
	    warning( "avlwalk: can't release lock" );
    }

    /* done */
}
/*

*//* avlwalk_nolock( treeP, rtc )

	Walk a tree, calling a routine at each node, without locking.

Accepts :

	treeP		Address of the tree header structure
	rtc		Address of the routine to call (see Notes)

Returns :

	<nothing>

Notes :

	rtc conforms to:

		int rtc( treeP, nodeP, depth )

		depth is the depth of the node; 0 == top, 1 is next, etc.

		rtc returns 0 to continue processing the tree,
		   non-zero to stop.

		rtc MUST NOT alter the tree.

	Caller is responsible for acquiring/maintaining any tree lock.

	There is a t_walkP element that the walk-caller can store
	something into for context during the walk.

*/

void
avlwalk_nolock ARGLIST( ( treeP, rtc ) )
   NFARG( AVLTREE	*treeP )	/* Ptr to tree header */
    FARG( int		(*rtc)() )	/* Routine to call per node */
{
    if ( treeP == NULL )
	return;

    /* Call local routine to recursively walk the tree */
    walk( treeP, treeP->t_rootP, rtc, 0 );

    /* End the walk with a NULL node, regardless of termination condition */
    (*rtc)( treeP, (AVLNODE *)NULL, 0 );

    /* done */
}
/*

*//* avlwalk_nounlock( treeP, rtc )

	Walk a tree, calling a routine at each node,
	  locking the tree and leaving it locked.

Accepts :

	treeP		Address of the tree header structure
	rtc		Address of the routine to call (see Notes)

Returns :

	<nothing>

Notes :

	A wrapper around avlwalk_nolock(), locking the tree and leaving
	  it locked.  See avlwalk_nolock() intro for other detail and notes.

	If the lock can not be acquired, no attempt is made to walk
	  the tree.

*/

void
avlwalk_nounlock ARGLIST( ( treeP, rtc ) )
   NFARG( AVLTREE	*treeP )	/* Ptr to tree header */
    FARG( int		(*rtc)() )	/* Routine to call per node */
{
    if ( treeP == NULL )
	return;

    if ( mml_mutex_lock( &treeP->t_lock ) != 0 )
	warning( "avlwalk: can't get lock" );
    else {
	/* Hand off to the non-locking walk function */
	avlwalk_nolock( treeP, rtc );
    }

    /* done */
}


/***********************************************************************
 *                                                                     *
 *                     Internal support routines                       *
 *                                                                     *
 ***********************************************************************/


/*

*//* balance( branchPP )

	Local routine to balance a branch

Accepts :

	branchPP	Addr of the variable pointing to the top node

Returns :

	<value>		0 if branch has stayed the same height;
			1 if branch shrunk by one.


Notes :

	This routine accepts a branch in conditions left by the
internal routines only.  No other cases are dealt with.

*/

static int
balance ARGLIST( ( branchPP ) )
    FARG( AVLNODE	**branchPP )	/* Ptr to top node */
{
	int		shrunk;		/* Whether we shrunk */
	AVLNODE		*nodeP;		/* Current top node */
	AVLNODE		*leftP;		/* Left child */
	AVLNODE		*rightP;	/* Right child */
	AVLNODE		*migP;		/* A ndoe that migrates */

    /* Pick up relevant information */
    nodeP = *branchPP;
    leftP = nodeP->n_leftP;
    rightP = nodeP->n_rightP;
    shrunk = 0;				/* Assume tree doesn't shrink */

    /* Process according to out-of-balance amount, if any */
    switch( nodeP->n_balance ) {
	case -2:			/* Too heavy on left */
	    if ( leftP->n_balance <= 0 ) {

		/* Single rotation */
		*branchPP = leftP;
		nodeP->n_leftP = leftP->n_rightP;
		leftP->n_rightP = nodeP;
		++leftP->n_balance;
		nodeP->n_balance = -(leftP->n_balance);
		if ( leftP->n_balance == 0 )
		    shrunk = 1;
	    }
	    else {			/* Migration of inner node to top */
		migP = leftP->n_rightP;
		leftP->n_rightP = migP->n_leftP;
		nodeP->n_leftP = migP->n_rightP;
		migP->n_leftP = leftP;
		migP->n_rightP = nodeP;
		*branchPP = migP;
		if ( migP->n_balance < 0 ) {
		    leftP->n_balance = 0;
		    nodeP->n_balance = 1;
		}
		else if ( migP->n_balance > 0 ) {
		    leftP->n_balance = -1;
		    nodeP->n_balance = 0;
		}
		else
		    leftP->n_balance = nodeP->n_balance = 0;
		migP->n_balance = 0;
		shrunk = 1;
	    }
	    break;
		    
	case  2:			/* Too heavy on right */
	    if ( rightP->n_balance >= 0 ) {

		/* Single rotation */
		*branchPP = rightP;
		nodeP->n_rightP = rightP->n_leftP;
		rightP->n_leftP = nodeP;
		--rightP->n_balance;
		nodeP->n_balance = -(rightP->n_balance);
		if ( rightP->n_balance == 0 )
		    shrunk = 1;
	    }
	    else {			/* Migration of inner node */
		migP = rightP->n_leftP;
		rightP->n_leftP = migP->n_rightP;
		nodeP->n_rightP = migP->n_leftP;
		migP->n_leftP = nodeP;
		migP->n_rightP = rightP;
		*branchPP = migP;
		if ( migP->n_balance < 0 ) {
		    nodeP->n_balance = 0;
		    rightP->n_balance = 1;
		}
		else if ( migP->n_balance > 0 ) {
		    nodeP->n_balance = -1;
		    rightP->n_balance = 0;
		}
		else
		    nodeP->n_balance = rightP->n_balance = 0;
		migP->n_balance = 0;
		shrunk = 1;
	    }
    }
    return( shrunk );
}
/*

*//*

*//* delete( treeP, topPP, keyP )

	Local routine to delete from a subtree

Accepts:

	treeP		Address of the tree definition structure
	topPP		Address of the pointer to this branch
	keyP		Address of the key for the node.  Interpretation
			 of the key is left to the key-compare routine
			 specified in the tree header.

Returns :

 	<value>		-1 if not found;
			 0 if deleted and tree did not shrink;
			 1 if deleted and tree shrunk by 1.

*/

static int
delete ARGLIST( ( treeP, topPP, keyP ) )
   NFARG( AVLTREE	*treeP )	/* Addr of tree block */
   NFARG( AVLNODE	**topPP )	/* Addr of ptr to branch */
    FARG( void		*keyP )		/* Addr of key */
{
	int		i;		/* Scratch */
	int		sts;
	int		cmp;		/* Comparison result */
	AVLNODE		*nodeP;		/* Node pointer */
	AVLNODE		*node1P;	/* Another one */
	AVLNODE		*tempP;		/* For exchanging */
	AVLNODE		**linkPP;	/* Addr of a node pointer */

    nodeP = *topPP;			/* Get addr of node */
    if ( nodeP == NULL )		/* If we hit bottom */
	return( -1 );			/*  then we didn't find it */

    cmp = (*treeP->t_cmpkey)( keyP, nodeP );
    if ( cmp == 0 ) {
	/* We've found the node to delete.  If it has no children we
	   can just get rid of it.  Otherwise we have to remove it
	   from the tree somehow.  If one or the other subtrees
 	   is empty, then it is easy. */

	if ( nodeP->n_leftP == NULL ) {
	    /* Just shorten the right branch (even if it doesn't exist) */
	    *topPP = nodeP->n_rightP;
	    (*treeP->t_rmnode)( treeP, nodeP );
	    return( 1 );		/* Branch shrunk */
	}

	if ( nodeP->n_rightP == NULL ) {
	    /* Shorten the left branch */
	    *topPP = nodeP->n_leftP;
	    (*treeP->t_rmnode)( treeP, nodeP );
	    return( 1 );
	}

	/* Both subtrees exist.  Exchange this node with the one
	   logically adjacent (in value) to it.  Then this node
	   will be at the end of a branch and we can recurse to
	   delete it. */

	if( nodeP->n_balance < 0 ) {
	    node1P = nodeP->n_leftP;
	    linkPP = &(node1P->n_leftP);
	    for( ; node1P->n_rightP != NULL; node1P = node1P->n_rightP )
		linkPP = &(node1P->n_rightP);
	    cmp = -1;			/* We went left */
	} else {
	    node1P = nodeP->n_rightP;
	    linkPP = &(node1P->n_rightP);
	    for( ; node1P->n_leftP != NULL; node1P = node1P->n_leftP )
		linkPP = &(node1P->n_leftP);
	    cmp = 1;			/* We're going right */
	}

	/* Exchange the two nodes.  We have to actually exchange by
	   relinking rather than copying since the node may imply
	   other stuff that we don't know about here. */

	tempP = node1P->n_rightP;	/* Exchange right pointers */
	node1P->n_rightP = nodeP->n_rightP;
	nodeP->n_rightP = tempP;

	tempP = node1P->n_leftP;	/* Exchange left pointers */
	node1P->n_leftP = nodeP->n_leftP;
	nodeP->n_leftP = tempP;

	i = node1P->n_balance;		/* Exchange balance values */
	node1P->n_balance = nodeP->n_balance;
	nodeP->n_balance = i;

	*topPP = node1P;		/* Exchange the nodes */
	*linkPP = nodeP;
	nodeP = node1P;			/* Pretend we're here */
    }

    /* Remove the node from left or right subtree depending on "cmp" */
    switch ( cmp ) {
	case -1:
	    sts = delete( treeP, &nodeP->n_leftP, keyP );
	    if ( sts == 1 )		/* If it shrunk */
		++nodeP->n_balance;	/*  reflect that in the balance */
	    break;

	case 1:
	    sts = delete( treeP, &nodeP->n_rightP, keyP );
	    if ( sts == 1 )		/* Right branch shrunk? */
		--nodeP->n_balance;	/*  adjust the balance */
	    break;

	default:
	    fprintf( stderr, "avl delete: invalid return from cmpkey.\n" );
	    sts = 0;
	    break;
    }

    if ( sts == 1 )			/* If we changed balance */
	if ( nodeP->n_balance != 0 )	/*  if it's 0 it shrunk but is ok */
	    sts = balance( topPP );	/*    otherwise fix it up */
    return( sts );
}
/*

*//* deltree( treeP, nodeP )

	Delete a subtree on behalf of avldeltree

Accepts :

	treeP		Ptr to the tree being deleted
	nodeP		Ptr to the next node being deleted

Returns :

	<nothing>

Notes :

	Recursively deletes any subtrees attached to this node

*/

static void
deltree ARGLIST( ( treeP, nodeP ) )
   NFARG( AVLTREE	*treeP )	/* Tree header block */
    FARG( AVLNODE	*nodeP )	/* This node.. */
{
    if ( nodeP != NULL ) {		/* Handle NULL branches */
	/* Delete the subtrees */
	deltree( treeP, nodeP->n_leftP );
	deltree( treeP, nodeP->n_rightP );

	/* Call node deletion routine to dispose of the node */
	(*treeP->t_rmnode)( treeP, nodeP );
    }
}
/*

*//* walk( treeP, nodeP, rtc, depth )

	Local routine to do tree walking on behalf of avlwalk

Accepts :

	treeP		Address of the tree header structure
	nodeP		next node being walked
	rtc		Address of the routine to call (see Notes)
	depth		Depth of tree at this node

Returns :

	<value>		0 to keep going
			non-zero to stop

Notes :

	tree-walking needs a recursive function, this is it.

*/

static int
walk ARGLIST( ( treeP, nodeP, rtc, depth ) )
   NFARG( AVLTREE	*treeP )
   NFARG( AVLNODE	*nodeP )
   NFARG( int		(*rtc)() )
    FARG( int		depth )
{
	int		status;

    if ( nodeP != NULL ) {		/* Filter out null branches */
	/* Descend the left branch */
	if ( ( status = walk( treeP, nodeP->n_leftP, rtc, depth+1 ) ) == 0 )
	    /* Process current node */
	    if ( ( status = (*rtc)( treeP, nodeP, depth ) ) == 0 )
		/* Descend the right branch */
		status = walk( treeP, nodeP->n_rightP, rtc, depth+1 );
    }
    else
	status = 0;			/* Empty branch */

    return( status );
}


/***********************************************************************
 *                                                                     *
 *                   Some general handler functions                    *
 *                                                                     *
 *  Some generic handler functions that can be used for simple AVL     *
 *  trees.                                                             *
 *                                                                     *
 ***********************************************************************/


/*****  avl support functions for string key-only trees.  ******/

/*

*//* mml_avl_str_cmpkey( keyP, nodeP )

	Compare string keys for avl tree routines

Accepts :

	keyP		Ptr to the subject key (a character string)
	nodeP		Ptr to existing node to test key against

Returns :

	< value >	-1, 0, or 1, as dictated by the rules of
			interface between the avl tree routines and
			the key comparison routine.

Notes:


*/


int
mml_avl_str_cmpkey( keyP, nodeP )
	char		*keyP;		/* Key to look for */
	STRNODE		*nodeP;		/* Node to compare against */
{
	int		cmpval;

    /* Compare the strings name */
    cmpval = stricmp( keyP, nodeP->sn_strP );

    return ( cmpval );
}
/*

*//* mml_avl_str_mknode( treeP, keyP, dataP, enodeP )

	Make a string avl node on behalf of AVL tree routines

Accepts :

	treeP		Address of tree header block
	keyP		Address of the key block (a string)
	dataP		Address of data (not used)
	enodeP		Address of any existing node

Returns :

	<value>		Address of the new node

Notes :


*/

AVLNODE *
mml_avl_str_mknode( treeP, keyP, dataP, enodeP )
	AVLTREE		*treeP;		/* Addr of tree header block */
	char		*keyP;		/* The key, a string */
	void		*dataP;		/* Addr of new data (n/a) */
	STRNODE		*enodeP;	/* Existing node... */
{
	int		nodesize;	/* Size of node required */
	STRNODE		*nodeP;		/* New node that we make */

    /* If enodeP is non-NULL, this is a call to overwrite the data.
       That should never happen, so return NULL in that case. */
    if ( enodeP != NULL )
	return( (AVLNODE *)NULL );

    /* Calculate new node size (we co-allocate space for the string) */
    nodesize = sizeof( STRNODE ) + strlen( keyP ) + 1;
    nodeP = (STRNODE *)emalloc( "mml_avl_str_mknode", "avl node",
					 nodesize );

    /* Setup the node struct */
    nodeP->sn_strP = (char *)&nodeP[1];
    strcpy( nodeP->sn_strP, keyP );

    return( (AVLNODE *)nodeP );
}
/*

*//* mml_avl_str_rmnode( treeP, nodeP )

	Delete a node on behalf of AVL tree routines.

Accepts :

	treeP		Address of tree header block
	nodeP		Address of the node being removed

Returns :

	<value>			0 for success, -1 for failure

Notes :

	Just a dumb deletion.

*/

int
mml_avl_str_rmnode( treeP, nodeP )
	AVLTREE		*treeP;
	STRNODE		*nodeP;
{
    /* Deallocate the node */
    dealloc( nodeP );

    /* Return OK */
    return( 0 );
}


/*****  end of avl support functions for string key-only trees.  ******/





/*****  avl support functions for string pointer key-only trees.  ******/


/*

*//* mml_avl_strp_cmpkey( keyP, nodeP )

	Compare string keys for avl tree routines

Accepts :

	keyP		Ptr to the subject key (a character string)
	nodeP		Ptr to existing node to test key against

Returns :

	< value >	-1, 0, or 1, as dictated by the rules of
			interface between the avl tree routines and
			the key comparison routine.

Notes:


*/


int
mml_avl_strp_cmpkey( keyP, nodeP )
	char		*keyP;		/* Key to look for */
	STRNODE		*nodeP;		/* Node to compare against */
{
	int		cmpval;

    /* Compare the strings */
    cmpval = stricmp( keyP, nodeP->sn_strP );

    return ( cmpval );
}
/*

*//* mml_avl_strp_mknode( treeP, keyP, dataP, enodeP )

	Make a string avl node on behalf of AVL tree routines

Accepts :

	treeP		Address of tree header block
	keyP		Address of the key block (a string)
	dataP		Address of data (not used)
	enodeP		Address of any existing node

Returns :

	<value>		Address of the new node

Notes :

	keyP must be a dynamically allocated string that we can point
	to and deallocate when we are done with it.  Once given to
	the avl functions, caller should forget it..

*/

AVLNODE *
mml_avl_strp_mknode( treeP, keyP, dataP, enodeP )
	AVLTREE		*treeP;		/* Addr of tree header block */
	char		*keyP;		/* Key, allocated string */
	void		*dataP;		/* Addr of new data (n/a) */
	STRNODE		*enodeP;	/* Existing node... */
{
	STRNODE		*nodeP;		/* New node that we make */

    /* If enodeP is non-NULL, this is a call to overwrite the data.
       That should never happen, so return NULL in that case. */
    if ( enodeP != NULL )
	return( (AVLNODE *)NULL );

    /* Allocate new node */
    nodeP = (STRNODE *)emalloc( "mml_avl_strp_mknode", "avl node",
					 sizeof(STRNODE) );

    /* Setup the node struct */
    nodeP->sn_strP = keyP;

    return( (AVLNODE *)nodeP );
}
/*

*//* mml_avl_strp_rmnode( treeP, nodeP )

	Delete a node on behalf of AVL tree routines.

Accepts :

	treeP		Address of tree header block
	nodeP		Address of the node being removed

Returns :

	<value>			0 for success, -1 for failure

Notes :

	Deallocates the string and node

*/

int
mml_avl_strp_rmnode( treeP, nodeP )
	AVLTREE		*treeP;
	STRNODE		*nodeP;
{
    /* Deallocate the string and node */
    dealloc( nodeP->sn_strP );
    dealloc( nodeP );

    /* Return OK */
    return( 0 );
}


/*****  end of avl support functions for string pointer key-only trees.  ******/
