/* mml-iopf.h -- definitions for the mml "iopf" module

	Copyright 2003,2004,2005,2006,2007
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Functions somewhat related to input, output, conversions, and parsing.

*/


#ifndef	H_MML_IOPF		/* For multiple inclusions */
#define	H_MML_IOPF

    /* Flags for tokenizing.  Note that many of these are expressed as
       "not"s since they are intended to override default behaviour.
    */
#define	TKF_NOSKIPLEAD	0x0001	/* Do not skip leading chars, depends on context */
#define	TKF_NOQUOTSTR	0x0002	/* Quoted strings are not OK */
#define	TKF_NOQUOTCHAR	0x0004	/* Quoted chars (via backslash) are not OK */

    /* Flags controlling interpretation of text records */
#define	TXR_END_CRLF	0x0001	/* CRLF will end a record */
#define	TXR_END_CR	0x0002	/* Naked CR will end a record */
#define	TXR_END_LF	0x0004	/* Naked LF will end a record */
#define	TXR_END_FULL	0x0008	/* Full buffer will end a record */

    /* This is the default control used by "getbline" for inputting
       a line of text.  With this default we accept lines of text
       terminated by LF or CRLF, but naked CRs are left alone.
    */
#define	TXR_LINE	((TXR_END_CRLF)|(TXR_END_LF))



#ifdef	MML_OLD_IOPFNAMES
    /* Macro equivalents for the old getxxx() functions that were
       renamed to fgetxxxx() in Feb 2005.  Defined (and further
       explained) in mml.h.
    */

# define getbline	fgetbline
# define getblinea	fgetblinea
# define getblinec	fgetblinec
# define getbtrec	fgetbtrec
# define getbtreca	fgetbtreca
# define getbtrecc	fgetbtrecc
# define gettkbline	fgettkbline

#endif	/* MML_OLD_IOPFNAMES */


    /* Macros to access token pool components.  Since some of these
     *  do one or more heavy dereferences, if many access are made to
     *  a token pool it may be faster to do some of these things by hand.
     *
     * TKN_SX( tknsP, tN ) -- the start offset of token number tN
     * TKN_EX( tknsP, tN ) -- the end offset (nul byte) of token number tN
     * TKN_P( tknsP, tN ) -- ptr to the start of token number tN
     */

#define TKN_SX( tknsP, tN ) \
    ((tknsP)->tp_tbP[tN].tb_startX)

#define TKN_EX( tknsP, tN ) \
    ((tknsP)->tp_tbP[tN].tb_endX)

#define TKN_P( tknsP, tN ) \
    (&(tknsP)->tp_bsP->bs_bP[ TKN_SX( tknsP, tN ) ])


    /* An object that's also used to control the interpretation of
       text records.

       Note: This structure isn't used, but we define it so we
       can pass extended information to text record routine(s)
       at some point in the future.  It just holds a place now
       in the function prototypes for any future need.

    */
typedef struct txrinfo {
    BOOL	ti_something;	/* Nothing really defined. */
}  TXRINFO;


    /* Keeps track of token boundaries within a token buffer */
typedef struct tknb {
    int		tb_startX;	/* Index to start of token */
    int		tb_endX;	/* Index to NUL byte after token */
}  TKNB;


    /* A collection of tokens, their buffer, and their needs and wants */
typedef struct tkns {
    int		tp_tC;		/* Current number of tokens */
    int		tp_tL;		/* Length of the token bounds array */
    int		tp_max;		/* Maximum #tokens accepted in this pool */
    UBYTE	*tp_vsepP;	/* "Visible" separators per gettoken() */
    UBYTE	*tp_isepP;	/* "Invisible" separators per gettoken() */
    BOOL	tp_sisepF;	/* Single invisible separator flag
				   per gettoken() */
    TKNB	*tp_tbP;	/* Ptr to token bounds array */
    BSTR	*tp_bsP;	/* Byte string for the tokens */
}  TKNS;


/* Declarations for functions contained in this module: */

/* Note: in general I like to arrange these alphabetically.  In this
   case there are functions with an initial letter indicating the
   kind of input: 'i' for MML_IN streams, 'f' for FILE streams.
   The major alphabetical ordering ignores the initial letters.
*/

    /* Input a line, storing into a bstr buffer */
int	fgetbline PROTO( ( FILE *fP, BSTR *bsP ) );
int	igetbline PROTO( ( MML_IN *inP, BSTR *bsP ) );

    /* Input a line, appending into a bstr buffer */
int	fgetblinea PROTO( ( FILE *fP, BSTR *bsP ) );
int	igetblinea PROTO( ( MML_IN *inP, BSTR *bsP ) );

    /* Input a continued line plus any continuation lines */
int	fgetblinec PROTO( ( FILE *fP, BSTR *bsP ) );
int	igetblinec PROTO( ( MML_IN *inP, BSTR *bsP ) );

    /* Input a text "record", storing into a bstr buffer */
int	fgetbtrec PROTO( ( FILE *fP, BSTR *bsP,
			   UWORD rctl, TXRINFO *tiP ) );
int	igetbtrec PROTO( ( MML_IN *inP, BSTR *bsP,
			   UWORD rctl, TXRINFO *tiP ) );

    /* Input a text record, appending into a bstr buffer */
int	fgetbtreca PROTO( ( FILE *fP, BSTR *bsP,
			    UWORD rctl, TXRINFO *tiP ) );
int	igetbtreca PROTO( ( MML_IN *inP, BSTR *bsP,
			    UWORD rctl, TXRINFO *tiP ) );

    /* Input a text record plus any continuation records */
int	fgetbtrecc PROTO( ( FILE *fP, BSTR *bsP,
			    UWORD rctl, TXRINFO *tiP ) );
int	igetbtrecc PROTO( ( MML_IN *inP, BSTR *bsP,
			    UWORD rctl, TXRINFO *tiP ) );

    /* Input and tokenize a line */
int	fgettkbline PROTO( ( FILE *fP, BSTR *bsP, TKNS *tknsP ) );
int	igettkbline PROTO( ( MML_IN *inP, BSTR *bsP, TKNS *tknsP ) );


    /* Find a token in a string via delimiters */
int	find_token_delimited PROTO( (UBYTE *strP, int strL, TKNB *tknbP,
			   UBYTE *vsepP, UBYTE *isepP, UWORD tflags) );

    /* Extract a token from a string */
BOOL	gettoken PROTO( ( BSTR *bsP, BSTR *tkbsP,
			   UBYTE *vsepP, UBYTE *isepP, BOOL sisepF ) );

    /* Tokenize a line */
BOOL	tkline PROTO( ( BSTR *bsP, TKNS *tknsP ) );

    /* Tokenize more of a line */
BOOL	tkline_more PROTO( ( BSTR *bsP, TKNS *tknsP ) );

    /* Make a new token pool */
TKNS	*tkns_new PROTO( (BSTR *bsP, int max,
			   UBYTE *vsepP, UBYTE *isepP, BOOL sisepF ) );

    /* Register a token into a token pool */
BOOL	tkns_reg PROTO( (TKNS *tknsP, int startX, int endX) );

#endif	/* H_MML_IOPF */
