/* mml-ll.c -- mm library (mml) linked list routines

	Copyright 2003,2004,2005,2006,2007
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Contains general routines to manipulate linked lists.

*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#include <mml/mml.h>

#include <mml/mml-ll.h>
#include <mml/mml-alloc.h>

/* Local definitions */



/* External data referenced */



/* External routines used */



/* Local data publicly available */



/* Local routines and forward references */



/* Private data */


/*

*//* ll_init( hdrP )

	Initialize a linked list header


Accepts :

	hdrP		Ptr to list head structure

Returns :

	<no value>

	header structure initialized.

Notes :

	Application may call ll_finit on the list when done with it, and
	  SHOULD do so when linked list creation/deletion is frequent or
	  when a list is being retired and the application is not exiting.

*/

void
ll_init ARGLIST( (hdrP) )
    FARG( LL_HEAD	*hdrP )		/* Ptr to list head */
{
    hdrP->llh_firstP = (LL_NODE *)NULL;
    hdrP->llh_lastP = (LL_NODE *)NULL;
    hdrP->llh_nodeC = 0;
    if ( mml_mutex_init( &hdrP->llh_lock ) != 0 )
	warning( "ll_init: can't init lock" );
}
/*

*//* ll_finit( hdrP )

	Finalize (destroy) a linked list header


Accepts :

	hdrP		Ptr to list head structure

Returns :

	<no value>

	header structure finalized

Notes :

	If any nodes are on the list, removes them (after giving a
	  warning) without regard to their contents.

	Destroys any mutex associated with the list.

*/

void
ll_finit ARGLIST( (hdrP) )
    FARG( LL_HEAD	*hdrP )		/* Ptr to list head */
{
	LL_NODE		*lnodeP;

    /* Lock while we are futzing with it */
    if ( mml_mutex_lock( &hdrP->llh_lock ) != 0 )
	warning( "ll_finit: can't get lock" );

    /* If there are any nodes, give a warning and remove them */
    if ( hdrP->llh_firstP != NULL ) {
	warning( "ll_finit: list not empty.  Removing nodes." );
	while ( ( lnodeP = hdrP->llh_firstP ) != NULL )
	    ll_rem_nolock( lnodeP );
    }

    /* This should be superfluous and unnecessary.  Both. */
    hdrP->llh_firstP = (LL_NODE *)NULL;
    hdrP->llh_lastP = (LL_NODE *)NULL;

    /* Destroy the mutex (stupid pthreads makes you unlock it first,
       opening up a nice window, this should be atomic).
    */
    if ( mml_mutex_unlock( &hdrP->llh_lock ) != 0 )
	warning( "ll_finit: can't release lock" );
    if ( mml_mutex_destroy( &hdrP->llh_lock ) != 0 )
	warning( "ll_finit: can't destroy lock" );
}
/*

*//* ll_node( noteP )

	Allocate a new linked list node


Accepts :

	noteP		Descriptive note (passed to memory allocator)

Returns :

	<value>		Ptr to list node structure allocated
or
	NULL		if there was an error making a node

Notes :

	Uses erualloc() to keep free nodes around.

*/

LL_NODE *
ll_node ARGLIST( (noteP) )
    FARG( char		*noteP )	/* Descriptive note */
{
	LL_NODE		*nodeP;		/* Ptr to node */

    nodeP = erualloc( "ll_node", noteP, sizeof( LL_NODE ) );

    if ( nodeP != NULL ) {
	nodeP->lln_nextP = NULL;
	nodeP->lln_prevP = NULL;
	nodeP->lln_hdrP = NULL;
	nodeP->lln_valP = (void *)NULL;
    }

    return ( nodeP );
}
/*

*//* ll_node_free( nodeP )

	Release a node

Accepts :

	nodeP		Ptr to node to free

Returns :

	<nothing>

Notes :

	Uses derualloc() in order to keep free nodes binned for
	  quick access another time.

*/

void
ll_node_free ARGLIST( (nodeP) )
    FARG( LL_NODE	*nodeP )	/* Node to remove */
{
    derualloc( nodeP, sizeof( LL_NODE ) );
}
/*

*//* ll_addvb( hdrP, nnodeP, valP )

	Add a value before an existing list node

Accepts :

	hdrP		Ptr to the list header
	nnodeP		Ptr to node to link new node prior to.
	valP		Ptr to value to be added to list

Returns :

	<value>		Ptr to list node structure where value is added,
			 or NULL if there was an error making a node

Notes :

	valP will be linked into the node directly.

	This function does NOT lock the linked list; it's assumed that
	if the caller is walking the list, caller already has access to
	it.

	If nnodeP is passed as NULL, the new node will be added to
	  the end of the list.  In this case, hdrP is necessary to
	  locate the list header.  When nnodeP is non-NULL, the
	  passed hdrP is not used, but determined from the header
	  pointer in nnodeP.

*/

LL_NODE *
ll_addvb ARGLIST( (hdrP, nnodeP, valP) )
   NFARG( LL_HEAD	*hdrP )		/* Ptr to header */
   NFARG( LL_NODE	*nnodeP )	/* Node to link prior to */
    FARG( void		*valP )		/* Ptr to value to add */
{
	LL_NODE		*nodeP;		/* Ptr to new node */

    if ( nnodeP != NULL )
	hdrP = nnodeP->lln_hdrP;

    nodeP = ll_node( "from ll_addvb" );

    nodeP->lln_valP = valP;		/* Set the value */
    nodeP->lln_hdrP = hdrP;		/* Link back to header */

    /* Insert new node prior to next one */
    if ( nnodeP == NULL ) {
	/* This is the new tail; update links accordingly */
	nodeP->lln_prevP = hdrP->llh_lastP;
	hdrP->llh_lastP = nodeP;
    }
    else {
	/* Not new tail; link new node before existing one. */
	nodeP->lln_prevP = nnodeP->lln_prevP;
	nnodeP->lln_prevP = nodeP;
    }
    if ( nodeP->lln_prevP == NULL )
	hdrP->llh_firstP = nodeP;	/* New node is new head */
    else
	nodeP->lln_prevP->lln_nextP = nodeP;

    nodeP->lln_nextP = nnodeP;

    ++hdrP->llh_nodeC;

    return ( nodeP );
}
/*

*//* ll_addvh( hdrP, valP )

	Add a value to the head of a list


Accepts :

	hdrP		Ptr to list head structure
	valP		Ptr to value to be added to list

Returns :

	<value>		Ptr to list node structure where value is added,
or
	NULL		if there was an error making a node

Notes :

	It's assumed that "valP" is safe to point to forever, or at
	least for the life of the list.

*/

LL_NODE *
ll_addvh ARGLIST( (hdrP, valP) )
   NFARG( LL_HEAD	*hdrP )		/* Ptr to list head */
    FARG( void		*valP )		/* Ptr to value to add */
{
	LL_NODE		*nodeP;		/* Ptr to new node */

    nodeP = ll_node( "from ll_addvh" );

    if ( nodeP != NULL ) {
	if ( mml_mutex_lock( &hdrP->llh_lock ) != 0 )
	    warning( "ll_addvh: can't get lock" );

	nodeP->lln_valP = valP;		/* Set the value */

	/* New node's "next" pointer is current first node */
	if ( ( nodeP->lln_nextP = hdrP->llh_firstP ) != NULL )
	    /* Link old first node back to this one */
	    (nodeP->lln_nextP)->lln_prevP = nodeP;
	else
	    /* List was empty, so new node is also last node */
	    hdrP->llh_lastP = nodeP;

	hdrP->llh_firstP = nodeP;
	nodeP->lln_hdrP = hdrP;

	++hdrP->llh_nodeC;

	if ( mml_mutex_unlock( &hdrP->llh_lock ) != 0 )
	    warning( "ll_addvh: can't release lock" );
    }

    return ( nodeP );
}
/*

*//* ll_addvs( hdrP, valP, cmpP )

	Add a value to its ordered position in a list


Accepts :

	hdrP		Ptr to list head structure
	valP		Ptr to value to be added to list
	cmpP		Ptr to a comparitor routine

Returns :

	<value>		Ptr to list node structure where value is added,
or
	NULL		if there was an error making a node

Notes :

	It's assumed that "valP" is safe to point to forever, or at
	least for the life of the list.

	The comparitor routine is passed two arguments:
	   valP -- the valP as above
	   val2P -- the value of the a node in the list
	and should return:
	   less than zero -- if valP precedes val2P in sorted order
	   zero -- if valP sorts to the same place as val2P
	   greater than zero -- if valP sorts after val2P

*/

LL_NODE *
ll_addvs ARGLIST( (hdrP, valP, cmpP) )
   NFARG( LL_HEAD	*hdrP )		/* Ptr to list head */
   NFARG( void		*valP )		/* Ptr to value to add */
    FARG( int		(*cmpP)() )	/* Ptr to comparison routine */
{
	LL_NODE		*nodeP;		/* Ptr to new node */
	LL_NODE		*nextP;		/* Ptr to next node */
	LL_NODE		**nodePP;	/* Ptr to node ptr */

    if ( mml_mutex_lock( &hdrP->llh_lock ) != 0 )
	warning( "ll_addvs: can't get lock" );

    for ( nodePP = &hdrP->llh_firstP; (nextP = *nodePP ) != NULL; 
		nodePP=&(nextP->lln_nextP) ) {
	if ( (*cmpP)( valP, nextP->lln_valP ) <= 0 )
	    break;
    }

    /* New node links into *nodePP and before nextP */
    nodeP = ll_node( "from ll_addvs" );
    if ( nodeP != NULL ) {
	nodeP->lln_nextP = nextP;
	*nodePP = nodeP;

	if ( nextP != NULL )
	    nextP->lln_prevP = nodeP;
	else
	    /* No "next" -- new node is now the tail */
	    hdrP->llh_lastP = nodeP;

	nodeP->lln_hdrP = hdrP;

	++hdrP->llh_nodeC;
    }

    if ( mml_mutex_unlock( &hdrP->llh_lock ) != 0 )
	warning( "ll_addvs: can't release lock" );

    return ( nodeP );
}
/*

*//* ll_addvt( hdrP, valP )

	Add a value to the tail of a list


Accepts :

	hdrP		Ptr to list head structure
	valP		Ptr to value to be added to list

Returns :

	<value>		Ptr to list node structure where value is added,
or
	NULL		if there was an error making a node

Notes :

	It's assumed that "valP" is safe to point to forever, or at
	least for the life of the list.

*/

LL_NODE *
ll_addvt ARGLIST( (hdrP, valP) )
   NFARG( LL_HEAD	*hdrP )		/* Ptr to list head */
    FARG( void		*valP )		/* Ptr to value to add */
{
	LL_NODE		*nodeP;		/* Ptr to new node */

    nodeP = ll_node( "from ll_addvt" );

    if ( mml_mutex_lock( &hdrP->llh_lock ) != 0 )
	warning( "ll_addvt: can't get lock" );

    if ( nodeP != NULL ) {
	nodeP->lln_valP = valP;		/* Set the value */

	/* New node's "prev" pointer is current last node */
	if ( ( nodeP->lln_prevP = hdrP->llh_lastP ) != NULL )
	    /* Link old last node forward to this one */
	    (nodeP->lln_prevP)->lln_nextP = nodeP;
	else
	    /* List was empty, so new node is also head node */
	    hdrP->llh_firstP = nodeP;

	hdrP->llh_lastP = nodeP;
	nodeP->lln_hdrP = hdrP;

	++hdrP->llh_nodeC;
    }

    if ( mml_mutex_unlock( &hdrP->llh_lock ) != 0 )
	warning( "ll_addvt: can't release lock" );

    return ( nodeP );
}
/*

*//* ll_rem( nodeP )

	Remove a node from a linked list

Accepts :

	nodeP		Ptr to node to remove

Returns :

	<nothing>

Notes :

	Calls ll_rem_nolock for most of the work.

*/

void
ll_rem ARGLIST( (nodeP) )
    FARG( LL_NODE	*nodeP )	/* Node to remove */
{
	LL_HEAD		*hdrP;		/* Ptr to list header */
	LL_NODE		*prevP, *nextP;	/* previous/next nodes */

    hdrP = nodeP->lln_hdrP;

    /* Lock the list and hand off to ll_rem_nolock to do
       the actual work.
    */
    if ( mml_mutex_lock( &hdrP->llh_lock ) != 0 )
	warning( "ll_rem: can't get lock" );
    ll_rem_nolock( nodeP );
    if ( mml_mutex_unlock( &hdrP->llh_lock ) != 0 )
	warning( "ll_rem: can't release lock" );

    /* done */
}
/*

*//* ll_rem_nolock( nodeP )

	Remove a node from a linked list without locking the list.
	Useful when the caller has already locked the list.

Accepts :

	nodeP		Ptr to node to remove

Returns :

	<nothing>

Notes :

	Places the node on a free list for quick access for another user.

*/

void
ll_rem_nolock ARGLIST( (nodeP) )
    FARG( LL_NODE	*nodeP )	/* Node to remove */
{
	LL_HEAD		*hdrP;		/* Ptr to list header */
	LL_NODE		*prevP, *nextP;	/* previous/next nodes */

    hdrP = nodeP->lln_hdrP;
    prevP = nodeP->lln_prevP;
    nextP = nodeP->lln_nextP;

    if ( prevP == NULL )
	/* Removing first node, update head */
	hdrP->llh_firstP = nextP;
    else
	/* Not removing first node, update previous node */
	prevP->lln_nextP = nextP;

    if ( nextP == NULL )
	/* Removing last node; update tail */
	hdrP->llh_lastP = prevP;
    else
	/* There's a node following, update it to point back. */
	nextP->lln_prevP = prevP;

    --hdrP->llh_nodeC;

    ll_node_free( nodeP );		/* Release the node */

    /* done */
}
