/* mml-symtab.c -- mml library (mml) hashed symbol table routines

	Copyright 2003,2004,2005,2006,2007
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Contains routines to manipulate hashed symbol tables

*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>

#include <mml/mml.h>

#include <mml/mml-err.h>
#include <mml/mml-alloc.h>
#include <mml/mml-symtab.h>

/* Local definitions */



/* External data referenced */



/* External routines used */



/* Local data publicly available */



/* Local routines and forward references */



/* Private data */



/*

*//* symtab_comp_str( symP, keyP )

	Comparison routine for SYMSTR types

Accepts :


	symP		Ptr to symbol table entry
	keyP		Ptr to potential key

Returns :

	<value>		< 0 if key in symP precedes (is less than) keyP
			0   if key in symP matches (equals) keyP
			>0  if key in symP follows (is greater than) keyP

Notes :


*/


int
symtab_comp_str ARGLIST( ( symP, keyP ) )
   NFARG( SYMSTR	*symP )		/* Ptr to SYMSTR header */
    FARG( char		*keyP )		/* Ptr to key to compare with */
{
    return( strcmp( symP->ss_nameP, keyP ) );
}
/*

*//* symtab_create( flags, hashsize, hashrtc, comprtc, keyassoc, keydis )

	Create a new symbol table

Accepts :

	flags		Symbol table flags
	hashsize	Size of the hash table (number of hash buckets)
	hashrtc		A function which produces a hash value for a key
	comprtc		A function to compare a key with a symtab entry
	keyassoc	A function to associate a key with a symtab entry
	keydis		A function to remove a key from a symtab entry

Returns :

	<value>		Address of symbol table object if successful
			NULL if not

Notes :

	hashrtc points to a function that, given a new key, returns
	a UWORD hash value.  The hash value will be mapped into the
	internal hash size after return from hashrtc.

	comprtc points to a function that accepts a pointer to an existing
	symbol table entry and a pointer to a new key, and returns
	less than zero if existing symbol compares less than the new key,
	zero if they are equal, and gretare than zero if greater than.

	keyassoc points to a function that is passed the address of
	a symbol table entry and the address of a key, and associates
	the key with the entry.  (attaches the key to the entry.)

	keydis points to a function that is passed the address of a
	symbol table entry, and disassociates a key from the entry.
	(removes the key attachment.)

*/

SYMTAB *
symtab_create ARGLIST( ( flags, hashsize, hashrtc, comprtc, keyassoc, keydis ) )
   NFARG( UWORD		flags )		/* Symbol table flags */
   NFARG( UWORD		hashsize )	/* Number of hash buckets */
   NFARG( UWORD		(*hashrtc)() )	/* Hash routine */
   NFARG( int		(*comprtc)() )	/* Key compare routine */
   NFARG( BOOL		(*keyassoc)() )	/* Key association routine */
    FARG( BOOL		(*keydis)() )	/* Key disassocition routine */
{
	int		i;		/* Scratch */
	SYMTAB		*tableP;	/* Ptr to table */

    /* Allocate the table control structure */
    tableP = (SYMTAB *)emalloc( "symtab_create", "new table",
			          sizeof(SYMTAB) );
    if ( tableP == NULL )
	return ( NULL );

    /* Allocate space for buckets */
    tableP->st_bucketPP = (SYMENT **)emalloc( "symtab_create", "buckets",
					     hashsize * sizeof(SYMENT *) );
    if ( tableP->st_bucketPP == NULL ) {
	dealloc( tableP );
	return( NULL );
    }

    /* Initialize stuff */
    tableP->st_flags = flags;
    tableP->st_hashsize = hashsize;
    for( i = 0; i < hashsize; ++i )
    	tableP->st_bucketPP[i] = NULL;

    tableP->st_hash = hashrtc;
    tableP->st_compare = comprtc;
    tableP->st_keyassoc = keyassoc;
    tableP->st_keydis = keydis;

    return( tableP );
}
/*

*//* symtab_delete( tableP )

	Delete a symbol table

Accepts :

	tableP		Address of symbol table

Returns :

	<nothing>

Notes :

	This routine is probably not going to be used by most applications.

	All table entries are removed and freed

	Deallocation errors are ignored

*/

void
symtab_delete ARGLIST( (tableP) )
    FARG( SYMTAB	*tableP )	/* Ptr to table object */
{
	int		i;		/* Scratch */
	SYMENT		*symP;		/* Ptr to symbol table entry */

    if ( tableP == NULL )
	return;

    /* Locking/unlocking may not make sense, but go for it. */
    if ( mml_mutex_lock( &tableP->st_lock ) != 0 )
	warning( "symtab_delete: can't get lock" );

    /* Remove all the entries in the table, bucket by bucket */
    for( i = 0; i < tableP->st_hashsize; ++i ) {
        while( ( symP = tableP->st_bucketPP[i] ) != NULL ) {
	    /* Unlink this entry from this bucket list */
	    tableP->st_bucketPP[i] = symP->se_nextP;

	    /* Disassociate the key */
	    (*tableP->st_keydis)( symP );

	    /* Free up the entry */
	    dealloc( symP );
	}
    }

    if ( mml_mutex_unlock( &tableP->st_lock ) != 0 )
	warning( "symtab_delete: can't release lock" );

    /* Lose the table */
    dealloc( tableP );
}
/*

*//* symtab_enter( tableP, keyP, size )

	Enter a new entry into a symbol table

Accepts :

	tableP		Address of symbol table block
	keyP		Address of key for new entry
	size		Size of the new symbol table entry

Returns :

	<value>		Address of new entry if successful
			NULL if failure (may take error exit)

Notes :

	Each symbol table entry is expected to begin with a SYMENT
	structure.  The enclosing structure may be (and usually will be)
	larger, thus the required size of the entry is passed here.

	If the symbol is already in the table, the address of that
	symbol is returned.  That characteristic makes this a
	"enter if not already present" function.

*/

SYMENT *
symtab_enter ARGLIST( ( tableP, keyP, size ) )
   NFARG( SYMTAB	*tableP )	/* Pointer to table block */
   NFARG( void		*keyP )		/* Opaque key */
    FARG( int		size )		/* Size of symbol table entry */
{
	UWORD		bucketN;	/* Bucket number */
	int		cmp;		/* Comparison result */
	SYMENT		*symP;		/* Symbol ptr */
	SYMENT		**symPP;	/* Where to link new entry to */

    if ( tableP == NULL )
	return ( NULL );

    if ( mml_mutex_lock( &tableP->st_lock ) != 0 )
	warning( "symtab_enter: can't get lock" );

    /* Find the bucket header */
    bucketN = (*tableP->st_hash)( keyP ) % tableP->st_hashsize;

    /* Search for place to put the new entry */
    for( cmp = 1, symPP = &(tableP->st_bucketPP[bucketN]);
		( symP = *symPP ) != NULL;
    			symPP = &(symP->se_nextP) ) {
	cmp = (*tableP->st_compare)( symP, keyP );
	if ( cmp >= 0 )
	    break;
    }

    /* If not found, enter it */
    if ( cmp != 0 ) {
	/* Make new symbol */
	symP = (SYMENT *)emalloc( "symtab_enter", "new symbol", size );
	if ( symP != NULL ) {
	    /* Zero it */
	    memset( symP, 0, size );
	    
	    /* Link it in */
	    symP->se_nextP = *symPP;
	    *symPP = symP;

	    /* Associate the key with the entry */
	    if ( !(*tableP->st_keyassoc)( symP, keyP ) ) {
		/* Some failure.. abort */
		dealloc( symP );
		symP = NULL;
	    }
	}
    }

    if ( mml_mutex_unlock( &tableP->st_lock ) != 0 )
	warning( "symtab_enter: can't release lock" );

    return( symP );
}
/*

*//* symtab_hash_str( keyP )

	Hash routine for SYMSTR types

Accepts :

	keyP		Ptr to potential key

Returns :

	<value>		16-bit hash value generated from the key

Notes :


	Could use improving

*/


UWORD
symtab_hash_str ARGLIST( ( keyP ) )
    FARG( char		*keyP )		/* Ptr to key to compare with */
{
	UBYTE		ch;		/* Character value */
	int		hashval;	/* Hash value */

    /* Do trivial character sum/fold */
    for( hashval = 0; ( ch = *keyP++ ) != NUL; )
    	hashval = ( hashval << 1 ) + ch;

    return( (UWORD)hashval );
}
/*

*//* symtab_keyassoc_str( symP, keyP )

	Key association routine for string type key

Accepts :

	symP		Ptr to symbol table entry
	keyP		Ptr to potential key

Returns :

	<value>		TRUE if OK
			FALSE if error (may take error exit)

Notes :

	This is a standard routine that may be specified as the "key
	association routine to call" in the symbol table definition.

	Makes a copy of the string.

*/


BOOL
symtab_keyassoc_str ARGLIST( ( symP, keyP ) )
   NFARG( SYMSTR	*symP )		/* Ptr to symbol table entry */
    FARG( char		*keyP )		/* Ptr to key to compare with */
{
    /* Save a copy and return the status. */
    symP->ss_nameP = newstr( "symtab_keyassoc_str", "new key", keyP );
    return ( symP->ss_nameP != NULL );
}
/*

*//* symtab_keydis_str( symP )

	Key disassociation routine for string type key

Accepts :

	symP		Ptr to symbol table entry

Returns :

	<value>		TRUE if OK
			FALSE if error (may take error exit)

Notes :

	This is a standard routine that may be specified as the "key
	disassociation routine to call" in the symbol table definition.

*/


BOOL
symtab_keydis_str ARGLIST( ( symP ) )
    FARG( SYMSTR	*symP )		/* Ptr to symbol table entry */
{
	char		*keyP;		/* Ptr to key string */

    if ( ( keyP = symP->ss_nameP ) == NULL ) {
	warning( "symtab_keydis_str: no associated key.\n" );
	return ( FALSE );
    }

    symP->ss_nameP = NULL;		/* Association be gone */
    dealloc( keyP );			/* Key be gone */
    return ( TRUE );
}
/*

*//* symtab_lookup( tableP, keyP )

	Find a symbol table entry given its key

Accepts :

	tableP		Address of symbol table block
	keyP		Address of a key

Returns :

	<value>		Address of the symbol table entry if found
			NULL if not

Notes :

	NULL is returned if the table has not yet been initialized.

*/

SYMENT *
symtab_lookup ARGLIST( (tableP, keyP) )
   NFARG( SYMTAB	*tableP )	/* Pointer to table block */
    FARG( void		*keyP )		/* Opaque key */
{
	UWORD		bucketN;	/* Bucket number */
	int		cmp;		/* Comparison result */
	SYMENT		*symP;		/* Sym ptr */

    if ( tableP == NULL )
	return ( NULL );

    if ( mml_mutex_lock( &tableP->st_lock ) != 0 )
	warning( "symtab_lookup: can't get lock" );

    /* Find the bucket header */
    bucketN = (*tableP->st_hash)( keyP ) % tableP->st_hashsize;

    /* Search for the entry */
    for( symP = tableP->st_bucketPP[bucketN]; symP != NULL;
    			symP = symP->se_nextP ) {
	cmp = ( *tableP->st_compare )( symP, keyP );
	if ( cmp == 0 ) {
	    mml_mutex_unlock( &tableP->st_lock );
	    return( symP );		/* Found it */
	}
	if ( cmp > 0 )
	    break;			/* Passed it. */
    }

    /* Not found */
    if ( mml_mutex_unlock( &tableP->st_lock ) != 0 )
	warning( "symtab_lookup: can't release lock" );
    return( NULL );
}
/*

*//* symtab_remove( tableP, keyP )

	Remove a symbol table entry given its key

Accepts :

	tableP		Address of symbol table block
	keyP		Address of a key

Returns :

	<nothing>

Notes :

*/

void
symtab_remove ARGLIST( (tableP, keyP) )
   NFARG( SYMTAB	*tableP )	/* Pointer to table block */
    FARG( void		*keyP )		/* Opaque key */
{
	UWORD		bucketN;	/* Bucket number */
	int		cmp;		/* Comparison result */
	SYMENT		*symP;		/* Sym ptr */
	SYMENT		**symPP;	/* Where symbol is linked. */

    if ( tableP == NULL )
	return;

    if ( mml_mutex_lock( &tableP->st_lock ) != 0 )
	warning( "symtab_remove: can't get lock" );

    /* Find the bucket header */
    bucketN = (*tableP->st_hash)( keyP ) % tableP->st_hashsize;

    /* Search for the entry, remembering where it's linked */
    for( symPP = &(tableP->st_bucketPP[bucketN]);
		( symP = *symPP ) != NULL;
    			symPP = &(symP->se_nextP) ) {
	cmp = (*tableP->st_compare)( symP, keyP );
	if ( cmp >= 0 )
	    break;
    }

    if ( ( symP != NULL ) && ( cmp == 0 ) ) {
	/* Found it-- remove it. */
	*symPP = symP->se_nextP;	/* Take it out of the list */
	(*tableP->st_keydis)( symP );	/* Disassociate from key */
	dealloc( symP );		/* Free the entry's memory */
    }

    if ( mml_mutex_unlock( &tableP->st_lock ) != 0 )
	warning( "symtab_remove: can't release lock" );
}
