/* b64test.c -- Test of base64 encode/decode functions

	Copyright 2005,2006
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <string.h>

#include <mml/mml.h>

#include <mml/mml-base64.h>


/* Local Definitions */


/* External data referenced */

extern	char	*optarg;		/* getopt() argument */
extern	int	optind;			/* getopt() index */

/* External routines used */


/* Local data publicly available */


/* Local routines and forward references */

static	void	b64decode PROTO( (UBYTE *argP) );
static	void	b64encode PROTO( (UBYTE *argP) );
static	void	printwrap PROTO( (char *strP, int strL) );
static	void	usage PROTO( (void) );

/* Private data */

static	BOOL	NulF = FALSE;		/* NUL-terminate the input */
static	int	ColC = 0;		/* Column counter for output */

main( argc, argv )
	int		argc;
	char		**argv;
{
	BOOL		usageOK;
	BOOL		encodeF;
	int		optch;
	char		*argP;		/* Any string arg */

    encodeF = TRUE;			/* Default is encode mode */
    NulF = FALSE;			/* Don't NUL-terminate */

    usageOK = TRUE;

    while ( usageOK &&
	   ( optch = getopt( argc, argv, "dez" ) ) != EOF ) {
	switch ( optch ) {
	    case	'd':		/* Decode mode */
		encodeF = FALSE;
		break;

	    case	'e':		/* Encode mode */
		encodeF = TRUE;
		break;

	    case	'z':		/* NUL-terminate the input */
		NulF = TRUE;
		break;

	    default:
		usageOK = FALSE;
		break;
	}
    }

    if ( usageOK ) {
	if ( optind < argc )
	    argP = argv[ optind++ ];
	else
	    argP = NULL;
    }

    if ( !usageOK ) {
	usage();
	exit( 1 );
    }

    if ( encodeF ) {
	b64encode( (UBYTE *)argP );
    }
    else {
	b64decode( (UBYTE *)argP );
    }

    exit( 0 );
}
/*

*//* usage()

	Print usage message

Accepts :

Returns :

*/

static void
usage NOARGLIST
{
    printf( "usage: b64test [flags]\n" );
    printf( "    Flags are:\n" );
    printf( "       -d               Decode mode.\n" );
    printf( "       -e               Encode mode (default).\n" );
    printf( "       -z               NUL-terminate the input.\n" );
    printf( "\n" );
}
/*

*//* b64decode( argP )

	Decode the source string, or stdin if no source string.

Accepts :

	argP		Ptr to encoded string, or NULL for stdin

Returns :

	<nothing>

Notes :


*/

static void
b64decode ARGLIST( (argP) )
    FARG( UBYTE		*argP )		/* Any string arg */
{
	UBYTE		b;
	int		ch;
	int		len;
	MMLB64		b64;
	char		buf[10];

    b64_init( &b64 );

    for ( ; ; ) {
	if ( argP != NULL ) {
	    if ( ( b = *argP++ ) == NUL )
		break;
	}
	else {
	    if ( ( ch = getchar() ) == EOF )
		break;

	    /* Ignore newlines and whitespace from files, for kicks */
	    if ( ( ch == '\r' ) ||
		 ( ch == '\n' ) ||
		 isspace( ch ) )
		continue;

	    b = ch;
	}

	len = b64_decode_byte( &b64, b, &buf[0], sizeof(buf) );
	if ( len < 0 ) {
	    fprintf( stderr, "b64_decode_byte returned %d\n", len );
	    return;
	}
	if ( len > 0 )
	    fwrite( &buf[0], 1, len, stdout );
    }

    len = b64_decode_finish( &b64, &buf[0], sizeof(buf) );
    if ( len < 0 ) {
	fprintf( stderr, "b64_decode_finish returned %d\n", len );
	return;
    }
    if ( len > 0 )
	fwrite( &buf[0], 1, len, stdout );

    ColC = 0;
}
/*

*//* b64encode( argP )

	Encode the source string, or stdin if no source string.

Accepts :

	argP		Ptr to source string, or NULL for stdin

Returns :

	<nothing>

Notes :


*/

static void
b64encode ARGLIST( (argP) )
    FARG( UBYTE		*argP )		/* Any string arg */
{
	UBYTE		b;
	int		ch;
	int		len;
	MMLB64		b64;
	char		buf[10];

    b64_init( &b64 );

    for ( ; ; ) {
	if ( argP != NULL ) {
	    if ( ( b = *argP++ ) == NUL )
		break;
	}
	else {
	    if ( ( ch = getchar() ) == EOF )
		break;
	    b = ch;
	}

	len = b64_encode_byte( &b64, b, &buf[0], sizeof(buf) );
	if ( len < 0 ) {
	    fprintf( stderr, "b64_encode_byte returned %d\n", len );
	    return;
	}
	if ( len > 0 )
	    printwrap( &buf[0], len );
    }

    if ( NulF ) {
	len = b64_encode_byte( &b64, 0, &buf[0], sizeof(buf) );
	if ( len < 0 ) {
	    fprintf( stderr, "b64_encode_byte returned %d\n", len );
	    return;
	}
	if ( len > 0 )
	    printwrap( &buf[0], len );
    }

    len = b64_encode_finish( &b64, &buf[0], sizeof(buf) );
    if ( len < 0 ) {
	fprintf( stderr, "b64_encode_finish returned %d\n", len );
	return;
    }
    if ( len > 0 )
	printwrap( &buf[0], len );

    printf( "\n" );
    ColC = 0;
}
/*

*//* printwrap(strP, strL)

	Print a string to stdout, with dumb line wrapping

Accepts :

	strP		String to output
	strL		Length of string

Returns :

	<nothing>

Notes :


*/

static void
printwrap ARGLIST( (strP, strL) )
   NFARG( char		*strP )		/* String to output */
    FARG( int		strL )		/* Number of bytes */
{
	char		ch;

    for ( ; strL-- > 0; ++ColC ) {
	ch = *strP++;
	if ( ColC >= 72 ) {
	    putchar( '\n' );
	    ColC = 0;
	}
	putchar( ch );
    }
}
