/* avldb.c -- Simple access routines using AVL library

	Copyright 2003,2004,2005,2006,2007
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

This file has routines for a simple single interface to the
avl routines.

It keeps track of one tree, and has a set of routines to insert,
find, delete objects in that tree, and to walk the tree.

Objects are tracked via a two-part key.  The parts of the key are:
information type, information name.  By attaching a type to each
name, multiple namesets (and datasets) can be tracked in a single
tree.

Given a complete key (type, name), access is provided via the
address of a pointer variable.  Routines calling this module can
make use of that pointer variable in any way desired.

This is not for programs wanting a fancier interface to AVL routines, or
wanting to maintain multiple trees.  However they can use these
routines as a guide.

Included are:

	avldb_init		Init this module
	avldb_find		Find an entry
	avldb_create		Create an entry
	avldb_delete		Delete an entry
	avldb_walk		Walk through entries.

	avldb_set_cmpkey	Specify the key compare routine

Requires the "mem" support routines in misc.c and strops.c

*/

#include <stdio.h>
#include <sys/types.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>

#include <mml/mml.h>

#include <mml/mml-alloc.h>
#include <mml/mml-avl.h>


/* Local definitions */


    /* Structure for the avldb AVL tree */
typedef
  struct {
    AVLTREE	adbt_tree;		/* AVL tree header */
  }  ADBTHDR;


    /* Structure used to hold an access key for tree ops. */
typedef
  struct {
    int		ak_type;		/* Entry type */
    char	*ak_nameP;		/* Entry name */
  }  ADBKEY;


    /* Structure for a node in the avldb AVL tree */
typedef
  struct {
    AVLNODE	at_avl;			/* AVL node header */
    int		at_type;		/* Type of entry */
    char	*at_nameP;		/* Ptr to the name */
    void	*at_thingP;		/* Pointer to the thing. */
  }  ADBNODE;

/* External data referenced */


/* External routines used */


/* Local data publicly available */


/* Local routines and forward references */

static	int	avldb_walk0 PROTO( ( AVLTREE *treeP, ADBNODE *nodeP,
					int depth ) );


    /* AVL support routines */
static	int	adbt_cmpkey PROTO( (ADBKEY *akeyP, ADBNODE *nodeP ) );

static	AVLNODE	*adbt_mknode PROTO( (AVLTREE *treeP, ADBKEY *akeyP,
				     void *dataP, AVLNODE *enodeP ) );

static	int	adbt_rmnode PROTO( (AVLTREE *treeP, ADBNODE *nodeP ) );

/* Private data */

static	ADBTHDR	Adbthdr;		/* Header for tree */
static	int	(*Cmpkey_rtc)();	/* Override of cmpkey routine */
static	int	(*Walk_rtc)();		/* Routine for avldb_walk */

/*

*//* avldb_init()

	Initialize the avldb module

Accepts :

Returns :

	< nothing >
*/

void
avldb_init NOARGLIST
{
    /* Create the AVL trees */
    avlnew( &Adbthdr.adbt_tree, adbt_cmpkey, adbt_mknode, adbt_rmnode );
}
/*

*//* avldb_set_cmpkey( cmpkey_rtc )

	Sets the key-compare routine to override the internal one

Accepts :

	cmpkey_rtc	ptr to new cmpkey

Returns :

	< nothing >

*/

void
avldb_set_cmpkey ARGLIST( ( cmpkey_rtc ) )
    FARG( int	(*cmpkey_rtc)() )
{
    Cmpkey_rtc = cmpkey_rtc;
}
/*

*//* avldb_find( etype, enameP )

	Find an entry in the avldb tree

Accepts :

	etype		Type of entry to look for
	enameP		Name attached to the entry

Returns :

	< value >	Address of the pointer variable, or
			NULL if not found.

Notes:

	e.g. to find a named entry of type 2:

	    struct thing *thingP;
	    struct thing **thingPP;
	        .
		.
		.
	    if ( ( thingPP = avldb_find( 2, "name-of-thing" ) ) != NULL )
		thingP = *thingPP;
	    else
		thingP = NULL;
	    
*/

void *
avldb_find ARGLIST( ( etype, enameP ) )
   NFARG( int		etype )		/* Type of entry */
    FARG( char		*enameP )	/* Name of entry */
{
	ADBKEY		akey;		/* Key for access */
	ADBNODE		*adbP;		/* Ptr to node */

    /* Try to find the entry */
    akey.ak_type = etype;
    akey.ak_nameP = enameP;
    adbP = (ADBNODE *)avlfind( &Adbthdr.adbt_tree, &akey );
    if ( adbP == NULL )
	return ( NULL );

    /* Return the address of the pointer variable */
    return ( &adbP->at_thingP );
}
/*

*//* avldb_insert( etype, enameP )

	Insert an entry into the avldb tree

Accepts :

	etype		Type of entry to look for
	enameP		Name attached to the entry

Returns :

	< value >	Address of the pointer variable, or
			NULL if not done.
Notes:

	e.g. to insert a named entry of type 2:

	    struct thing *thingP;
	    struct thing **thingPP;
	       .
	       .
	       .
	    if ( ( thingPP = avldb_find( 2, "name-of-thing" ) ) == NULL ) {
		(* not already there, insert it *)
		thingPP = avldb_insert( 2, "name-of-thing" );
		*thingPP = thingP;
		thingP = *thingPP;
	    }
	    
*/

void *
avldb_insert ARGLIST( ( etype, enameP ) )
   NFARG( int		etype )		/* Type of entry */
    FARG( char		*enameP )	/* Name of entry */
{
	ADBKEY		akey;		/* Key for access */
	ADBNODE		*adbP;		/* Ptr to node */

    /* Perform the insert */
    akey.ak_type = etype;
    akey.ak_nameP = newstr( "avldb_insert", "name", enameP );
    avlinsert( &Adbthdr.adbt_tree, &akey, (void *)NULL );

    /* Try to find the entry */
    adbP = (ADBNODE *)avlfind( &Adbthdr.adbt_tree, &akey );
    if ( adbP == NULL )
	return ( NULL );

    /* Return the address of the pointer variable */
    return ( &adbP->at_thingP );
}
/*

*//* avldb_delete( etype, enameP )

	Delete an entry from the avldb.

Accepts :

	etype		Type of entry to look for
	enameP		Name attached to the entry

Returns :

	< value >	TRUE if succeeded,  FALSE if not.

Notes:

	e.g. to delete a named entry of type 2:

	    avldb_delete( 2, "name-of-thing" );

*/

int
avldb_delete ARGLIST( ( etype, enameP ) )
   NFARG( int		etype )		/* Type of entry */
    FARG( char		*enameP )	/* Name of entry */
{
	ADBKEY		akey;		/* Key for access */

    /* Perform the delete */
    akey.ak_type = etype;
    akey.ak_nameP = enameP;
    return ( avldelete( &Adbthdr.adbt_tree, &akey ) == 0 );
}
/*

*//* avldb_walk( rtc )

	Walk the avldb tree.

Accepts :

	rtc		Address of the routine to call

Returns :

	< nothing >
*/

void
avldb_walk ARGLIST( ( rtc ) )
    FARG( int		(*rtc)() )	/* Routine to call */
{
    /* Store the routine to call */
    Walk_rtc = rtc;

    /* Do the avlwalk via local avldb_walk0() routine */
    avlwalk( &Adbthdr.adbt_tree, avldb_walk0 );
}

/* The avldb_walk0() routine */
static int
avldb_walk0 ARGLIST( ( treeP, nodeP, depth ) )
   NFARG( AVLTREE	*treeP )	/* Tree header */
   NFARG( ADBNODE	*nodeP )	/* Tree node */
    FARG( int		depth )		/* Depth */
{
	int		status;

    if ( nodeP == NULL ) 		/* Indicates end-of-walk */
	status = (*Walk_rtc)( -1, (char *)NULL, (void **) NULL );
    else
	status = (*Walk_rtc)( nodeP->at_type, nodeP->at_nameP,
					&nodeP->at_thingP );

    return ( status ? 0 : 1 );
}
/*

*//******************************************************************
*                                                                   *
*       support routines for avldb AVL tree             *
*                                                                   *
*********************************************************************/
/*

*//* adbt_cmpkey( akeyP, nodeP )

	Compare name keys for avl tree routines

Accepts :

	akeyP		Ptr to target avldb key block
	nodeP		Ptr to node to test

Returns :

	< value >	-1, 0, or 1, as dictated by the rules of
			interface between the avl tree routines and
			the key comparison routine.

Note:

	This routine assumes that the first member of the node structure
	is the AVL tree node structure, passed.  That is, that nodeP
	points to the top of the node struct.

*/

static int
adbt_cmpkey ARGLIST( ( akeyP, nodeP ) )
   NFARG( ADBKEY	*akeyP )	/* Addr of new avldb key block */
    FARG( ADBNODE	*nodeP )	/* Addr of node */
{
	int		cmpval;

    if ( akeyP->ak_type < nodeP->at_type )
	return ( -1 );
    else if ( akeyP->ak_type > nodeP->at_type )
	return ( 1 );

    /* Now sort the key strings.  If any key comparison routine
       has been specified, use that.
    */
    if ( Cmpkey_rtc != NULL )
	return ( (*Cmpkey_rtc)( akeyP->ak_nameP, nodeP->at_nameP ) );

    /* Sort by name */
    cmpval = strcmp( akeyP->ak_nameP, nodeP->at_nameP );
    if ( cmpval < 0 )
	return ( -1 );
    if ( cmpval > 0 )
	return ( 1 );
    return ( 0 );
}
/*

*//* adbt_mknode( treeP, akeyP, dataP, enodeP )

	Make a avldb node on behalf of AVL tree routines

Accepts :

	treeP		Address of tree header block
	akeyP		Address of the avldb key block
	dataP		Address of data (not used)
	enodeP		Address of any existing node

Returns :

	<value>		Address of the new node

Notes :

	This routine assumes that the first member of the node structure
	is the AVL tree node structure, passed.  That is, that nodeP
	points to the top of the node struct.
*/

static AVLNODE *
adbt_mknode ARGLIST( ( treeP, akeyP, dataP, enodeP ) )
   NFARG( AVLTREE	*treeP )	/* Addr of tree header block */
   NFARG( ADBKEY	*akeyP )	/* Addr of avldb key block */
   NFARG( void		*dataP )	/* Addr of new data (n/a) */
    FARG( AVLNODE	*enodeP )	/* Existing node... */
{
	int		nodesize;	/* Size of node required */
	ADBNODE		*nodeP;		/* New node that we make */

    /* If enodeP is non-NULL, this is a call to overwrite the data.
       That should never happen, so return NULL in that case. */
    if ( enodeP != NULL )
	return( (AVLNODE *)NULL );

    /* Calculate new node size (we co-allocate space for the name) */
    nodesize = sizeof( ADBNODE ) + strlen( akeyP->ak_nameP ) + 1;
    nodeP = (ADBNODE *)emalloc( "adbt_mknode", "avldb node",
					 nodesize );

    /* Setup the node struct */
    nodeP->at_type = akeyP->ak_type;
    nodeP->at_nameP = (char *)&nodeP[1];
    strcpy( nodeP->at_nameP, akeyP->ak_nameP );
    nodeP->at_thingP = NULL;

    return( &nodeP->at_avl );
}
/*

*//* adbt_rmnode( treeP, nodeP )

	Delete a node on behalf of AVL tree routines.

Accepts :

	treeP		Address of tree header block
	nodeP		Address of the node being removed

Returns :

	<value>			0 for success, -1 for failure

Notes :

	This routine assumes that the first member of the node structure
	is the AVL tree node structure, passed.  That is, that nodeP
	points to the top of the node struct.

*/

static int
adbt_rmnode ARGLIST( ( treeP, nodeP ) )
   NFARG( AVLTREE	*treeP )
    FARG( ADBNODE	*nodeP )
{
    /* Deallocate the node */
    dealloc( (char *)nodeP );

    /* Return OK */
    return( 0 );
}
