/* mml-fmt.c -- Formatting/conversion routines

	Copyright 2003,2004,2005,2006,2007
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Contains routines for string formatting and conversion

*/

#include <stdlib.h>
#include <sys/types.h>
#include <time.h>
#include <sys/time.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include <mml/mml.h>
#include <mml/mml-ll.h>
#include <mml/mml-str.h>
#include <mml/mml-fmt.h>


/* Local definitions */



/* External data referenced */



/* External routines used */

extern	time_t	tmclock();

/* Local data publicly available */



/* Local routines and forward references */

static	int	get_sub_num PROTO( (char **strPP, int digitC) );

/* Private data */



/*

*//* atod(strP)

	Convert string of decimal digits to a value

Accepts :

	strP		Ptr to the digit string

Returns :

	<value>		The value

Notes :

	May use strtol() if compiled that way

*/

long
atod ARGLIST( (strP) )
    FARG( char		*strP )		/* digit string */
{

#ifdef	HAVE_STRTOL
	char		*endP;		/* Ptr to end */

    return ( strtol( strP, &endP, 10 ) );

#else

	BOOL		negF;		/* If negative */
	char		ch;		/* Character */
	long		value;		/* The value determined. */

    strP = solbat( strP, FALSE );	/* Skip spaces/tabs */

    /* Process leading sign */
    if ( *strP == '-' ) {
	negF = TRUE;
	++strP;
    }
    else {
	negF = FALSE;
	if ( *strP == '+' )
	    ++strP;
    }

    for( value = 0; ; ) {
    	ch = *strP++;
	if ( ( ch >= '0' ) & ( ch <= '9' ) )
	    ch = ch - '0';
	else
	    break;

	value = ( value * 10 ) + ch;
    }

    /* Apply negative sign if any */
    if ( negF )
	value = -value;

    return( value );

#endif		/* HAVE_STRTOL */
}
/*

*//* atodu(strP)

	Convert string of decimal digits to an unsigned long value

Accepts :

	strP		Ptr to the digit string

Returns :

	<value>		The value

Notes :


*/

ULONG
atodu ARGLIST( (strP) )
    FARG( char		*strP )		/* digit string */
{

#if 0
	/* Resist the temptation to use strtoll */
#else
	char		ch;		/* Character */
	ULONG		value;		/* The value determined. */

    strP = solbat( strP, FALSE );	/* Skip spaces/tabs */

    for( value = 0; ; ) {
    	ch = *strP++;
	if ( ( ch >= '0' ) & ( ch <= '9' ) )
	    ch = ch - '0';
	else
	    break;

	value = ( value * 10 ) + ch;
    }

    return( value );

#endif	/* 0 */
}
/*

*//* atoh(strP)

	Convert string of hext digits to a value

Accepts :

	strP		Ptr to the digit string

Returns :

	<value>		The value

Notes :

	May use strtol() if compiled that way

*/

long
atoh ARGLIST( (strP) )
    FARG( char		*strP )		/* digit string */
{

#ifdef	HAVE_STRTOL
	char		*endP;		/* Ptr to end */

    return ( strtol( strP, &endP, 16 ) );

#else

	BOOL		negF;		/* If negative */
	char		ch;		/* Character */
	long		value;		/* The value determined. */

    strP = solbat( strP, FALSE );	/* Skip spaces/tabs */

    /* Process leading sign */
    if ( *strP == '-' ) {
	negF = TRUE;
	++strP;
    }
    else {
	negF = FALSE;
	if ( *strP == '+' )
	    ++strP;
    }

    for( value = 0; ; ) {
    	ch = *strP++;
	if ( ( ch >= '0' ) & ( ch <= '9' ) )
	    ch = ch - '0';
	else if ( ( ch >= 'a' ) && ( ch <= 'f' ) )
	    ch = ( ch - 'a' ) +10;
	else if ( ( ch >= 'A' ) && ( ch <= 'F' ) )
	    ch = ( ch - 'A' ) +10;
	else
	    break;

	value = ( value << 4 ) + ch;
    }

    /* Apply negative sign if any */
    if ( negF )
	value = -value;

    return( value );

#endif		/* HAVE_STRTOL */
}
/*

*//* atoo(strP)

	Convert string of octal digits to a value

Accepts :

	strP		Ptr to the digit string

Returns :

	<value>		The value

Notes :

	May use strtol() if compiled that way

*/

long
atoo ARGLIST( (strP) )
    FARG( char		*strP )		/* digit string */
{

#ifdef	HAVE_STRTOL
	char		*endP;		/* Ptr to end */

    return ( strtol( strP, &endP, 8 ) );

#else

	BOOL		negF;		/* If negative */
	char		ch;		/* Character */
	long		value;		/* The value determined. */

    strP = solbat( strP, FALSE );	/* Skip spaces/tabs */

    /* Process leading sign */
    if ( *strP == '-' ) {
	negF = TRUE;
	++strP;
    }
    else {
	negF = FALSE;
	if ( *strP == '+' )
	    ++strP;
    }

    for( value = 0; ; ) {
    	ch = *strP++;
	if ( ( ch >= '0' ) & ( ch <= '7' ) )
	    ch = ch - '0';
	else
	    break;

	value = ( value << 3 ) + ch;
    }

    /* Apply negative sign if any */
    if ( negF )
	value = -value;

    return( value );

#endif		/* HAVE_STRTOL */
}
/*

*//* fullstr_to_time( dstrP )

	Convert a "full string date" into a time value

Acccepts :

	dstrP		Ptr to date/time string

Returns :

	< value >	Date/time corresponding to the passed string

Notes :

    String is of the form:

	yyyymmdd[.hhmmss]

*/

long
fullstr_to_time ARGLIST( ( dstrP ) )
    FARG( char		*dstrP )	/* Ptr to date/time string */
{
	char		*dateP, *timeP;
	char		scopy[50];

    strncpy( &scopy[0], dstrP, sizeof(scopy) );
    dateP = &scopy[0];
    timeP = strchr( dateP, '.' );
    if ( timeP == NULL )
	timeP = "000000";
    else
	*timeP++ = NUL;

    return ( str2_to_time( dateP, timeP ) );
}
/*

*//* str2_to_time( dstrP, tstrP )

	Convert date/time descriptions into a time value

Acccepts :

	dstrP		Ptr to date string
	tstrP		Ptr to time string

Returns :

	< value >	Date/time corresponding to the passed strings

Notes :

	Date string is of the form   yymmdd
	Time string is of the form   hhmmss

*/

long
str2_to_time ARGLIST( ( dstrP, tstrP ) )
   NFARG( char		*dstrP )	/* Ptr to date string */
    FARG( char		*tstrP )	/* Ptr to time string */
{
	time_t		timenow;
	struct tm	tmblock;

    /* Get a couple of time-now values */
    time( &timenow );
    memcpy( &tmblock, localtime( &timenow ), sizeof( tmblock ) );

    /* Pick up values from the date string */
    if ( strlen( dstrP ) == 8 ) {
	tmblock.tm_year = get_sub_num( &dstrP, 4 );
	if ( tmblock.tm_year >= 1900 )
	    tmblock.tm_year -= 1900;
	if ( tmblock.tm_year < 39 )	/* 2000-base century, probably */
	    tmblock.tm_year += 100;
    }
    else
	tmblock.tm_year = get_sub_num( &dstrP, 2 );

    tmblock.tm_mon = get_sub_num( &dstrP, 2 ) -1; 
    tmblock.tm_mday = get_sub_num( &dstrP, 2 );

    /* Trap special flag dates */
    if ( ( tmblock.tm_mon == -1 ) &&
         ( tmblock.tm_year == 0 ) &&
	 ( tmblock.tm_mday == 0 ) ) {
	/* 000000: substitute june 1, 1991 */
	tmblock.tm_year = 91;
	tmblock.tm_mon = 5;
	tmblock.tm_mday = 1;
    }
    else if ( ( tmblock.tm_mon == 98 ) &&
         ( ( tmblock.tm_year == 99 ) || ( tmblock.tm_year == 8099 ) ) &&
	 ( tmblock.tm_mday == 99 ) ) {
	/* 999999 or 99999999: substitute jan 17, 2038 */

	tmblock.tm_year = 138;
	tmblock.tm_mon = 0;
	tmblock.tm_mday = 17;
    }


    tmblock.tm_hour = get_sub_num( &tstrP, 2 );
    tmblock.tm_min = get_sub_num( &tstrP, 2 );
    tmblock.tm_sec = get_sub_num( &tstrP, 2 );

    return ( tmclock( &tmblock ) );
}
/*

*//* time_to_fullstr( timeval )

	Produce a fullstr-format string from a time value

Accepts :

	timeval		Time to convert

Returns :

	< value >	Ptr to static string containing yymmdd.hhmmss

*/

char *
time_to_fullstr ARGLIST( ( timeval ) )
    FARG( time_t	timeval )	/* The time value */
{
	struct tm	*tmP;
static	char		strbuf[20];

    tmP = localtime( &timeval );
    sprintf( &strbuf[0], "%04d%02d%02d.%02d%02d%02d",
		   tmP->tm_year + 1900, tmP->tm_mon +1, tmP->tm_mday,
		   tmP->tm_hour, tmP->tm_min, tmP->tm_sec );

    return ( &strbuf[0] );
}
/*

*//* time_to_yyyymmdd( timeval )

	Convert a time to an yyyymmdd string

Accepts :

	timeval		The time value

Returns :

	< value >	Ptr to a static yymmdd string

*/

char *
time_to_yyyymmdd ARGLIST( ( timeval ) )
    FARG( time_t	timeval )	/* Time value */
{
	struct tm	*tmP;
static	char		strbuf[17];

    tmP = localtime( &timeval );
    sprintf( &strbuf[0], "%04d%02d%02d",
		   tmP->tm_year + 1900, tmP->tm_mon +1, tmP->tm_mday );

    return ( &strbuf[0] );
}
/*

*//* time_to_hhmmss( timeval )

	Convert a time to an hhmmss string

Accepts :

	timeval		The time value

Returns :

	< value >	Ptr to a static hhmmss string

*/

char *
time_to_hhmmss ARGLIST( ( timeval ) )
    FARG( time_t		timeval )	/* Time value */
{
	long		ltime;
	int		lhh, lmm, lss;
static	char		timebuf[15];

    ltime = timeval;
    lhh = ltime/3600;
    ltime = ltime%3600;
    lmm = ltime/60;
    ltime = ltime%60;
    lss = ltime;

    sprintf(  &timebuf[0], "%02d%02d%02d", lhh, lmm, lss );

    return ( &timebuf[0] );
}



/***********************************************************************
 *                                                                     *
 *                     Internal support routines                       *
 *                                                                     *
 ***********************************************************************/


/*

*//* get_sub_num( strPP, digitC )

	Get a number from an initial substring

Accepts :

	strPP		Ptr to string pointer
	digitC		# of digits to use

Returns :

	< value >	The number
	*strP		Updated

Notes :

	This routine will not step past a non-digit.

*/

static int
get_sub_num ARGLIST( ( strPP, digitC ) )
    NFARG( char		**strPP )	/* String pointer pointer */
     FARG( int		digitC )	/* Number of digits */
{
	int		ch;
	int		val;
	char		*strP;

    val = 0;
    strP = *strPP;

    for ( ; digitC > 0; --digitC, ++strP ) {
	ch = *strP;
	if ( ( ch < '0' ) || ( ch > '9' ) )
	    break;
	val = val * 10 + ( ch - '0' );
    }

    *strPP = strP;
    return ( val );
}
