/* mml-str.c -- string-related functions for "mml" suite

	Copyright 2003,2004,2005,2006,2007
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Contains routines strictly related to strings: storage, copying, comparison,
stuff like that.

*/

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <string.h>
#include <ctype.h>

#include <mml/mml.h>

#include <mml/mml-err.h>
#include <mml/mml-ll.h>
#include <mml/mml-str.h>
#include <mml/mml-alloc.h>
#include <mml/mml-fmt.h>

/* Local definitions */



/* External data referenced */



/* External routines used */



/* Local data publicly available */



/* Local routines and forward references */

static	BOOL	bstr_freev PROTO( ( BSTR *bsP ) );
static	BOOL	bstr_newv PROTO( ( char *noteP, int size, int max,
					BSTR *bsP ) );
static	BOOL	estr_match_wild_sub PROTO( (ESTR *patP, int pX,
					    ESTR *esP, int sX,
					    int level,
					    UWORD mflags) );
static	ESTR	*estr_tmp PROTO( (void) );
static	BOOL	match_keep PROTO( ( int partN, ESTR *esP, int sX, int len ) );
static	REFSTR	*refstr_add PROTO( (REFSTR_BASE *brsP, char *noteP) );
static	void	refstr_adjust PROTO( (REFSTR *bsP, int aaX, int delta) );
static	REFSTR	*refstr_new_common PROTO( (REFSTR_BASE *brsP, char *noteP) );

/* Private data */

static	BSTR		*Bstrmatch[MATCHKEEP]; /* Match results */

static	ESTR		*Estr_poolP;	/* List of free ESTRs */
#ifdef	MML_PTHREADS
static	pthread_mutex_t	Estr_pool_lock = PTHREAD_MUTEX_INITIALIZER;
#endif	/* MML_PTHREADS */

static	STRMATTYPE	Lastmatch = SM_NONE;  /* Last match type */
static	int		LastmatchC;	/* # of matches kept */
static	regmatch_t	Regmatch[MATCHKEEP]; /* Regex results */


    /* Quote character table.  Each index/entry corresponds to a byte value;
       the entry contains a bitmask (QF_XXX flags) indicating what context
       the byte needs to be quoted for, and an alternate string value to
       use in quoting the character.

       For any character that might be quoted, and for which there is no
       alternate text string, there may also be a QS_xxx style flag in
       the mask word that specifies how to quote the character.  The default
       is simple backslash quoting (backslash plus the input char).

       This is the way we waste lots of space just so we can do table
       lookups..

    */
static struct {
    UWORD	qct_mask;		/* QF_xxx masks */
    char	*qct_altP;		/* Alt string for quoting */
} QCtbl[256] = {
  { QS_QOCTAL|QF_DQUOTE|QF_REGEX|QF_WILD,	/* 0x00   (0)  */
	NULL },

  { 0,						/* 0x01   (1)  */
	NULL },

  { 0,						/* 0x02   (2)  */
	NULL },

  { 0,						/* 0x03   (3)  */
	NULL },

  { 0,						/* 0x04   (4)  */
	NULL },

  { 0,						/* 0x05   (5)  */
	NULL },

  { 0,						/* 0x06   (6)  */
	NULL },

  { 0,						/* 0x07   (7)  */
	NULL },

  { 0,						/* 0x08   (8)  */
	NULL },

  { 0,						/* 0x09   (9)  */
	NULL },

  { 0,						/* 0x0a  (10)  */
	"\\n" },

  { 0,						/* 0x0b  (11)  */
	NULL },

  { 0,						/* 0x0c  (12)  */
	NULL },

  { 0|QF_DQUOTE,				/* 0x0d  (13)  */
	"\\r" },

  { 0,						/* 0x0e  (14)  */
	NULL },

  { 0,						/* 0x0f  (15)  */
	NULL },

  { 0,						/* 0x10  (16)  */
	NULL },

  { 0,						/* 0x11  (17)  */
	NULL },

  { 0,						/* 0x12  (18)  */
	NULL },

  { 0,						/* 0x13  (19)  */
	NULL },

  { 0,						/* 0x14  (20)  */
	NULL },

  { 0,						/* 0x15  (21)  */
	NULL },

  { 0,						/* 0x16  (22)  */
	NULL },

  { 0,						/* 0x17  (23)  */
	NULL },

  { 0,						/* 0x18  (24)  */
	NULL },

  { 0,						/* 0x19  (25)  */
	NULL },

  { 0,						/* 0x1a  (26)  */
	NULL },

  { 0,						/* 0x1b  (27)  */
	NULL },

  { 0,						/* 0x1c  (28)  */
	NULL },

  { 0,						/* 0x1d  (29)  */
	NULL },

  { 0,						/* 0x1e  (30)  */
	NULL },

  { 0,						/* 0x1f  (31)  */
	NULL },

  { 0,						/* 0x20  (32)  ' '  */
	NULL },

  { 0,						/* 0x21  (33)  '!'  */
	NULL },

  { 0|QF_DQUOTE,				/* 0x22  (34)  '"'  */
	NULL },

  { 0,						/* 0x23  (35)  '#'  */
	NULL },

  { 0,						/* 0x24  (36)  '$'  */
	NULL },

  { 0,						/* 0x25  (37)  '%'  */
	NULL },

  { 0,						/* 0x26  (38)  '&'  */
	NULL },

  { 0,						/* 0x27  (39)  '''  */
	NULL },

  { 0,						/* 0x28  (40)  '('  */
	NULL },

  { 0,						/* 0x29  (41)  ')'  */
	NULL },

  { 0|QF_REGEX|QF_WILD,				/* 0x2a  (42)  '*'  */
	NULL },

  { 0|QF_REGEX,					/* 0x2b  (43)  '+'  */
	NULL },

  { 0,						/* 0x2c  (44)  ','  */
	NULL },

  { 0,						/* 0x2d  (45)  '-'  */
	NULL },

  { 0|QF_REGEX,					/* 0x2e  (46)  '.'  */
	NULL },

  { 0,						/* 0x2f  (47)  '/'  */
	NULL },

  { 0,						/* 0x30  (48)  '0'  */
	NULL },

  { 0,						/* 0x31  (49)  '1'  */
	NULL },

  { 0,						/* 0x32  (50)  '2'  */
	NULL },

  { 0,						/* 0x33  (51)  '3'  */
	NULL },

  { 0,						/* 0x34  (52)  '4'  */
	NULL },

  { 0,						/* 0x35  (53)  '5'  */
	NULL },

  { 0,						/* 0x36  (54)  '6'  */
	NULL },

  { 0,						/* 0x37  (55)  '7'  */
	NULL },

  { 0,						/* 0x38  (56)  '8'  */
	NULL },

  { 0,						/* 0x39  (57)  '9'  */
	NULL },

  { 0,						/* 0x3a  (58)  ':'  */
	NULL },

  { 0,						/* 0x3b  (59)  ';'  */
	NULL },

  { 0,						/* 0x3c  (60)  '<'  */
	NULL },

  { 0,						/* 0x3d  (61)  '='  */
	NULL },

  { 0,						/* 0x3e  (62)  '>'  */
	NULL },

  { 0|QF_WILD,					/* 0x3f  (63)  '?'  */
	NULL },

  { 0,						/* 0x40  (64)  '@'  */
	NULL },

  { 0,						/* 0x41  (65)  'A'  */
	NULL },

  { 0,						/* 0x42  (66)  'B'  */
	NULL },

  { 0,						/* 0x43  (67)  'C'  */
	NULL },

  { 0,						/* 0x44  (68)  'D'  */
	NULL },

  { 0,						/* 0x45  (69)  'E'  */
	NULL },

  { 0,						/* 0x46  (70)  'F'  */
	NULL },

  { 0,						/* 0x47  (71)  'G'  */
	NULL },

  { 0,						/* 0x48  (72)  'H'  */
	NULL },

  { 0,						/* 0x49  (73)  'I'  */
	NULL },

  { 0,						/* 0x4a  (74)  'J'  */
	NULL },

  { 0,						/* 0x4b  (75)  'K'  */
	NULL },

  { 0,						/* 0x4c  (76)  'L'  */
	NULL },

  { 0,						/* 0x4d  (77)  'M'  */
	NULL },

  { 0,						/* 0x4e  (78)  'N'  */
	NULL },

  { 0,						/* 0x4f  (79)  'O'  */
	NULL },

  { 0,						/* 0x50  (80)  'P'  */
	NULL },

  { 0,						/* 0x51  (81)  'Q'  */
	NULL },

  { 0,						/* 0x52  (82)  'R'  */
	NULL },

  { 0,						/* 0x53  (83)  'S'  */
	NULL },

  { 0,						/* 0x54  (84)  'T'  */
	NULL },

  { 0,						/* 0x55  (85)  'U'  */
	NULL },

  { 0,						/* 0x56  (86)  'V'  */
	NULL },

  { 0,						/* 0x57  (87)  'W'  */
	NULL },

  { 0,						/* 0x58  (88)  'X'  */
	NULL },

  { 0,						/* 0x59  (89)  'Y'  */
	NULL },

  { 0,						/* 0x5a  (90)  'Z'  */
	NULL },

  { 0|QF_REGEX,					/* 0x5b  (91)  '['  */
	NULL },

  { 0|QF_REGEX|QF_WILD,				/* 0x5c  (92)  '\'  */
	NULL },

  { 0,						/* 0x5d  (93)  ']'  */
	NULL },

  { 0,						/* 0x5e  (94)  '^'  */
	NULL },

  { 0,						/* 0x5f  (95)  '_'  */
	NULL },

  { 0,						/* 0x60  (96)  '`'  */
	NULL },

  { 0,						/* 0x61  (97)  'a'  */
	NULL },

  { 0,						/* 0x62  (98)  'b'  */
	NULL },

  { 0,						/* 0x63  (99)  'c'  */
	NULL },

  { 0,						/* 0x64 (100)  'd'  */
	NULL },

  { 0,						/* 0x65 (101)  'e'  */
	NULL },

  { 0,						/* 0x66 (102)  'f'  */
	NULL },

  { 0,						/* 0x67 (103)  'g'  */
	NULL },

  { 0,						/* 0x68 (104)  'h'  */
	NULL },

  { 0,						/* 0x69 (105)  'i'  */
	NULL },

  { 0,						/* 0x6a (106)  'j'  */
	NULL },

  { 0,						/* 0x6b (107)  'k'  */
	NULL },

  { 0,						/* 0x6c (108)  'l'  */
	NULL },

  { 0,						/* 0x6d (109)  'm'  */
	NULL },

  { 0,						/* 0x6e (110)  'n'  */
	NULL },

  { 0,						/* 0x6f (111)  'o'  */
	NULL },

  { 0,						/* 0x70 (112)  'p'  */
	NULL },

  { 0,						/* 0x71 (113)  'q'  */
	NULL },

  { 0,						/* 0x72 (114)  'r'  */
	NULL },

  { 0,						/* 0x73 (115)  's'  */
	NULL },

  { 0,						/* 0x74 (116)  't'  */
	NULL },

  { 0,						/* 0x75 (117)  'u'  */
	NULL },

  { 0,						/* 0x76 (118)  'v'  */
	NULL },

  { 0,						/* 0x77 (119)  'w'  */
	NULL },

  { 0,						/* 0x78 (120)  'x'  */
	NULL },

  { 0,						/* 0x79 (121)  'y'  */
	NULL },

  { 0,						/* 0x7a (122)  'z'  */
	NULL },

  { 0,						/* 0x7b (123)  '{'  */
	NULL },

  { 0,						/* 0x7c (124)  '|'  */
	NULL },

  { 0,						/* 0x7d (125)  '}'  */
	NULL },

  { 0,						/* 0x7e (126)  '~'  */
	NULL },

  { 0,						/* 0x7f (127)  DEL  */
	NULL },

  { 0,						/* 0x80 (128)  */
	NULL },

  { 0,						/* 0x81 (129)  */
	NULL },

  { 0,						/* 0x82 (130)  */
	NULL },

  { 0,						/* 0x83 (131)  */
	NULL },

  { 0,						/* 0x84 (132)  */
	NULL },

  { 0,						/* 0x85 (133)  */
	NULL },

  { 0,						/* 0x86 (134)  */
	NULL },

  { 0,						/* 0x87 (135)  */
	NULL },

  { 0,						/* 0x88 (136)  */
	NULL },

  { 0,						/* 0x89 (137)  */
	NULL },

  { 0,						/* 0x8a (138)  */
	NULL },

  { 0,						/* 0x8b (139)  */
	NULL },

  { 0,						/* 0x8c (140)  */
	NULL },

  { 0,						/* 0x8d (141)  */
	NULL },

  { 0,						/* 0x8e (142)  */
	NULL },

  { 0,						/* 0x8f (143)  */
	NULL },

  { 0,						/* 0x90 (144)  */
	NULL },

  { 0,						/* 0x91 (145)  */
	NULL },

  { 0,						/* 0x92 (146)  */
	NULL },

  { 0,						/* 0x93 (147)  */
	NULL },

  { 0,						/* 0x94 (148)  */
	NULL },

  { 0,						/* 0x95 (149)  */
	NULL },

  { 0,						/* 0x96 (150)  */
	NULL },

  { 0,						/* 0x97 (151)  */
	NULL },

  { 0,						/* 0x98 (152)  */
	NULL },

  { 0,						/* 0x99 (153)  */
	NULL },

  { 0,						/* 0x9a (154)  */
	NULL },

  { 0,						/* 0x9b (155)  */
	NULL },

  { 0,						/* 0x9c (156)  */
	NULL },

  { 0,						/* 0x9d (157)  */
	NULL },

  { 0,						/* 0x9e (158)  */
	NULL },

  { 0,						/* 0x9f (159)  */
	NULL },

  { 0,						/* 0xa0 (160)  */
	NULL },

  { 0,						/* 0xa1 (161)  */
	NULL },

  { 0,						/* 0xa2 (162)  */
	NULL },

  { 0,						/* 0xa3 (163)  */
	NULL },

  { 0,						/* 0xa4 (164)  */
	NULL },

  { 0,						/* 0xa5 (165)  */
	NULL },

  { 0,						/* 0xa6 (166)  */
	NULL },

  { 0,						/* 0xa7 (167)  */
	NULL },

  { 0,						/* 0xa8 (168)  */
	NULL },

  { 0,						/* 0xa9 (169)  */
	NULL },

  { 0,						/* 0xaa (170)  */
	NULL },

  { 0,						/* 0xab (171)  */
	NULL },

  { 0,						/* 0xac (172)  */
	NULL },

  { 0,						/* 0xad (173)  */
	NULL },

  { 0,						/* 0xae (174)  */
	NULL },

  { 0,						/* 0xaf (175)  */
	NULL },

  { 0,						/* 0xb0 (176)  */
	NULL },

  { 0,						/* 0xb1 (177)  */
	NULL },

  { 0,						/* 0xb2 (178)  */
	NULL },

  { 0,						/* 0xb3 (179)  */
	NULL },

  { 0,						/* 0xb4 (180)  */
	NULL },

  { 0,						/* 0xb5 (181)  */
	NULL },

  { 0,						/* 0xb6 (182)  */
	NULL },

  { 0,						/* 0xb7 (183)  */
	NULL },

  { 0,						/* 0xb8 (184)  */
	NULL },

  { 0,						/* 0xb9 (185)  */
	NULL },

  { 0,						/* 0xba (186)  */
	NULL },

  { 0,						/* 0xbb (187)  */
	NULL },

  { 0,						/* 0xbc (188)  */
	NULL },

  { 0,						/* 0xbd (189)  */
	NULL },

  { 0,						/* 0xbe (190)  */
	NULL },

  { 0,						/* 0xbf (191)  */
	NULL },

  { 0,						/* 0xc0 (192)  */
	NULL },

  { 0,						/* 0xc1 (193)  */
	NULL },

  { 0,						/* 0xc2 (194)  */
	NULL },

  { 0,						/* 0xc3 (195)  */
	NULL },

  { 0,						/* 0xc4 (196)  */
	NULL },

  { 0,						/* 0xc5 (197)  */
	NULL },

  { 0,						/* 0xc6 (198)  */
	NULL },

  { 0,						/* 0xc7 (199)  */
	NULL },

  { 0,						/* 0xc8 (200)  */
	NULL },

  { 0,						/* 0xc9 (201)  */
	NULL },

  { 0,						/* 0xca (202)  */
	NULL },

  { 0,						/* 0xcb (203)  */
	NULL },

  { 0,						/* 0xcc (204)  */
	NULL },

  { 0,						/* 0xcd (205)  */
	NULL },

  { 0,						/* 0xce (206)  */
	NULL },

  { 0,						/* 0xcf (207)  */
	NULL },

  { 0,						/* 0xd0 (208)  */
	NULL },

  { 0,						/* 0xd1 (209)  */
	NULL },

  { 0,						/* 0xd2 (210)  */
	NULL },

  { 0,						/* 0xd3 (211)  */
	NULL },

  { 0,						/* 0xd4 (212)  */
	NULL },

  { 0,						/* 0xd5 (213)  */
	NULL },

  { 0,						/* 0xd6 (214)  */
	NULL },

  { 0,						/* 0xd7 (215)  */
	NULL },

  { 0,						/* 0xd8 (216)  */
	NULL },

  { 0,						/* 0xd9 (217)  */
	NULL },

  { 0,						/* 0xda (218)  */
	NULL },

  { 0,						/* 0xdb (219)  */
	NULL },

  { 0,						/* 0xdc (220)  */
	NULL },

  { 0,						/* 0xdd (221)  */
	NULL },

  { 0,						/* 0xde (222)  */
	NULL },

  { 0,						/* 0xdf (223)  */
	NULL },

  { 0,						/* 0xe0 (224)  */
	NULL },

  { 0,						/* 0xe1 (225)  */
	NULL },

  { 0,						/* 0xe2 (226)  */
	NULL },

  { 0,						/* 0xe3 (227)  */
	NULL },

  { 0,						/* 0xe4 (228)  */
	NULL },

  { 0,						/* 0xe5 (229)  */
	NULL },

  { 0,						/* 0xe6 (230)  */
	NULL },

  { 0,						/* 0xe7 (231)  */
	NULL },

  { 0,						/* 0xe8 (232)  */
	NULL },

  { 0,						/* 0xe9 (233)  */
	NULL },

  { 0,						/* 0xea (234)  */
	NULL },

  { 0,						/* 0xeb (235)  */
	NULL },

  { 0,						/* 0xec (236)  */
	NULL },

  { 0,						/* 0xed (237)  */
	NULL },

  { 0,						/* 0xee (238)  */
	NULL },

  { 0,						/* 0xef (239)  */
	NULL },

  { 0,						/* 0xf0 (240)  */
	NULL },

  { 0,						/* 0xf1 (241)  */
	NULL },

  { 0,						/* 0xf2 (242)  */
	NULL },

  { 0,						/* 0xf3 (243)  */
	NULL },

  { 0,						/* 0xf4 (244)  */
	NULL },

  { 0,						/* 0xf5 (245)  */
	NULL },

  { 0,						/* 0xf6 (246)  */
	NULL },

  { 0,						/* 0xf7 (247)  */
	NULL },

  { 0,						/* 0xf8 (248)  */
	NULL },

  { 0,						/* 0xf9 (249)  */
	NULL },

  { 0,						/* 0xfa (250)  */
	NULL },

  { 0,						/* 0xfb (251)  */
	NULL },

  { 0,						/* 0xfc (252)  */
	NULL },

  { 0,						/* 0xfd (253)  */
	NULL },

  { 0,						/* 0xfe (254)  */
	NULL },

  { 0,						/* 0xff (255)  */
	NULL }

};




/*

*//* bstr_cat( bsP, strP )

	Concatenate to a BSTR from a NUL-terminated character string

Accepts :

	bsP		Pointer to a byte string object
	strP		Ptr to NUL-terminated string

Returns :

	<value>		TRUE if OK
			FALSE if not (allocation error)


Notes :


*/

BOOL
bstr_cat ARGLIST( (bsP, strP) )
   NFARG( BSTR		*bsP )		/* Ptr to byte string object */
    FARG( char		*strP )		/* Ptr to string */
{
    /* Easy enough... */
    while ( *strP != NUL )
	if ( ! bstr_putb( bsP, *strP++ ) )
	    return ( FALSE );

    return ( TRUE );
}
/*

*//* bstr_catb( bsP, inbsP, startX );

	Concatenate to a bstr from a bstr

Accepts :

	bsP		Pointer to a byte string object
	inbsP		Ptr to the input byte string object
	startX		Offset in the source of where to start

Returns :

	<value>		TRUE if OK
			FALSE if not (allocation error)


Notes :

*/

BOOL
bstr_catb ARGLIST( (bsP, inbsP, startX) )
   NFARG( BSTR		*bsP )		/* Ptr to byte string object */
   NFARG( BSTR		*inbsP )	/* Ptr to input byte string object */
    FARG( int		startX )	/* Offset to first input character */
{
    /* Just hand off to bstr_catn */
    return( bstr_catn( bsP,
		       &(inbsP->bs_bP[ startX ]),
		       inbsP->bs_bC - startX ) );
}
/*

*//* bstr_cate( bsP, esP, startX );

	Concatenate to a bstr from an ESTR

Accepts :

	bsP		Pointer to a byte string object
	esP		Ptr to the ESP to copy from
	startX		Offset in the source of where to start

Returns :

	<value>		TRUE if OK
			FALSE if not (allocation error)


Notes :

*/

BOOL
bstr_cate ARGLIST( (bsP, esP, startX) )
   NFARG( BSTR		*bsP )		/* Ptr to byte string object */
   NFARG( ESTR		*esP )		/* Ptr to the input ESTR */
    FARG( int		startX )	/* Offset to first input character */
{
    /* Just hand off to bstr_catn */
    return( bstr_catn( bsP,
		       &(esP->es_strP[ startX ]),
		       estr_len( esP ) - startX ) );
}
/*

*//* bstr_catn( bsP, strP, cC )

	Concatenate to a bstr from a counted byte string

Accepts :

	bsP		Pointer to a byte string object
	strP		Ptr to source byte string
	cC		Character count

Returns :

	<value>		TRUE if OK
			FALSE if not (allocation error)


Notes :


*/

BOOL
bstr_catn ARGLIST( (bsP, strP, cC) )
   NFARG( BSTR		*bsP )		/* Ptr to byte string object */
   NFARG( UBYTE		*strP )		/* Ptr to string */
    FARG( int		cC )		/* Character count */
{
    while ( cC-- > 0 )
	if ( ! bstr_putb( bsP, *strP++ ) )
	    return ( FALSE );

    return ( TRUE );
}
/*

*//* bstr_dup( bsP, extra )

	Duplicates a bstr

Accepts :

	bsP		Ptr to BSTR to copy
	extra		Number of extra bytes to allocate in buffer

Returns :

	<value>		Ptr to new BSTR if successful
			FALSE (or error exit) if not

Notes :

	The new buffer allocated is the size needed to contain the data in
	the source (not the source buffer size, just the stored data) plus
	the number of extra bytes specified in the function call.  Other
	elements of the original are copied.

	This function does NOT add a NUL terminator, on the theory
	that the caller will do that if desired.  However, this function
	does copy one extra byte (e.g. any terminator) from the original;
	so if the original was nul-terminated, the new one will be
	too.

*/

BSTR *
bstr_dup ARGLIST( (bsP, extra) )
   NFARG( BSTR		*bsP )		/* Ptr to BSTR to duplicate */
    FARG( int		extra )		/* Number of extra bytes */
{
	int		newlen;
	BSTR		*newP;		/* Ptr to the new one */

    /* Create new object */
    newlen = bsP->bs_bC + extra;
    newP = bstr_new( "bstr_dup", newlen,
        newlen > bsP->bs_bM ? newlen : bsP->bs_bM );
    if ( newP == NULL )
	return ( NULL );

    /* Duplicate the relevant structure elements */
    newP->bs_bC = bsP->bs_bC;
    newP->bs_bX = bsP->bs_bX;

    /* Copy the buffer plus any extra termination byte */
    memcpy( newP->bs_bP, bsP->bs_bP, newP->bs_bC + 1 );

    return ( newP );
}
/*

*//* bstr_free( bsP )

	Delete a bstr

Accepts :

	bsP		Pointer to byte string object

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :


*/

BOOL
bstr_free ARGLIST( (bsP) )
    FARG( BSTR		*bsP )		/* Ptr to byte string object */
{
    /* Use common code to free the data values */
    (void)bstr_freev( bsP );

    /* Release the thing */
    dealloc( bsP );

    return ( TRUE );
}
/*

*//* bstr_getb( bsP )

	Get next byte from BSTR

Accepts :

	bsP		Pointer to byte string object

Returns :

	<value>		value of next byte if not at end
			EOF it at end.

Notes :

	Uses the bs_bX as the fetch point.

*/

int
bstr_getb ARGLIST( (bsP) )
    FARG( BSTR		*bsP )		/* Ptr to byte string object */
{
    if ( bsP->bs_bX >= bsP->bs_bC )
	return ( EOF );

    return ( bsP->bs_bP[ bsP->bs_bX++ ] );
}
/*

*//* bstr_grow( bsP, chunksize )

	Grow a string object by a chunk

Accepts :

	bsP		Pointer to byte string object
	chunksize	Override of size to expand buffer by.
			 If 0, either the stored chunk value for
			 this string or a default will be used.

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :


*/

BOOL
bstr_grow ARGLIST( (bsP, chunksize) )
   NFARG( BSTR		*bsP )		/* Ptr to byte string object */
    FARG( int		chunksize )	/* Size to grow by */
{
	int		newlen;		/* New string length */

    /* Allocate new string, including slop */
    newlen = chunksize;
    if ( newlen <= 0 )
	newlen = bsP->bs_chunk;
    if ( newlen <= 0 )
	newlen = BSTR_CHUNK;
    newlen += bsP->bs_bL;
    bsP->bs_bP = (UBYTE *)erealloc( "bstr_grow", "new byte string",
			     bsP->bs_bP, newlen + BSTR_SLOP );

    if ( bsP->bs_bP == NULL ) {
	bsP->bs_bL = 0;
	return ( FALSE );
    }

    bsP->bs_bL = newlen;
    return ( TRUE );
}
/*

*//* bstr_new( noteP, size, max )

	Create a new byte string object

Accepts :

	noteP		A descriptive note for this new object
	size		Initial string size to allocate.
			0 indicates a default value.
	max		The maximum byte length for this object.
			0 indicates a default value.

Returns :

	<value>		The pointer to the new byte string object.

Notes :


*/

BSTR *
bstr_new ARGLIST( (noteP, size, max) )
   NFARG( char		*noteP )	/* Descriptive note */
   NFARG( int		size )		/* Initial buffer size */
    FARG( int		max )		/* Maximum string length, if any */
{
	BSTR		*bsP;		/* Byte string object */

    /* Make the new object */
    bsP = (BSTR *)emalloc( "bstr_new", noteP, sizeof( BSTR ) );
    if ( bsP == NULL )
	return ( NULL );

    /* Use common code to fill in the info */
    if ( ! bstr_newv( noteP, size, max, bsP ) ) {
	dealloc( bsP );
	return ( NULL );
    }

    return ( bsP );
}
/*

*//* bstr_nulterm( bstrP )

	NUL-terminate a byte string

Accepts :

	bstrP		Ptr to byte string object

Returns :

	<nothing>

Notes :

	nulterm is a special case: it writes a NUL byte
	without counting it in the character count.  By definition
	there is always room for it.

*/

void
bstr_nulterm ARGLIST( (bstrP) )
    FARG( BSTR		*bstrP )	/* Ptr to byte string object */
{
    bstrP->bs_bP[ bstrP->bs_bC ] = NUL;    
}
/*

*//* bstr_printf( bsP, fmtP, args... )

	Perform a printf into a BSTR

Accepts :

	bsP		Ptr to BSTR
	fmtP		Format string for text return
	args...		Any arguments to the format string

Returns :

	<value>		TRUE if OK
			FALSE if not (overflow that couldn't be covered)

Notes :

	Calls bstr_vprintf, q.v.

*/

BOOL
bstr_printf ARGLIST( (bsP, fmtP, va_decl) )
   NFARG( BSTR		*bsP )		/* BSTR to print into */
   NFARG( char		*fmtP )		/* Format string */
    FARG( ... )				/* stdc variable arguments */
{
	BOOL		okF;
	va_list		ap;		/* Varargs arg ptr */

    /* Start varargs */
    va_start( ap, fmtP );

    /* Hand off to bstr_vprintf which uses the stdargs/varargs handle */
    okF = bstr_vprintf( bsP, fmtP, ap );

    va_end( ap );

    return ( okF );
}
/*

*//* bstr_putb( bsP, bval )

	Add a byte to a byte string

Accepts :

	bsP		Pointer to a byte string object
	bval		Byte value to append to string

Returns :

	<value>		TRUE if OK
			FALSE if not (allocation error)


Notes :

	Might be implemented as a macro

*/

#ifndef	bstr_putb
BOOL
bstr_putb ARGLIST( (bsP, bval) )
   NFARG( BSTR		*bsP )		/* Ptr to byte string object */
    FARG( UBYTE		bval )		/* Byte value to add */
{
    /* Only process if we haven't yet reached the max
	( remember an overflow condition is one byte greater than
	   the defined maximum)
    */
    if ( bsP->bs_bC <= bsP->bs_bM ) {
	/* If we need to expand the buffer, do so here */
	if ( bsP->bs_bC == bsP->bs_bL )
	    if ( !bstr_grow( bsP, 0 ) )
		return ( FALSE );

	/* OK to add byte */
	bsP->bs_bP[ bsP->bs_bC++ ] = bval;
    }

    return ( TRUE );
}
#endif	/* bstr_putb */

/*

*//* bstr_quote_str( bsP, strP, qmask )

	Copy from a nul-terminated string into a bstr, quoting special chars

Accepts :

	bsP		Ptr to byte string object
	strP		Ptr to source string
	qmask		Bitmap expressing which kinds of characters to quote

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	Just calls bstr_quote_str_n, q.v.

*/

BOOL
bstr_quote_str ARGLIST( (bsP, strP, qmask) )
   NFARG( BSTR		*bsP )		/* Ptr to byte string object */
   NFARG( UBYTE		*strP )		/* Ptr to source string */
    FARG( UWORD		qmask )		/* Bitmap for quoting */
{
    /* Use bstr_quote_str_n */
    return ( bstr_quote_str_n( bsP, strP, -1, qmask ) );
}
/*

*//* bstr_quote_str_n( bsP, strP, len, qmask )

	Copy from a counted string into a bstr, quoting special chars

Accepts :

	bsP		Ptr to byte string object
	strP		Ptr to source string
	len		The length of the source string,
			  or -1 for nul-terminated string.
	qmask		Bitmap expressing which kinds of characters to quote

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :


	"Quoting" refers to replacing a character with a specially escaped
	sequence.  Most quoting is done by prefixing the character with
	a backslash ('\'); some quoting is done with a backslash and
	other special characters.

	Caller may include a QS_xxx style bit in qmask to force a particular
	quoting style for all characters that need quoting.

	The destination BSTR is *not* nultermed by this function; it's
	assumed that the caller may want to manipulate it further.

	FALSE return is given if the target string overflows or there is
	some other error placing a character into it.

*/

BOOL
bstr_quote_str_n ARGLIST( (bsP, strP, len, qmask) )
   NFARG( BSTR		*bsP )		/* Ptr to byte string object */
   NFARG( UBYTE		*strP )		/* Ptr to source string */
   NFARG( int		len )		/* Length of source string */
    FARG( UWORD		qmask )		/* Bitmap for quoting */
{
	UBYTE		ch;		/* The character */
	UWORD		fstyle;		/* Any style forced by caller */
	UWORD		style;		/* Chosen style */

    if ( ( qmask & (QS_QOCTAL) ) != 0 ) {
	/* Some style flag is set, pick one */
	if ( ( qmask & QS_QOCTAL ) != 0 )
	    fstyle = QS_QOCTAL;
    }
    else
	fstyle = 0;

    /* Loop swapping/quoting bytes */
    for ( ; ; ) {
	/* Fetch char */
	if ( len >= 0 ) {
	    if ( --len < 0 )
		break;
	    ch = *strP++;
	}
	else if ( ( ch = *strP++ ) == NUL )
	    break;

	/* Check char */
	if ( ( QCtbl[ch].qct_mask & qmask ) != 0 ) {
	    /* Needs quoting; force a style if caller picked one */
	    if ( fstyle != 0 )
		style = fstyle;
	    else
		/* Use style from the table if any */
		style = ( QCtbl[ch].qct_mask & (QS_QOCTAL) );

	    /* Quote according to style */
	    switch ( style ) {
		case  QS_QOCTAL:	/* backslash + octal */
		    /* Might want to special-case \0 if it's possible. */
		    if ( !bstr_printf( bsP, "\\%3o", ch ) )
			return ( FALSE );
		    break;

		default:		/* Nothing specific */
		    if ( QCtbl[ch].qct_altP != NULL ) {
			/* Table has given a quoting string */
			if ( !bstr_cat( bsP, QCtbl[ch].qct_altP ) )
			    return ( FALSE );
		    }
		    else {
			/* Just do backslash + char */
			if ( !bstr_putb( bsP, '\\' ) )
			    return ( FALSE );
			if ( !bstr_putb( bsP, ch ) )
			    return ( FALSE );
		    }
		    break;
	    }
	}
	else {
	    /* No quoting, swap the char */
	    if ( !bstr_putb( bsP, ch ) )
		return ( FALSE );
	}
    }

    return ( TRUE );
}
/*

*//* bstr_replace( bsP, startX, endX, newP )

	Replace part of a BSTR content with a NUL-terminated string

Accepts :

	bsP		Ptr to BSTR
	startX		Index of first byte to replace
	endX		Index of first byte to keep (1 after end
			 of replacement area)
	newP		Ptr to NUL-terminated string

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	Simply calls bstr_replacen, so see any notes there

*/

BOOL
bstr_replace ARGLIST( (bsP, startX, endX, newP) )
   NFARG( BSTR		*bsP )		/* Ptr to byte string object */
   NFARG( int		startX )	/* Index: start of replacement area */
   NFARG( int		endX )		/* 1 after end of replacement area */
    FARG( UBYTE		*newP )		/* NUL-terminated new string */
{
    return ( bstr_replacen( bsP, startX, endX,
			         newP, strlen( (BYTE *)newP ) ) );
}
/*

*//* bstr_replacen( bsP, startX, endX, newP, newlen )

	Replace part of a BSTR content with a counted string

Accepts :

	bsP		Ptr to BSTR
	startX		Index of first byte to replace
	endX		Index of first byte to keep (1 after end
			 of replacement area)
	newP		Ptr to new string
	newlen		Number of bytes in new string

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	If startX == endX this acts like an insert.

	calls bstr_nulterm when done, i.e. the new contents
	will be nul-terminated.

*/

BOOL
bstr_replacen ARGLIST( (bsP, startX, endX, newP, newlen) )
   NFARG( BSTR		*bsP )		/* Ptr to byte string object */
   NFARG( int		startX )	/* Index: start of replacement area */
   NFARG( int		endX )		/* 1 after end of replacement area */
   NFARG( UBYTE		*newP )		/* new string */
    FARG( int		newlen )	/* length of new string */
{
	int		delta;		/* Change in content length */
	int		movelen;	/* # bytes to move */

    /* Idiot check on passed values */
    if ( ( endX < startX ) ||
	 ( startX < 0 ) ||
	 ( endX > bsP->bs_bC ) ) {
	warning( "bad startX/endX in bstr_replacen (%d/%d bs_bC=%d)",
		startX, endX, bsP->bs_bC );
	return ( FALSE );
    }

    /* Change in size of content */
    delta = newlen - ( endX - startX );

    /* If it's growing, we have to make sure the string has room. */
    if ( ( delta > 0 ) &&
         ( bsP->bs_bC + delta ) > bsP->bs_bL  )
	if ( !bstr_grow( bsP, delta ) )
	    return ( FALSE );

    if ( delta != 0 ) {
	/* Slide the upper part of the string (starting at endX) by
           delta bytes to fit to the end of the new string.
	*/
	movelen = bsP->bs_bC - endX;
	if ( movelen > 0 )
	    memmove( &bsP->bs_bP[ endX + delta ],
		     &bsP->bs_bP[ endX ],
		     movelen );

	/* String length changes */
	bsP->bs_bC += delta;

	/* Be nice and update the working index if it was in the
           move area.
	*/
	if ( bsP->bs_bX >= ( endX + delta ) )
	    bsP->bs_bX += delta;
    }

    /* New string goes at the start point. */
    memcpy( &bsP->bs_bP[ startX ], newP, newlen );

    /* Make sure new string is NUL terminated */
    bstr_nulterm( bsP );

    /* That does it. */
    return ( TRUE );
}
/*

*//* bstr_term( bstrP, bval )

	Add a terminating byte to a string

Accepts :

	bstrP		Ptr to byte string object
	bval		Value to add to the string

Returns :

	<nothing>

Notes :

	This puts the terminating byte into the byte string,
	making sure that the value is actually put into the string
	even if the string is full.  Remember the string is defined
	so that there is always room for a terminating byte even
	if there is an overflow condition.  This is probably NOT
	desirable for NUL termination, since this function
	counts the terminator in the byte count, whereas NUL
	termination probably doesn't want that.

*/

void
bstr_term ARGLIST( (bstrP, bval) )
   NFARG( BSTR		*bstrP )	/* Ptr to byte string object */
    FARG( UBYTE		bval )		/* Value to add */
{

    /* Add the byte in the normal way.  This won't add the byte
       if the buffer is in a "full" condition, so we also store
       the byte in the buffer just to make sure it gets there.
       However we don't try if we get a failure return from the
       bstr_putb operation.
    */
    if ( bstr_putb( bstrP, bval ) )
	bstrP->bs_bP[ bstrP->bs_bC ] = bval;
}
/*

*//* bstr_vprintf( bsP, fmtP, ap )

	printf into a BSTR using varargs handle

Accepts :

	bsP		Ptr to BSTR
	fmtP		Format string for text return
	ap		varargs argument pointer

Returns :

	<value>		TRUE if OK
			FALSE if not (overflow that couldn't be covered)

Notes :

	The printed string is appended to anything that is already in
	the BSTR.  Caller should clear the BSTR first if that's
	desired.

	This is a vprintf-like function that is called by any other
	function that receives a varargs argument list.

*/

BOOL
bstr_vprintf ARGLIST( (bsP, fmtP, ap) )
   NFARG( BSTR		*bsP )		/* BSTR to print into */
   NFARG( char		*fmtP )		/* Format string */
    FARG( va_list	ap )		/* varargs handle */
{
	int		i;
	BOOL		okF;
	va_list		lap;

    /* Make a copy of the args handle to work with */
    va_copy( lap, ap );

    i = vsnprintf( bsP->bs_bP + bsP->bs_bC,
		   bsP->bs_bL - bsP->bs_bC,
		   fmtP, lap );

    if ( i >= ( bsP->bs_bL - bsP->bs_bC ) ) {
	/* Need to grow the bstr */
	bstr_grow( bsP, (i - bsP->bs_bL) + bsP->bs_bC + 10 );

	/* Try again */
	va_copy( lap, ap );
	i = vsnprintf( bsP->bs_bP + bsP->bs_bC,
		       bsP->bs_bL - bsP->bs_bC,
		       fmtP, lap );
    }

    if ( i >= ( bsP->bs_bL - bsP->bs_bC ) ) {
	bsP->bs_bC = 0;
	bstr_cat( bsP, "Too much data in bstr_vprintf.  Sorry." );
	okF = FALSE;
    }
    else {
	bsP->bs_bC += i;
	okF = TRUE;
    }

    bstr_nulterm( bsP );

    return ( okF );
}
/*

*//* estr_bstr(bstrP)

	Instantiate an ESTR from a BSTR

Accepts :

	bstrP		Ptr to the BSTR

Returns :

	<value>		Ptr to the ESTR
			NULL if there's a problem

Notes :

	An estr_bstr is a snapshot of a BSTR string at the time of
	this instantiation.  It must be used and released before any
	changes are made to the underlying BSTR.

*/

ESTR *
estr_bstr ARGLIST( (bstrP) )
    FARG( BSTR		*bstrP )	/* The BSTR */
{
	ESTR		*esP;		/* The ESTR */

    if ( ( esP = estr_init( ES_BSTR ) ) != NULL ) {
#if 0  /* not really needed */
	esP->es_objP = bstrP;
#endif
	esP->es_strP = bstrP->bs_bP;
	esP->es_len = bstrP->bs_bC;
    }

    return ( esP );
}
/*

*//* estr_cmp( es1P, es2P )

	Compare two ESTR strings

Accepts :

	es1P		Ptr to first ESTR
	es2P		Ptr to second ESTR

Returns :

	<value>		Like strcmp():
			   < 0 if es1P compares less than es2P
			     0 if es1P compares greater than es2P
			   > 0 if es1P compares greater than es2P

Notes :



*/

int
estr_cmp ARGLIST( (es1P, es2P) )
   NFARG( ESTR		*es1P )		/* First string */
    FARG( ESTR		*es2P )		/* Second string */
{
	int	c1, c2;			/* Bytes to compare */
	int	e1X, e2X;		/* Byte indexes */

    for( e1X = e2X = 0; ; ) {
	c1 = estr_nextx( es1P, e1X ); /* Get the bytes */
	c2 = estr_nextx( es2P, e2X );

	/* Return correct value if not equal */
	if ( c1 < c2 )
	    return( -1 );
	if ( c1 > c2 )
	    return( 1 );

	/* Equal so far, return match if end of both strings */
	if ( c1 == EOF )
	    return( 0 );
    }
}
/*

*//* estr_ncmp( es1P, strP )

	Compare an ESTR with a counted string

Accepts :

	es1P		Ptr to ESTR
	strP		Ptr to string to ESTR with
	len		number of chars to compare

Returns :

	<value>		Like strcmp():
			   < 0 if es1P compares less than strP
			     0 if es1P compares greater than strP
			   > 0 if es1P compares greater than strP

Notes :



*/

int
estr_cmp_nstr ARGLIST( (es1P, strP, len) )
   NFARG( ESTR		*es1P )		/* First string */
   NFARG( UBYTE		*strP )		/* Second string */
    FARG( int		len )		/* Number of bytes to compare */
{
	int	c1, c2;			/* Bytes to compare */
	int	e1X;			/* Index for ESTR */

    for( e1X = 0; len-- > 0; ) {
	c1 = estr_nextx( es1P, e1X ); /* Get the bytes */
	c2 = *strP++;

	/* Return correct value if not equal */
	if ( c1 < c2 )
	    return( -1 );
	if ( c1 > c2 )
	    return( 1 );
    }

    /* Matched */
    return( 0 );
}
/*

*//* estr_cmp( es1P, strP )

	Compare an ESTR with a nul-terminated string

Accepts :

	es1P		Ptr to ESTR
	strP		Ptr to string to ESTR with

Returns :

	<value>		Like strcmp():
			   < 0 if es1P compares less than strP
			     0 if es1P compares greater than strP
			   > 0 if es1P compares greater than strP

Notes :



*/

int
estr_cmp_str ARGLIST( (es1P, strP) )
   NFARG( ESTR		*es1P )		/* First string */
    FARG( UBYTE		*strP )		/* Second string */
{
	int	c1, c2;			/* Bytes to compare */
	int	e1X;			/* Index for ESTR */

    for( e1X = 0; ; ) {
	c1 = estr_nextx( es1P, e1X ); /* Get the bytes */
	if ( ( c2 = *strP++ ) == NUL ) 
	    c2 = EOF;

	/* Return correct value if not equal */
	if ( c1 < c2 )
	    return( -1 );
	if ( c1 > c2 )
	    return( 1 );

	/* Equal so far, return match if end of both strings */
	if ( c1 == EOF )
	    return( 0 );
    }
}
/*

*//* estr_estr(estrP)

	Instantiate an ESTR from an ESTR

Accepts :

	estrP		Ptr to the source ESTR

Returns :

	<value>		Ptr to the new ESTR
			NULL if there's a problem

Notes :

	Creates a copy of an ESTR that points to the same string.
	Useful to create another working view.

*/

ESTR *
estr_estr ARGLIST( (estrP) )
    FARG( ESTR		*estrP )	/* The BSTR */
{
	ESTR		*esP;		/* The ESTR */

    if ( ( esP = estr_init( ES_BSTR ) ) != NULL ) {
#if 0  /* not really needed */
	esP->es_objP = estrP;
#endif
	esP->es_strP = estrP->es_strP;
	esP->es_len = estrP->es_len;
    }

    return ( esP );
}
/*

*//* estr_finish( esP )

	Release an ESTR (finish up with it)

Accepts :

	esP		Ptr to the ESP being released

Returns :

	<nothing>

Notes :


*/

void
estr_finish ARGLIST( (esP) )
    FARG( ESTR		*esP )		/* The ESTR */
{
    /* Freed ESTRs are kept on a free list for later re-access.
       They are linked together by their object pointer.
    */
    if ( mml_mutex_lock( &Estr_pool_lock ) != 0 )
	warning( "estr_finish: can't get Estr_pool_lock" );
    esP->es_objP = Estr_poolP;
    Estr_poolP = esP;
    if ( mml_mutex_unlock( &Estr_pool_lock ) != 0 )
	warning( "estr_finish: can't release Estr_pool_lock" );
}
/*

*//* estr_icmp( es1P, es2P )

	Compare two ESTR strings in case insensitive manner

Accepts :

	es1P		Ptr to first ESTR
	es2P		Ptr to second ESTR

Returns :

	<value>		Like stricmp():
			   < 0 if es1P compares less than es2P
			     0 if es1P compares greater than es2P
			   > 0 if es1P compares greater than es2P

Notes :



*/

int
estr_icmp ARGLIST( (es1P, es2P) )
   NFARG( ESTR		*es1P )		/* First string */
    FARG( ESTR		*es2P )		/* Second string */
{
	int	c1, c2;			/* Bytes to compare */
	int	e1X, e2X;		/* Indexes */

    for( e1X = e2X = 0; ; ) {
	c1 = estr_nextx( es1P, e1X ); /* Get the bytes */
	c2 = estr_nextx( es2P, e2X );

	if ( ( c1 >= 'A' ) && ( c1 <= 'Z' ) ) {
	    /* c1 is upper, make c2 upper */
	    if ( ( c2 >= 'a' ) && ( c2 <= 'z' ) )
		c2 -= ( 'a' - 'A' );
	}
	/* c1 is not upper, make c2 not upper */
	else if ( ( c2 >= 'A' ) && ( c2 <= 'Z' ) )
	    c2 += ( 'a' - 'A' );

	/* Return correct value if not equal */
	if ( c1 < c2 )
	    return( -1 );
	if ( c1 > c2 )
	    return( 1 );

	/* Equal so far, return match if end of both strings */
	if ( c1 == EOF )
	    return( 0 );
    }
}
/*

*//* estr_icmp_nstr( es1P, strP, len )

	Compare ESTR against a counted string in a case-insensitive manner

Accepts :

	es1P		Ptr to ESTR
	strP		Ptr to nul-terminated string to compare es1P to
	len		Number of bytes to compare

Returns :

	<value>		Like stricmp():
			   < 0 if es1P compares less than strP
			     0 if es1P compares greater than strP
			   > 0 if es1P compares greater than strP

Notes :



*/

int
estr_icmp_nstr ARGLIST( (es1P, strP, len) )
   NFARG( ESTR		*es1P )		/* First string */
   NFARG( UBYTE		*strP )		/* Second string */
    FARG( int		len )		/* Number of bytes to compare */
{
	int	c1, c2;			/* Bytes to compare */
	int	e1X;			/* es1P index */

    for( e1X = 0; len-- > 0 ; ) {
	c1 = estr_nextx( es1P, e1X ); /* Get the bytes */
	c2 = *strP++;

	if ( ( c1 >= 'A' ) && ( c1 <= 'Z' ) ) {
	    /* c1 is upper, make c2 upper */
	    if ( ( c2 >= 'a' ) && ( c2 <= 'z' ) )
		c2 -= ( 'a' - 'A' );
	}
	/* c1 is not upper, make c2 not upper */
	else if ( ( c2 >= 'A' ) && ( c2 <= 'Z' ) )
	    c2 += ( 'a' - 'A' );

	/* Return correct value if not equal */
	if ( c1 < c2 )
	    return( -1 );
	if ( c1 > c2 )
	    return( 1 );
    }

    /* Matched */
    return( 0 );
}
/*

*//* estr_icmp_str( es1P, strP )

	Compare ESTR against a string in a case-insensitive manner

Accepts :

	es1P		Ptr to ESTR
	strP		Ptr to nul-terminated string to compare es1P to

Returns :

	<value>		Like stricmp():
			   < 0 if es1P compares less than strP
			     0 if es1P compares greater than strP
			   > 0 if es1P compares greater than strP

Notes :



*/

int
estr_icmp_str ARGLIST( (es1P, strP) )
   NFARG( ESTR		*es1P )		/* First string */
    FARG( UBYTE		*strP )		/* Second string */
{
	int	c1, c2;			/* Bytes to compare */
	int	e1X;			/* es1P index */

    for( e1X = 0; ; ) {
	c1 = estr_nextx( es1P, e1X ); /* Get the bytes */
	if ( ( c2 = *strP++ ) == NUL ) 
	    c2 = EOF;

	if ( ( c1 >= 'A' ) && ( c1 <= 'Z' ) ) {
	    /* c1 is upper, make c2 upper */
	    if ( ( c2 >= 'a' ) && ( c2 <= 'z' ) )
		c2 -= ( 'a' - 'A' );
	}
	/* c1 is not upper, make c2 not upper */
	else if ( ( c2 >= 'A' ) && ( c2 <= 'Z' ) )
	    c2 += ( 'a' - 'A' );

	/* Return correct value if not equal */
	if ( c1 < c2 )
	    return( -1 );
	if ( c1 > c2 )
	    return( 1 );

	/* Equal so far, return match if end of both strings */
	if ( c1 == EOF )
	    return( 0 );
    }
}
/*

*//* estr_init( estype )

	Create an empty ESTR

Accepts :

	estype		Type of ESTR being created

Returns :

	<value>		Ptr to ESTR if successful
			NULL if not

Notes :

	We keep a free list of previously allocated ESTRs for rapid
	access.

	The string length (es_len) is initialized to -1.

*/

ESTR *
estr_init ARGLIST( (estype) )
    FARG( ESTRTYPE	estype )	/* Type to make */
{
	ESTR		*esP;		/* The ESTR */

    /* Lock the entire function since the free list may be altered */
    if ( mml_mutex_lock( &Estr_pool_lock ) != 0 )
	warning( "estr_init: can't get Estr_pool_lock" );

    /* Check the free list first. */
    if ( ( esP = Estr_poolP ) != NULL ) {
	/* Free ones are linked together via their es_objP pointer. */
	Estr_poolP = (ESTR *)esP->es_objP;
    }
    else {
	/* None available, must allocate a new one. */
	esP = (ESTR *)emalloc( "estr_init", "new ESTR", sizeof(ESTR) );
    }

    if ( esP != NULL ) {
	esP->es_type = estype;
	esP->es_objP = NULL;
	esP->es_strP = NULL;
	esP->es_len = -1;	/* Default to nul-terminated */
	esP->es_strX = 0;

	esP->es_xflags = 0;
	esP->es_xP = NULL;
    }

    if ( mml_mutex_unlock( &Estr_pool_lock ) != 0 )
	warning( "estr_init: can't release Estr_pool_lock" );

    return ( esP );
}
/*

*//* estr_match( patP, esP, mtype, subokF, mflags)

	General ESTR string match interface for various match types

Accepts :

	patP		Pattern ESTR
	esP		ESTR to test
	mtype		Match type (STRMATCHTYPE)
	mflags		Flag bits that refine the match

Returns :

	<value>		result (see notes)

Notes :


	Not all flags are valid for all match types.

	Result is whatever comes out of the specific comparison
	routine.  For the SM_CMP type, it's a comparison value
	(<0, 0, or >0.  For the SM_MAT and SM_RGX types, it's
	TRUE/FALSE.

	This is basically a clone of strmatch, only for ESTRs.

*/

int
estr_match ARGLIST( (patP, esP, mtype, mflags) )
   NFARG( ESTR		*patP )		/* Pattern (or first string) */
   NFARG( ESTR		*esP )		/* String to test */
   NFARG( STRMATTYPE	mtype )		/* Match type */
    FARG( UWORD		mflags )	/* Match flags */
{
	int		result;

    switch ( mtype ) {
	case	SM_STRCMP:
	    if ( ( mflags & MF_IGNCASE ) != 0 )
		result = estr_icmp( patP, esP );
	    else
		result = estr_cmp( patP, esP );
	    break;

	case	SM_MATCH:		/* Match, no wildcards */
	    result = estr_match_exact( NULL, patP, esP, mflags );
	    break;

	case	SM_WILD:		/* Match with glob-like wildcards */
	    /* Note: substring match not relevant here. */
	    result = estr_match_wild( NULL, patP, esP, mflags );
	    break;

	case	SM_REGEX:		/* Regular expression */
	    result = estr_match_regex( NULL, patP, esP, mflags );
	    break;

	default:
	    /* Bad... return false */
	    result = -1;
	    break;
    }

    return ( result );
}
/*

*//* estr_match_exact( cpatP, patP, esP, mflags )

	ESTR match: exact pattern match

Accepts :

	cpatP		Ptr to precompiled pattern, or NULL if none
	patP		Pattern ESTR
	esP		ESTR to test
	mflags		Match flags

Returns :

	<value>		TRUE if pattern matched
			FALSE if not.

Notes :

	mflags contains MF_xxx bits that help control the match.  Not all
	bits necessary apply.

	cpatP is for future use

*/

BOOL
estr_match_exact ARGLIST( (cpatP, patP, esP, mflags) )
   NFARG( void		*cpatP )	/* Precompiled pattern */
   NFARG( ESTR		*patP )		/* Pattern (or first string) */
   NFARG( ESTR		*esP )		/* String to test */
    FARG( UWORD		mflags )	/* Match flags */
{
	BOOL		igncaseF;
	BOOL		subokF;
	int		pX, sX;		/* Indexes into patP and esP */
	int		wpX, wsX;	/* Working indexes */
	int		pch, sch;	/* Chars */

    igncaseF = ( (mflags & MF_IGNCASE) != 0 );
    subokF = ( (mflags & MF_SUBSTR) != 0 );

    /* Initialize stuff for remembering the match. */
    Lastmatch = SM_MATCH;
    LastmatchC = 0;

    /* Loop through successive substrings of the test string */
    for ( pX = sX = 0; ; ++sX ) {
	/* Test the pattern against this substring */
	wpX = pX;
	wsX = sX;
	for ( ; ; ) {
	    pch = estr_nextx( patP, wpX );  /* Next pattern char */
	    sch = estr_nextx( esP, wsX );   /* Next string char */

	    if ( igncaseF ) {
		/* Fold case */
		if ( ( pch >= 'A' ) && ( pch <= 'Z' ) ) {
		    /* pch is upper, make sch upper */
		    if ( ( sch >= 'a' ) && ( sch <= 'z' ) )
			sch -= ( 'a' - 'A' );
		}
		/* pch is not upper, make sch not upper */
		else if ( ( sch >= 'A' ) && ( sch <= 'Z' ) )
		    sch += ( 'a' - 'A' );
	    }

	    if ( pch != sch ) {
		/* Didn't match - we're done if we ran off end of pattern */
		if ( pch == EOF ) {
		    if ( subokF )
			match_keep( 0, esP, sX, (wsX - sX) -1 );
		    return ( subokF );
		}
		    
		/* optimize for running off end of string */
		if ( sch == EOF )
		    /* End of string, can't possibly match. */
		    return ( FALSE );

		break;		/* Exit to try next substring */
	    }

	    /* Matched .. if end, we matched.  Otherwise keep looking. */
	    if ( pch == EOF ) {
		/* Remember this match */
		match_keep( 0, esP, sX, ( wsX - sX ) -1 );
		return ( TRUE );
	    }
	}

	/* Here if the match failed */
	if ( !subokF )
	    return ( FALSE );		/* Substring not allowed */

	/* Loop to advance to next substring */
    }

    /* Can't really get here */
    warning( "strmatch_exact: can't get here." );
    return ( FALSE );
}
/*

*//* estr_match_regex( cpatP, patP, esP, mflags )

	ESTR match: regular expression pattern match

Accepts :

	cpatP		Ptr to precompiled pattern, or NULL if none
	patP		Pattern ESTR
	esP		ESTR to find pattern in
	mflags		Match flags

Returns :

	<value>		TRUE if pattern matched
			FALSE if not.

Notes :

	In this context, substring match is a non-anchored match.

	This procedure interface is kinda hacked in right now, to be
	similar to the strmatch_exact and strmatch_wild calling sequence.
	At some point it should be cleaned up so that the regex
	compile is done separately from the execution.  (Thus the
	cpatP argument.)

	cpatP is for future use


*/

BOOL
estr_match_regex ARGLIST( (cpatP, patP, esP, mflags) )
   NFARG( void		*cpatP )	/* Precompiled pattern */
   NFARG( ESTR		*patP )		/* Pattern (or first string) */
   NFARG( ESTR		*esP )		/* String to test */
    FARG( UWORD		mflags )	/* Match flags */
{
	BOOL		matchF;		/* "matched" flag */
	UBYTE		ch;
	int		i;		/* Scratch */
	regex_t		regex;		/* Compiled state for regex */
	UBYTE		*bP;		/* Scratch ptr */

    /* Initialize stuff for remembering the match. */
    Lastmatch = SM_REGEX;
    LastmatchC = 0;

    /* Compile the regular expresion that's in patP,
       with some hackery to first make the string nul-terminated.
    */
    if ( patP->es_len >= 0 ) {
	bP = &patP->es_strP[ patP->es_len ];
	ch = *bP;
	*bP = NUL;
    }
    i = regcomp( &regex, (BYTE *)patP->es_strP,
		  REG_EXTENDED|
		    ( ((mflags & MF_IGNCASE) != 0) ? REG_ICASE : 0)
		);
    if ( patP->es_len >= 0 )
	*bP = ch;
    if ( i != 0 ) {
	warning( "regcomp failed." );
	matchF = FALSE;
    }
    else {
	/* Apply the pattern to the string */
	if ( esP->es_len >= 0 ) {
	    bP = &esP->es_strP[ esP->es_len ];
	    ch = *bP;
	    *bP = NUL;
	}
	i = regexec( &regex, (BYTE *)esP->es_strP,
		     MATCHKEEP, &Regmatch[0], 0 );
	if ( esP->es_len >= 0 )
	    *bP = ch;
	if ( i == 0 ) {
	    matchF = TRUE;
	}
	else {
	    matchF = FALSE;
	}

	/* Release the compiled pattern */
	regfree( &regex );
    }

    /* Record the matched substrings */
    if ( matchF ) {
	for ( i = 0; i < MATCHKEEP; ++i ) {
	    if ( Regmatch[i].rm_so == -1 )
		break;
	    else
		match_keep( i, esP, Regmatch[i].rm_so,
			       Regmatch[i].rm_eo - Regmatch[i].rm_so );
	}
    }

    return ( matchF );
}
/*

*//* estr_match_wild( cpatP, patP, esP, mflags )

	ESTR match: modified wildcard pattern match

Accepts :

	cpatP		Ptr to precompiled pattern, or NULL if none
	patP		Pattern ESTR
	esP		ESTR to find pattern in
	mflags		Flags to refine the match

Returns :

	<value>		TRUE if pattern matched
			FALSE if not.

Notes :

	Modified wildcard :
	    ? matches any single char
	    * matches zero or more chars

	cpatP is for future use

*/

BOOL
estr_match_wild ARGLIST( (cpatP, patP, esP, mflags) )
   NFARG( void		*cpatP )	/* Precompiled pattern */
   NFARG( ESTR		*patP )		/* Pattern (or first string) */
   NFARG( ESTR		*esP )		/* String to test */
    FARG( UWORD		mflags )	/* Match flags */
{
	BOOL		resultF;

    /* Initialize the match memory */
    Lastmatch = SM_MATCH;
    LastmatchC = 0;

    /* The internal recursive match routine does all the work. */
    resultF = estr_match_wild_sub( patP, 0, esP, 0, 1, mflags );

    if ( resultF ) {
	/* Remember the entire match string as submatch 0 */
	match_keep( 0, esP, 0, estr_len( esP ) );
    }

    return ( resultF );
}
/*

*//* estr_nstr(strP, len)

	Instantiate an ESTR from a counted string

Accepts :

	strP		Ptr to the base string
	len		Length of the string

Returns :

	<value>		Ptr to the ESTR
			NULL if there's a problem

Notes :

	An estr_nstr is a snapshot of a counted string at the time of
	this instantiation.  It must be used and released before any
	changes are made to the underlying string.

*/

ESTR *
estr_nstr ARGLIST( (strP, len) )
   NFARG( UBYTE		*strP )		/* Base string */
    FARG( int		len )		/* String length */
{
	ESTR		*esP;		/* The ESTR */

    if ( ( esP = estr_init( ES_COUNT ) ) != NULL ) {
	esP->es_strP = strP;
	esP->es_len = len;
    }

    return ( esP );
}
/*

*//* estr_refstr(rsP)

	Instantiate an ESTR from a REFSTR

Accepts :

	rsP		Ptr to the REFSTR

Returns :

	<value>		Ptr to the ESTR
			NULL if there's a problem

Notes :

	An estr_refstr is a snapshot of a REFSTR string at the time of
	this instantiation.  It must be used and released before any
	changes are made to the underlying REFSTR

*/

ESTR *
estr_refstr ARGLIST( (rsP) )
    FARG( REFSTR	*rsP )		/* The REFSTR */
{
	ESTR		*esP;		/* The ESTR */

    if ( ( esP = estr_init( ES_BSTR ) ) != NULL ) {
#if 0  /* not really needed */
	esP->es_objP = rsP;
#endif
	esP->es_strP = refstr_bstr( rsP )->bs_bP + refstr_startx( rsP );
	esP->es_len = refstr_len( rsP );
    }

    return ( esP );
}
/*

*//* estr_solbat( esP, eolF )

	Skip over leading blanks and tabs (whitespace) in an ESTR

Accepts :

	esP		Encapsulated string
	eolF		If end of line chars are considered whitespace

Returns :

	<nothing>

Notes :

	The given ESTR's "fetch pointer" is adjusted to have skipped
	  over the whitespace.

	Name "solbat" stolen from bam

*/

void
estr_solbat ARGLIST( (esP, eolF) )
   NFARG( ESTR		*esP )		/* The ESTR */
    FARG( BOOL		eolF )		/* Whether eol chars are whitespace */
{
	int		ch;
	int		sX;		/* Copy of string index */

    /* Skip whitespace. */
    for ( sX = esP->es_strX; ; ++sX ) {
	/* Get byte at current index.  Note that EOF will be returned
	   if at the end of string.
	*/
	ch = estr_bytex( esP, sX );
	if ( ( ch != ' ' ) &&
	     ( ch != '\t' ) &&
	     ( !eolF || ( ( ch != '\r' ) && ( ch != '\n' ) ) ) )
	    break;
    }

    esP->es_strX = sX;
}
/*

*//* estr_str(strP)

	Instantiate an ESTR from a nul-terminated string

Accepts :

	strP		Ptr to the nul-terminated string

Returns :

	<value>		Ptr to the ESTR
			NULL if there's a problem

Notes :


*/

ESTR *
estr_str ARGLIST( (strP) )
    FARG( UBYTE		*strP )		/* The source string */
{
	ESTR		*esP;		/* The ESTR */

    if ( ( esP = estr_init( ES_BSTR ) ) != NULL ) {
	esP->es_strP = strP;
	esP->es_len = -1;		/* nul-terminated string */
    }

    return ( esP );
}
/*

*//* estr_tmp_bstr(bsP)

	Instantiate a temp ESTR from a BSTR

Accepts :

	bsP		Ptr to the BSTR to take string from

Returns :

	<value>		Ptr to the ESTR

Notes :

	Uses a temporary ESTR.  See estr_tmp().

	Makes a reference to the BSTR's string, NOT a copy.

*/

ESTR *
estr_tmp_bstr ARGLIST( (bsP) )
    FARG( BSTR		*bsP )		/* The BSTR */
{
    return ( estr_tmp_nstr( bsP->bs_bP, bsP->bs_bC ) );
}
/*

*//* estr_tmp_estr_new( estrP )

	Make a copy of an ESTR from a given ESTR

Accepts :

	estrP		ESTR to make a copy of.

Returns :

	<value>		Ptr to the new ESTR

Notes :

	Uses a temporary ESTR.  See estr_tmp().

	Makes a new copy of the input string, not a reference to it.
	This new copy should be deallocated when it is no longer needed.

	The ESTR is set up as a counted string encapsulation, but the string
	is also NUL-terminated.

*/

ESTR *
estr_tmp_estr_new ARGLIST( (estrP) )
    FARG( ESTR		*estrP )	/* Source ESTR to copy */
{
	int		len;
	ESTR		*esP;		/* The ESTR */

    len = estr_len( estrP );

    esP = estr_tmp();			/* Grab a temporary ESTR */

    /* Fill in the parts that are variable. */
    esP->es_type = ES_COUNT;
    esP->es_strP = emalloc( "estr_tmp_estr_new", "new buffer", len +1 );
    if ( esP->es_strP == NULL )
	return ( NULL );
    memcpy( esP->es_strP, estrP->es_strP, len );
    esP->es_strP[len] = NUL;		/* NUL-terminate it too. */
    esP->es_len = len;
    esP->es_strX = 0;

    return ( esP );
}
/*

*//* estr_tmp_estr_tmpstr( estrP )

	Make a copy of an ESTR from a given ESTR

Accepts :

	estrP		ESTR to make a copy of.

Returns :

	<value>		Ptr to the new ESTR

Notes :

	Uses a temporary ESTR.  See estr_tmp().

	Makes a tmpstr of the input string, not a reference to it.
	The life of a tmpstr is short; caller must use this with
	all of the restrictions of a tmpstr.  The main point is that
	it doesn't have to be freed, but neither can it be kept for
	long.

	The ESTR is set up as a counted string encapsulation, but the string
	is also NUL-terminated.

*/

ESTR *
estr_tmp_estr_tmpstr ARGLIST( (estrP) )
    FARG( ESTR		*estrP )	/* Source ESTR to copy */
{
	int		len;
	ESTR		*esP;		/* The ESTR */

    len = estr_len( estrP );

    esP = estr_tmp();			/* Grab a temporary ESTR */

    /* Fill in the parts that are variable. */
    esP->es_type = ES_COUNT;
    esP->es_strP = tmpstr( len +1 );
    if ( esP->es_strP == NULL )
	return ( NULL );
    memcpy( esP->es_strP, estrP->es_strP, len );
    esP->es_strP[len] = NUL;		/* NUL-terminate it too. */
    esP->es_len = len;
    esP->es_strX = 0;

    return ( esP );
}
/*

*//* estr_tmp_nstr(strP, len)

	Instantiate a temp ESTR from a counted string

Accepts :

	strP		Ptr to the source string
	len		Number of bytes in the source string

Returns :

	<value>		Ptr to the ESTR

Notes :

	Uses a temporary ESTR.  See estr_tmp().

	Makes a reference to the input string, NOT a copy.  Use another
	 function (or do some extra stuff) if you need a copy.

*/

ESTR *
estr_tmp_nstr ARGLIST( (strP, len) )
   NFARG( UBYTE		*strP )		/* The source string */
    FARG( int		len )		/* Source string length */
{
	ESTR		*esP;		/* The ESTR */

    esP = estr_tmp();			/* Grab a temporary ESTR */

    /* Fill in the parts that are variable. */
    esP->es_type = ES_COUNT;
    esP->es_strP = strP;
    esP->es_len = len;
    esP->es_strX = 0;

    return ( esP );
}
/*

*//* estr_tmp_nstr_new(strP, len)

	Instantiate a temp ESTR from a counted string, copying the string

Accepts :

	strP		Ptr to the source string
	len		Number of bytes in the source string

Returns :

	<value>		Ptr to the ESTR

Notes :

	Uses a temporary ESTR.  See estr_tmp().

	Makes a new copy of the input string, NOT a reference.

*/

ESTR *
estr_tmp_nstr_new ARGLIST( (strP, len) )
   NFARG( UBYTE		*strP )		/* The source string */
    FARG( int		len )		/* Source string length */
{
	ESTR		*esP;		/* The ESTR */

    esP = estr_tmp();			/* Grab a temporary ESTR */

    /* Fill in the parts that are variable. */
    esP->es_type = ES_COUNT;
    esP->es_strP = emalloc( "estr_tmp_nstr_new", "new buffer", len +1 );
    if ( esP->es_strP == NULL )
	return ( NULL );
    memcpy( esP->es_strP, strP, len );
    esP->es_len = len;
    esP->es_strX = 0;

    return ( esP );
}
/*

*//* estr_tmp_refstr(rsP)

	Instantiate a temp ESTR from a REFSTR

Accepts :

	rsP		Ptr to REFSTR to take info from

Returns :

	<value>		Ptr to the ESTR

Notes :

	Uses a temporary ESTR.  See estr_tmp().

*/

ESTR *
estr_tmp_refstr ARGLIST( (rsP) )
    FARG( REFSTR	*rsP )		/* Source REFSTR */
{
	ESTR		*esP;		/* The ESTR */

    esP = estr_tmp();			/* Grab a temporary ESTR */

    /* Fill in the parts that are variable. */
    esP->es_type = ES_BSTR;

#if 0  /* not really needed */
    esP->es_objP = rsP;
#endif
    esP->es_strP = refstr_bstr( rsP )->bs_bP + refstr_startx( rsP );
    esP->es_len = refstr_len( rsP );
    esP->es_strX = 0;

    return ( esP );
}
/*

*//* estr_tmp_str(strP)

	Instantiate a temp ESTR from a nul-terminated string

Accepts :

	strP		Ptr to the nul-terminated string

Returns :

	<value>		Ptr to the ESTR

Notes :

	Uses a temporary ESTR.  See estr_tmp().

	If strP is passed as NULL, the returned ESTR is also NULL.
	This provides a convenient way to propagate a NULL value.

*/

ESTR *
estr_tmp_str ARGLIST( (strP) )
    FARG( UBYTE		*strP )		/* The source string */
{
	ESTR		*esP;		/* The ESTR */

    if ( strP == NULL )
	esP = NULL;
    else {
	esP = estr_tmp();		/* Grab a temporary ESTR */

	/* Fill in the parts that are variable. */
	esP->es_type = ES_BSTR;
	esP->es_strP = strP;
	esP->es_len = -1;
	esP->es_strX = 0;
    }

    return ( esP );
}
/*

*//* estr_token_ws( esP, bufP, bufL, eolF )

	Extract a token delimited by whitespace

Accepts :

	esP		ESTR ptr
	bufP		Where to put the token
	bufL		Length of buffer for token
	eolF		If end of line chars are considered whitespace

Returns :

	<value>		Number of bytes in source token


Notes :

	Very simple token grabber which only recognizes whitespace
	as token separators.

	The token is fetched at the current ESTR index, which is
	updated to point to the whitespace after the token.

	The returned byte count may be larger than the buffer size
	provided; this would indicate that the returned token is
	truncated.

	In any case, the returned token is always NUL-terminated.
	Thus the maximum token length returned is one less than the
	buffer length specified.

*/

int
estr_token_ws ARGLIST( (esP, bufP, bufL, eolF) )
   NFARG( ESTR		*esP )		/* The ESTR */
   NFARG( UBYTE		*bufP )		/* Where to put the token */
   NFARG( int		bufL )		/* Size of buffer */
    FARG( BOOL		eolF )		/* Whether eol chars are whitespace */
{
	int		ch;
	int		sX;
	int		cC;

    --bufL;				/* Account for trailing NUL */

    /* Skip whitespace prior to token */
    estr_solbat( esP, eolF );

    /* Grab characters until we see more whitespace */
    for ( sX = esP->es_strX, cC = 0; ; ++sX, ++cC ) {
	ch = estr_bytex( esP, sX );
	if ( ( ch == EOF ) ||
	     ( ch == ' ' ) ||
	     ( ch == '\t' ) ||
	     ( eolF && ( ch == '\r' ) || ( ch == '\n' ) ) )
	    break;

	if ( cC < bufL )
	    *bufP++ = ch;
    }

    esP->es_strX = sX;
    *bufP = NUL;

    return ( cC );
}
/*

*//* estr_to_f(esP)

	Interpret an ESTR as a floating point value

Accepts :

	esP		Ptr to an ESTR

Returns :

	<value>		the floating point value

Notes :

	The interpretation is limited to some practical string
	length.  This should not cause troubles.

*/

double
estr_to_f ARGLIST( (esP) )
    FARG( ESTR		*esP )		/* The ESTR */
{
	char		*strP;		/* A string to interpret */
	int		len;

    if ( ( len = esP->es_len ) == -1 )
	strP = (BYTE *)esP->es_strP;	/* Already nul-terminated */
    else {
	/* Limit to a practical length */
	if ( len > 20 )
	    len = 20;

	if ( ( strP = (BYTE *)tmpstr( len +1 ) ) == NULL )
	    strP = "0";
	else {
	    strncpy( strP, (BYTE *)esP->es_strP, len );
	    strP[len] = NUL;
	}
    }

    return ( atof( strP ) );
}
/*

*//* estr_to_l(esP)

	Interpret an ESTR as a long int (a la atol())

Accepts :

	esP		Ptr to an ESTR

Returns :

	<value>		the long int value

Notes :

	The interpretation is limited to some practical string
	length.  This should not cause troubles.

*/

long
estr_to_l ARGLIST( (esP) )
    FARG( ESTR		*esP )		/* The ESTR */
{
	char		*strP;		/* A string to interpret */
	int		len;

    if ( ( len = esP->es_len ) == -1 )
	strP = (BYTE *)esP->es_strP;		/* Already nul-terminated */
    else {
	/* Limit to a practical length */
	if ( len > 20 )
	    len = 20;

	if ( ( strP = (BYTE *)tmpstr( len +1 ) ) == NULL )
	    strP = "0";
	else {
	    strncpy( strP, (BYTE *)esP->es_strP, len );
	    strP[len] = NUL;
	}
    }

    return ( atod( strP ) );
}
/*

*//* isblankstr( strP )

	See if a string is blank

Accepts :

	strP		Ptr to the string

Returns :

	< value >	TRUE if blank,
			FALSE if not.

Notes :

	Blank means containing only spaces, tabs, or newlines.
*/

BOOL
isblankstr ARGLIST( ( strP ) )
    FARG( char		*strP )		/* Ptr to string */
{
	char		ch;

    while ( ( ch = *strP++ ) != NUL )
	if ( ( ch != ' ' ) && ( ch != '\t' ) && ( ch != '\n' ) )
	    return ( FALSE );
    return ( TRUE );
}
/*

*//* newstr_estr(facP, nameP, esP)

	Create a new nul-terminated string from an ESTR

Accepts :

	esP		Ptr to ESTR to copy
	nameP		Name of item being allocated
	oldP		Ptr to old string to duplicate

Returns :

	<value>		Ptr to new string if successful
			NULL if not

Notes :

	Like newstr() but with an ESTR as a source.

*/

char *
newstr_estr ARGLIST( (facP, nameP, esP) )
   NFARG( char		*facP )		/* Ptr to facility name */
   NFARG( char		*nameP )	/* Ptr to name of thing */
    FARG( ESTR		*esP )		/* Source for new string */
{
    return ( newstr_n( facP, nameP, (BYTE *)esP->es_strP, estr_len( esP ) ) );
}
/*

*//* refstr_access( rsP )

	Make an access to a REFSTR

Accepts :

	rsP		Ptr to the REFSTR

Returns :

	<value>		Pointer to the REFSTR

Notes :

	When you want multiple pointers to the same refstr, use
	refstr_access to take an access.  refstr_free() won't release
	the refstr storage until all accesses have been freed.

	Be sure to use refstr_free() to get rid of any access gained
	this way, otherwise there will be REFSTRs hanging around
	needlessly.

	Note that this could be a macro, but then where would we put
	all these comments?

*/

REFSTR *
refstr_access ARGLIST( (rsP) )
    FARG( REFSTR	*rsP )		/* REFSTR ptr */
{
    if ( rsP->rs_refC <= 0 ) {
	warning( "refstr_access: bad refC" );
	rsP = NULL;
    }
    else
	++rsP->rs_refC;

    return ( rsP );
}
/*

*//* refstr_endx( rsP )

	Translate refstr's end point to its BSTR offset

Accepts :

	rsP		Ptr to the REFSTR

Returns :

	<value>		End point's offset in the underlying BSTR

Notes :


*/

int
refstr_endx ARGLIST( (rsP) )
    FARG( REFSTR	*rsP )		/* REFSTR ptr */
{
	int		bsX;		/* Resulting offset */
	BSTR		*bsP;		/* Underlying BSTR */

    bsP = refstr_bstr( rsP );

    /* Offset of -1 is the end of the BSTR;
       any non-negative value is the simple offset.
    */
    if ( rsP->rs_endX < 0 )
	bsX = bsP->bs_bC;
    else {
	bsX = rsP->rs_endX;

	/* Make sure the offset is within the stored data */
	if ( bsX >= bsP->bs_bC )
	    bsX = bsP->bs_bC;
    }

    return ( bsX );
}
/*

*//* refstr_free(rsP)

	Delete a refstr

Accepts :

	rsP		Ptr to REFSTR to delete

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	The underlying REFSTR_BASE will be deleted if this is the
	only refstr copy.

*/

BOOL
refstr_free ARGLIST( (rsP) )
    FARG( REFSTR	*rsP )		/* Ptr to REFSTR */
{
	REFSTR_BASE	*brsP;		/* Ptr to base string */

    if ( --rsP->rs_refC == 0 ) {

	brsP = rsP->rs_baseP;

	/* Remove this reference from the linked list */
	ll_rem( rsP->rs_llP );

	/* Remove this structure */
	dealloc( rsP );

	/* OK if this isn't the final reference */
	if ( brsP->bbs_ll.llh_firstP != NULL )
	    return ( TRUE );

	/* Must also now delete the base object. */
	bstr_freev( &brsP->bbs_bstr );
	ll_finit( &brsP->bbs_ll );
	dealloc( brsP );
    }
    else if ( rsP->rs_refC < 0 ) {
	warning( "refstr_free: negative reference count." );
	return ( FALSE );
    }

    return ( TRUE );
}
/*

*//* refstr_getb(rsP)

	Fetch the next byte from a REFSTR fetch point

Accepts :

	rsP		Ptr to REFSTR

Returns :

	<value>		The next byte if there are any,
			EOF if fetch point is at end of refstr

Notes :


*/

int
refstr_getb ARGLIST( (rsP) )
    FARG( REFSTR	*rsP )		/* Ptr to REFSTR */
{
	int		bsX;		/* Access offset within bstr */
	int		rbX;		/* Normalized copy of rs_bX */
	int		bsendX;		/* End offset within bstr */
	BSTR		*bsP;		/* Base BSTR */

    /* Translate refstr offsets to underlying bstr offsets */
    rbX = rsP->rs_bX;
    if ( rbX == -1 )
	rbX = 0;

    bsX = refstr_startx( rsP ) + rbX;
    bsendX = refstr_endx( rsP );

    /* Return EOF if past the end */
    if ( bsX >= bsendX )
	return ( EOF );

    /* Fetch the next char and advance the current pointer */
    rsP->rs_bX = rbX +1;		/* Advance the index */
    bsP = refstr_bstr( rsP );		/* Get BSTR */
    return ( bsP->bs_bP[ bsX ] );
}
/*

*//* refstr_len( rsP )

	Returns the number of bytes mapped by a REFSTR

Accepts :

	rsP		Ptr to the REFSTR

Returns :

	<value>		Number of bytes contained in this refstr

Notes :


*/

int
refstr_len ARGLIST( (rsP) )
    FARG( REFSTR	*rsP )		/* REFSTR ptr */
{
    /* Could obviously optimize this */
    return ( refstr_endx( rsP ) - refstr_startx( rsP ) );
}
/*

*//* refstr_move( rsP, n )

	Move a refstr by n bytes

Accepts :

	rsP		Ptr to the REFSTR
	n		Number of bytes to move

Returns :

	<value>		TRUE if some motion was done,
			FALSE if could not be moved

Notes :

	Moves a REFSTR's reference point.  A REFSTR is essentially
	a window into a base string; this moves the window.  It
	doesn't change any data; it changes the reference by
	moving the window up or down.  A positive value moves the
	reference towards the end, and a negative one towards the
	beginning.

	If the move is positive, the size of the REFSTR may shrink
	if the end point would be beyond the end of the base string.

	REFSTR beginning or end points that are anchored (offset
	  values of -1) will not be changed.

*/

BOOL
refstr_move ARGLIST( (rsP, n) )
   NFARG( REFSTR	*rsP )		/* REFSTR ptr */
    FARG( int		n )		/* How far to move */
{
	BOOL		movedF;
	int		nstartX;	/* New start index */
	int		nendX;		/* New end index */

    nstartX = rsP->rs_startX;		/* Will start with current locs */
    nendX = rsP->rs_endX;

    if ( n < 0 ) {
	/* Move towards beginning */
	if ( nstartX >= 0 ) {
	    /* Beginning is not anchored; move it if we can */
	    nstartX += n;
	    if ( nstartX < 0 )
		nstartX = 0;
	}
	if ( nendX >= 0 ) {
	    /* End is not anchored; move it.. */
	    nendX += n;
	    if ( nendX < 0 )
		nendX = 0;
	}
    }
    else {
	/* Move towards end */
	if ( nstartX >= 0 ) {
	    /* Beginning isn't anchored, move it if possible */
	    nstartX += n;
	    if ( nstartX >= refstr_bstr( rsP )->bs_bC )
		nstartX = refstr_bstr( rsP )->bs_bC;
	}
	if ( nendX >= 0 ) {
	    /* End isn't anchored, move it if we can */
	    nendX += n;
	    if ( nendX >= refstr_bstr( rsP )->bs_bC )
		nendX = refstr_bstr( rsP )->bs_bC;
	}
    }

    movedF = ( ( nstartX != rsP->rs_startX ) ||
	       ( nendX != rsP->rs_endX ) );
    rsP->rs_startX = nstartX;
    rsP->rs_endX = nendX;

    return ( movedF );
}
/*

*//* refstr_new( noteP, size, max )

	Allocate a new refstr

Accepts :

	noteP		A note to go with the allocation (see emalloc)
	size		Initial size of the string to allocate
	max		Maximum size (see bstr_new)

Returns :

	<value>		Ptr to REFSTR created if successful,
			NULL if not

Notes :


*/

REFSTR *
refstr_new ARGLIST( (noteP, size, max) )
   NFARG( char		*noteP )	/* Descriptive note */
   NFARG( int		size )		/* Initial buffer size */
    FARG( int		max )		/* Maximum string length, if any */
{
	REFSTR_BASE	*brsP;		/* New base object */

    /* Create the new base object */
    brsP = (REFSTR_BASE *)emalloc( "refstr_new", noteP, sizeof(REFSTR_BASE) );
    if ( brsP == NULL )
	return ( NULL );

    /* Init the BSTR */
    if ( ! bstr_newv( noteP, size, max, &brsP->bbs_bstr ) ) {
	dealloc( brsP );
	return ( NULL );
    }

    /* Do common initialization and creation of first attached REFSTR */
    return ( refstr_new_common( brsP, noteP ) );
}
/*

*//* refstr_new_bstr_copy( noteP, bsP )

	Allocate a new refstr, copying from an existing BSTR

Accepts :

	noteP		A note to go with the allocation (see emalloc)
	bsP		The existing BSTR

Returns :

	<value>		Ptr to REFSTR created if successful,
			NULL if not

Notes :

	REFSTRs have their own embedded BSTR structure in the refstr base.
	This new BSTR will be initialized with the data from the passed
	bsP -- i.e., the data in the buffer will be copied.

	The new BSTR will be created to be just large enough to contain
	the data from the passed BSTR (plus a byte or three).

	The passed bsP remains unaffected.

*/

REFSTR *
refstr_new_bstr_copy ARGLIST( (noteP, bsP) )
   NFARG( char		*noteP )	/* Descriptive note */
    FARG( BSTR		*bsP )		/* BSTR to steal from */
{
	REFSTR_BASE	*brsP;		/* New base object */

    /* Create the new base object */
    brsP = (REFSTR_BASE *)emalloc( "refstr_new", noteP, sizeof(REFSTR_BASE) );
    if ( brsP == NULL )
	return ( NULL );

    /* Create new BSTR for the refstr base */
    if ( !bstr_newv( noteP, bsP->bs_bC, 0, &brsP->bbs_bstr ) ) {
	dealloc( brsP );
	return ( NULL );
    }

    /* Copy the data, and politely nulterm it */
    bstr_catn( &brsP->bbs_bstr, bsP->bs_bP, bsP->bs_bC );
    bstr_nulterm( &brsP->bbs_bstr );

    /* Do common initialization and creation of first attached REFSTR */
    return ( refstr_new_common( brsP, noteP ) );
}
/*

*//* refstr_new_bstr_steal( noteP, bsP )

	Allocate a new refstr, stealing from an existing BSTR

Accepts :

	noteP		A note to go with the allocation (see emalloc)
	bsP		The existing BSTR

Returns :

	<value>		Ptr to REFSTR created if successful,
			NULL if not

Notes :

	REFSTRs have their own embedded BSTR structure in the refstr base.
	The passed bsP structure will be incorporated into the new REFSTR
	base, and the passed bsP will be destroyed.  The buffer pointer,
	not the data, is copied.

	Note well:  the BSTR must not be part of another REFSTR base or
	of anything that will make direct use of the BSTR in the future.

	This function will free the BSTR passed, even if an error is
	returned.  Caller must not use it again.
*/

REFSTR *
refstr_new_bstr_steal ARGLIST( (noteP, bsP) )
   NFARG( char		*noteP )	/* Descriptive note */
    FARG( BSTR		*bsP )		/* BSTR to steal from */
{
	REFSTR_BASE	*brsP;		/* New base object */

    /* Create the new base object */
    brsP = (REFSTR_BASE *)emalloc( "refstr_new", noteP, sizeof(REFSTR_BASE) );
    if ( brsP == NULL )
	return ( NULL );

    /* Steal the BSTR from the one passed, and delete the passed one. */
    brsP->bbs_bstr = *bsP;
    bsP->bs_bP = NULL;
    bstr_freev( bsP );

    /* Do common initialization and creation of first attached REFSTR */
    return ( refstr_new_common( brsP, noteP ) );
}
/*

*//* refstr_ref( rrsP, startX, endX )

	Create a new REFSTR reference based on an existing one

Accepts :

	rrsP		The REFSTR to make a reference to
	startX		offset of beginning of reference string, relative
			  to rrsP's string.
	endX		offset to end of reference string.


Returns :

	<value>		Ptr to REFSTR created if successful,
			NULL if not

Notes :

	Per REFSTR definition, a startX of -1 indicates the start of the
	  reference string, and an endX of -1 indicates the end of the
	  reference string.  These are value that are set in the refstr
	  and interpreted when used.

	Additionally, the following magic values are interpreted and
	  translated by this function:

	  -2:  for startX, will be changed to 0 (start of the base
	       string).  For endX, will be changed to the current length
	       of the base string.

	  -3:  for startX, one past the current end of the reference
	       string, if it's within the base string.
	       For endX, one less than the current start of the
	       reference string, if it's within the base string.

*/

REFSTR *
refstr_ref ARGLIST( (rrsP, startX, endX) )
   NFARG( REFSTR	*rrsP )		/* Reference REFSTR */
   NFARG( int		startX )	/* Beginning offset */
    FARG( int		endX )		/* Ending offset */
{
	REFSTR		*rsP;		/* Ptr to new refstr */

    /* Add refstr via internal routine */
    rsP = refstr_add( rrsP->rs_baseP, "via refstr_ref" );
    if ( rsP != NULL ) {
	/* Calculate new offsets based on the base REFSTR. */
	if ( startX < 0 ) {		/* Magic values */
	    switch( startX ) {
		case -1:		/* Already magic, keep it */
		    break;

		case -2:		/* Set to absolute 0 */
		    startX = 0;
		    break;

		case -3:		/* One past the current end */
		    startX = refstr_endx( rrsP ) +1;
		    if ( startX > refstr_bstr( rrsP )->bs_bC )
			startX = refstr_bstr( rrsP )->bs_bC;
		    break;

		default:		/* Unknown */
		    startX = -1;
		    break;
	    }
	}
	else {
	    startX += refstr_startx( rrsP );
	    if ( startX > refstr_bstr( rrsP )->bs_bC )
		startX = refstr_bstr( rrsP )->bs_bC;
	}


	if ( endX < 0 ) {		/* Magic values for endX */
	    switch( endX ) {
		case -1:		/* Already magic */
		    break;

		case -2:		/* Set to end of base string */
		    endX = refstr_bstr( rrsP )->bs_bC;
		    break;

		case -3:		/* One less than reference string start */
		    endX = refstr_startx( rrsP ) -1;
		    if ( endX < 0 )
			endX = 0;
		    break;

		default:		/* Unknown */
		    endX = -1;
		    break;
	    }
	}
	else {
	    endX += refstr_startx( rrsP );
	    if ( endX > refstr_bstr( rrsP )->bs_bC )
		endX = refstr_bstr( rrsP )->bs_bC;
	}

	/* If both values are non-negative make sure they are rational. */
	if ( ( startX >= 0 ) && ( endX >= 0 ) && ( startX > endX ) )
	    startX = endX;
	
	rsP->rs_startX = startX;
	rsP->rs_endX = endX;
    }

    return ( rsP );
}
/*

*//* refstr_replace( rsP, startX, endX, newP )

	Replace part of a REFSTR with nul-terminated string

Accepts :

	rsP		REFSTR to operate on
	startX		Starting offset of the area to replace
	endX		Ending offset of the area to replace
	newP		Ptr to NUL-terminated replacement string

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	Simply hands off to refstr_replacen.  See any notes there.

*/

BOOL
refstr_replace ARGLIST( (rsP, startX, endX, newP) )
   NFARG( REFSTR	*rsP )		/* Ptr to REFSTR */
   NFARG( int		startX )	/* Beginning offset */
   NFARG( int		endX )		/* Ending offset */
    FARG( UBYTE		*newP )		/* Replacement string */
{
    return ( refstr_replacen( rsP, startX, endX,
				   newP, strlen( (BYTE *)newP ) ) );
}
/*

*//* refstr_replacee( rsP, startX, endX, newP )

	Replace part of a REFSTR with ESTR

Accepts :

	rsP		REFSTR to operate on
	startX		Starting offset of the area to replace
	endX		Ending offset of the area to replace
	newP		Ptr to ESTR replacement string

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	Simply hands off to refstr_replacen.  See any notes there.

*/

BOOL
refstr_replacee ARGLIST( (rsP, startX, endX, newP) )
   NFARG( REFSTR	*rsP )		/* Ptr to REFSTR */
   NFARG( int		startX )	/* Beginning offset */
   NFARG( int		endX )		/* Ending offset */
    FARG( ESTR		*newP )		/* Replacement string */
{
    return ( refstr_replacen( rsP, startX, endX,
			      newP->es_strP, estr_len( newP ) ) );
}
/*

*//* refstr_replacen( rsP, startX, endX, newP )

	Replace part of a REFSTR with counted string

Accepts :

	rsP		REFSTR to operate on
	startX		Starting offset of the area to replace
	endX		Ending offset of the area to replace
	newP		Ptr to replacement string
	newlen		Number of bytes in replacement string

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :


*/

BOOL
refstr_replacen ARGLIST( (rsP, startX, endX, newP, newlen) )
   NFARG( REFSTR	*rsP )		/* Ptr to REFSTR */
   NFARG( int		startX )	/* Beginning offset */
   NFARG( int		endX )		/* Ending offset */
   NFARG( UBYTE		*newP )		/* Replacement string */
    FARG( int		newlen )	/* Number of bytes in newP */
{
	int		r_sX, r_eX;	/* BSTR offsets where
					    the REFSTR starts/ends */
	int		sX, eX;		/* BSTR offsets of the original
					    range to be replaced  */
	int		delta;		/* How far data was moved */
	int		osX;		/* Original rs_startX */
	BSTR		*bsP;		/* Underlying BSTR */
	LL_NODE		*llP;		/* Linked list node */

    bsP = refstr_bstr( rsP );		/* Get underlying BSTR ptr */

    /* Get the BSTR range of the REFSTR */
    r_sX = refstr_startx( rsP );
    r_eX = refstr_endx( rsP );

    /* Do the replacement on the underlying BSTR after translating
       the offsets.  
       bstr_replacen() validates these so we'll just pass them along
       without checking (but will check the return of course).
    */
    sX = r_sX + startX;
    eX = r_sX + endX;

    if ( !bstr_replacen( bsP, sX, eX, newP, newlen ) )
	return ( FALSE );

    /* If the new data size is different than the size it replaced,
       some data was shifted to account for the difference.  Any
       offsets that pointed into the old area before it moved will
       be affected.
    */
    delta = newlen - ( eX - sX );	/* Change in data size. */

    /* Protect the original startX: we never want to change the anchor
       position on the rsP being worked on.
    */
    osX = rsP->rs_startX;
    for ( llP = rsP->rs_baseP->bbs_ll.llh_firstP;
		llP != NULL;  llP = llP->lln_nextP ) {
	refstr_adjust( (REFSTR *)llP->lln_valP, eX, delta );
    }
    rsP->rs_startX = osX;

    return ( TRUE );
}
/*

*//* refstr_startx( rsP )

	Translate refstr's start point to its BSTR offset

Accepts :

	rsP		Ptr to the REFSTR

Returns :

	<value>		Start point's offset in the underlying BSTR

Notes :


*/

int
refstr_startx ARGLIST( (rsP) )
    FARG( REFSTR	*rsP )		/* REFSTR ptr */
{
	int		bsX;		/* Resulting offset */
	BSTR		*bsP;		/* Underlying BSTR */

    bsP = refstr_bstr( rsP );

    /* Offset of -1 is the beginning of the BSTR;
       any non-negative value is the simple offset.
    */
    if ( rsP->rs_startX < 0 )
	bsX = 0;
    else {
	bsX = rsP->rs_startX;

	/* Make sure the offset is within the stored data */
	if ( bsX > bsP->bs_bC )
	    bsX = bsP->bs_bC;
    }

    return ( bsX );
}
/*

*//* solbat( strP, eolF )

	Skip over leading blanks and tabs (whitespace)

Accepts :

	strP		String
	eolF		If end of line chars are considered whitespace

Returns :

	<value>		Ptr to string after whitespace

Notes :

	Name stolen from bam

*/

char *
solbat ARGLIST( (strP, eolF) )
   NFARG( char		*strP )		/* String */
    FARG( BOOL		eolF )		/* Whether eol chars are whitespace */
{
	char		ch;

    for ( ; ( ch = *strP ) != NUL; ++strP )
	if ( ( ch != ' ' ) &&
	     ( ch != '\t' ) &&
	     ( !eolF || ( ( ch != '\r' ) && ( ch != '\n' ) ) ) )
	    break;

    return ( strP );
}
/*

*//* strbyte( strP, bval )

	Find a byte in a byte string, return its index

Accepts :

	strP		Ptr to byte string to look in
	bval		Byte value to find

Returns :

	<value>		Index of the found byte, starting from zero;
			if not found returns index of NUL terminator.

Notes :


*/

int
strbyte ARGLIST( (strP, bval) )
   NFARG( UBYTE		*strP )		/* Ptr to string */
    FARG( UBYTE		bval )		/* Byte to locate */
{
	UBYTE		ch;
	int		bX;

    for ( bX = 0; (ch = *strP++) != NUL; ++bX )
	if ( ch == bval )
	    break;

    return ( bX );
}
/*

*//* strbyte_n( strP, strL, bval )

	Find a byte in a counted byte string, return its index

Accepts :

	strP		Ptr to byte string to look in
	strL		Number of bytes in the string
	bval		Byte value to find

Returns :

	<value>		Index of the found byte, starting from zero;
			if not found returns index character after the
			string's end (i.e., returns strL)

Notes :


*/

int
strbyte_n ARGLIST( (strP, strL, bval) )
   NFARG( UBYTE		*strP )		/* Ptr to string */
   NFARG( int		strL )		/* Length of string */
    FARG( UBYTE		bval )		/* Byte to locate */
{
	int		bX;

    for ( bX = 0; strL-- > 0; ++bX ) {
	if ( *strP++ == bval )
	    break;
    }

    return ( bX );
}
/*

*//* stricmp() and strnicmp()

	Like strcmp() and strncmp() but ignoring case.  These may
	not be present in the C library in use, in which case these
	routines may be conditionally compiled.

	This routine has separate paths depending on the case of the
	first character; this is to try to minimize conversion time.

*/

#ifndef HAVE_STRICMP


int
stricmp ARGLIST( ( s1P, s2P ) )
   NFARG( BYTE		*s1P )		/* First string ptr */
    FARG( BYTE		*s2P )		/* Second string ptr */
{
	BYTE		c1, c2;

    for( ; ; ) {
	c1 = *s1P++;
	c2 = *s2P++;
	if ( ( c1 >= 'A' ) && ( c1 <= 'Z' ) ) {
	    /* c1 is upper, make c2 upper */
	    if ( ( c2 >= 'a' ) && ( c2 <= 'z' ) )
		c2 -= ( 'a' - 'A' );
	}
	/* c1 is not upper, make c2 not upper */
	else if ( ( c2 >= 'A' ) && ( c2 <= 'Z' ) )
	    c2 += ( 'a' - 'A' );

	if ( c1 < c2 )
	    return( -1 );
	if ( c1 > c2 )
	    return( 1 );
	if ( c1 == NUL )
	    return( 0 );
    }
}

int
strnicmp ARGLIST( ( s1P, s2P, max ) )
   NFARG( BYTE		*s1P )		/* First string ptr */
   NFARG( register BYTE	*s2P )		/* Second string ptr */
    FARG( int		max )		/* Max length */
{
	BYTE		c1, c2;

    for( ; max > 0; --max ) {
	c1 = *s1P++;
	c2 = *s2P++;
	if ( ( c1 >= 'A' ) && ( c1 <= 'Z' ) ) {
	    /* c1 is upper, make c2 upper */
	    if ( ( c2 >= 'a' ) && ( c2 <= 'z' ) )
		c2 -= ( 'a' - 'A' );
	}
	/* c1 is not upper, make c2 not upper */
	else if ( ( c2 >= 'A' ) && ( c2 <= 'Z' ) )
	    c2 += ( 'a' - 'A' );

	if ( c1 < c2 )
	    return( -1 );
	if ( c1 > c2 )
	    return( 1 );
	if ( c1 == NUL )
	    return( 0 );
    }
    return( 0 );
}

#endif  /* HAVE_STRICMP */


/*

*//* strmatch( patP, strP, mtype, subokF, mflags )

	General string match interface for various match types

Accepts :

	patP		Pattern string
	strP		String to test
	mtype		Match type (STRMATCHTYPE)
	mflags		Flags for match control

Returns :

	<value>		result (see notes)

Notes :


	Not all flags are valid for all match types.

	Result is whatever comes out of the specific comparison
	routine.  For the SM_CMP type, it's a comparison value
	(<0, 0, or >0.  For the SM_MAT and SM_RGX types, it's
	TRUE/FALSE.

*/

int
strmatch ARGLIST( (patP, strP, mtype, mflags) )
   NFARG( UBYTE		*patP )		/* Pattern (or first string) */
   NFARG( UBYTE		*strP )		/* String to test */
   NFARG( STRMATTYPE	mtype )		/* Match type */
    FARG( UWORD		mflags )	/* Match flags */
{
	int		result;

    switch ( mtype ) {
	case	SM_STRCMP:
	    if ( ( mflags & MF_IGNCASE ) != 0 )
		result = stricmp( (BYTE *)patP, (BYTE *)strP );
	    else
		result = strcmp( (BYTE *)patP, (BYTE *)strP );
	    break;

	case	SM_MATCH:		/* Match, no wildcards */
	    result = strmatch_exact( NULL, patP, strP, mflags );
	    break;

	case	SM_WILD:		/* Match with glob-like wildcards */
	    result = strmatch_wild( NULL, patP, strP, mflags );
	    break;

	case	SM_REGEX:		/* Regular expression */
	    result = strmatch_regex( NULL, patP, strP, mflags );
	    break;

	default:
	    /* Bad... return false */
	    result = -1;
	    break;
    }

    return ( result );
}
/*

*//* strmatch_exact( cpatP, patP, strP, mflags )

	Pattern match: exact pattern match

Accepts :

	cpatP		Ptr to precompiled pattern, or NULL if none
	patP		Ptr to pattern to look for
	strP		Ptr to string to find pattern in
	mflags		Match flagbits

Returns :

	<value>		TRUE if pattern matched
			FALSE if not.

Notes :

	Calls estr_match_exact, so see notes there

	cpatP is for future use

*/

BOOL
strmatch_exact ARGLIST( (cpatP, patP, strP, mflags) )
   NFARG( void		*cpatP )	/* Precompiled pattern */
   NFARG( UBYTE		*patP )		/* The pattern */
   NFARG( UBYTE		*strP )		/* String to find pattern in */
    FARG( UWORD		mflags )	/* Match flags */
{
	BOOL		result;		/* The result */
	ESTR		*epatP, *estrP;	/* ESTRs for pattern and target */

    /* Create ESTRs from the inputs and use the general ESTR function */
    if ( ( epatP = estr_str( patP ) ) == NULL )
	return ( -1 );
    if ( ( estrP = estr_str( strP ) ) == NULL ) {
	estr_finish( epatP );
	return ( -1 );
    }

    result = estr_match_exact( cpatP, epatP, estrP, mflags );

    estr_finish( estrP );
    estr_finish( epatP );

    return ( result );
}
/*

*//* strmatch_regex( cpatP, patP, strP, mflags )

	Pattern match: regular expression pattern match

Accepts :

	cpatP		Ptr to precompiled pattern, or NULL if none
	patP		Ptr to pattern to look for
	strP		Ptr to string to find pattern in
	mflags		Match flagbits

Returns :

	<value>		TRUE if pattern matched
			FALSE if not.

Notes :

	Calls estr_match_regex to do the work, so see any notes there

	cpatP is for future use

*/

BOOL
strmatch_regex ARGLIST( (cpatP, patP, strP, mflags) )
   NFARG( void		*cpatP )	/* Precompiled pattern */
   NFARG( UBYTE		*patP )		/* The pattern */
   NFARG( UBYTE		*strP )		/* String to find pattern in */
    FARG( UWORD		mflags )	/* Match flags */
{
	BOOL		result;		/* The result */
	ESTR		*epatP, *estrP;	/* ESTRs for pattern and target */

    /* Create ESTRs from the inputs and use the general ESTR function */
    if ( ( epatP = estr_str( patP ) ) == NULL )
	return ( -1 );
    if ( ( estrP = estr_str( strP ) ) == NULL ) {
	estr_finish( epatP );
	return ( -1 );
    }

    result = estr_match_regex( cpatP, epatP, estrP, mflags );

    estr_finish( estrP );
    estr_finish( epatP );

    return ( result );
}
/*

*//* strmatch_wild( cpatP, patP, strP, mflags )

	Pattern match: modified wildcard pattern match

Accepts :

	cpatP		Ptr to precompiled pattern, or NULL if none
	patP		Ptr to pattern to look for
	strP		Ptr to string to find pattern in
	mflags		Match flagbits

Returns :

	<value>		TRUE if pattern matched
			FALSE if not.

Notes :

	Calls estr_match_wild, so see any notes there.

	cpatP is for future use

*/

BOOL
strmatch_wild ARGLIST( (cpatP, patP, strP, mflags) )
   NFARG( void		*cpatP )	/* Precompiled pattern */
   NFARG( UBYTE		*patP )		/* The pattern */
   NFARG( UBYTE		*strP )		/* String to find pattern in */
    FARG( UWORD		mflags )	/* Match flags */
{
	BOOL		result;		/* The result */
	ESTR		*epatP, *estrP;	/* ESTRs for pattern and target */

    /* Create ESTRs from the inputs and use the general ESTR function */
    if ( ( epatP = estr_str( patP ) ) == NULL )
	return ( -1 );
    if ( ( estrP = estr_str( strP ) ) == NULL ) {
	estr_finish( epatP );
	return ( -1 );
    }

    result = estr_match_wild( cpatP, epatP, estrP, mflags );

    estr_finish( estrP );
    estr_finish( epatP );

    return ( result );
}
/*

*//* strmatched_part( partN, lenP )

	Return the nth matched subpart

Accepts :

	partN		Match part number
	lenP		Where to store the string length

Returns :

	<value>		Ptr to matched [sub]string if one found
			NULL if not

Notes :

	Returned string is NUL-terminated.

	The 0th submatch is the entire match.

*/

UBYTE *
strmatched_part ARGLIST( (partN, lenP) )
   NFARG( int		partN )		/* Which subpart */
    FARG( int		*lenP )		/* Where to store the length. */
{
	int		len;		/* Length of subpart */
	UBYTE		*partP;		/* Pointer to subpart */
	BSTR		*bsP;		/* Ptr to BSTR for match part */

    /* Process according to last match type */
    switch ( Lastmatch ) {
	case SM_STRCMP:			/* Plain strcmp */
	    /* This really shouldn't get here: we don't remember
	       anything about strcmp matches, including the fact that
	       one was done.

	       Fall through to "NONE" case
	    */

	case SM_NONE:			/* No prior match type */
	    partP = NULL;		/* No return */
	    len = 0;
	    break;


	case SM_MATCH:			/* Exact match */
	case SM_WILD:			/* Wildcard match */
	case SM_REGEX:
	    if ( ( partN < 0 ) ||
		 ( partN >= LastmatchC ) ||
		 ( ( bsP = Bstrmatch[ partN ] ) == NULL ) ) {
		partP = NULL;		/* No kept match */
		len = 0;		/* No length */
	    }
	    else {
		/* Return the BSTR info */
		partP = bsP->bs_bP;
		len = bsP->bs_bC;
	    }
	    break;


	default:
	    warning( "strmatched_part: bad match type %d remembered.",
		      Lastmatch );
	    partP = NULL;
	    len = 0;
	    break;
    }

    *lenP = len;
    return ( partP );
}

/*

*//* strlc( strP )

	Converts a string to lowercase, in place

Accepts:
	strP		Ptr to string


Returns :


*/

void
strlc ARGLIST( (strP) )
    FARG( char		*strP )		/* Ptr to string */
{
	char		ch;

    while ( ( ch = *strP ) != NUL ) {
	if ( isupper( ch ) )
	    ch = tolower( ch );
	*strP++ = ch;
    }
}
/*

*//* strstr( s1P, s2P )

	Find string 2 within string 1

Accepts :

	s1P		Ptr to first string
	s2P		Ptr to second string

Returns :

	<value>		ptr to matched string,
			NULL if not found

*/

#ifndef	HAVE_STRSTR

char *
strstr ARGLIST( ( s1P, s2P ) )
   NFARG( char		*s1P )		/* First string */
    FARG( char		*s2P )		/* Second string */
{
	int		i;		/* Counter */
	int		l2;		/* String 2 length */

    /* Brute force.. */

    /* init counter */
    l2 = strlen( s2P );
    i = strlen( s1P ) - strlen( s2P );

    /* Try. */
    for( ; i-- >= 0; ++s1P )
	if ( strncmp( s1P, s2P, l2 ) == 0 )
	    return( s1P );

    return( NULL );
}

#endif	/* HAVE_STRSTR */



/*

*//* strtoken_ws( strPP, bufP, bufL, eolF )

	Extract a whitespace-delimited token from NUL-terminated string

Accepts :

	strPP		Ptr to the variable that points to the source string.
	bufP		Where to put the token
	bufL		Length of buffer for token
	eolF		If end of line chars are considered whitespace

Returns :

	<value>		Number of bytes in source token

Notes :

	Similar to estr_token_ws, but using plain NUL-terminated string
	instead of ESTR.  A token grabber which only recognizes whitespace
	as token separators.

	The string pointer at *strPP is updated to point after the
	extracted token.

	The returned byte count may be larger than the buffer size
	provided; this would indicate that the returned token is
	truncated.

	In any case, the returned token is always NUL-terminated.
	Thus the maximum token length returned is one less than the
	buffer length specified.

*/

int
strtoken_ws ARGLIST( (strPP, bufP, bufL, eolF) )
   NFARG( UBYTE		**strPP )	/* Ptr to string pointer */
   NFARG( UBYTE		*bufP )		/* Where to put the token */
   NFARG( int		bufL )		/* Size of buffer */
    FARG( BOOL		eolF )		/* Whether eol chars are whitespace */
{
	UBYTE		ch;
	int		cC;
	UBYTE		*strP;

    --bufL;				/* Account for trailing NUL */

    /* Skip whitespace prior to token */
    strP = solbat( *strPP, eolF );

    /* Grab characters until we see more whitespace */
    for ( cC = 0; ( ch = *strP ) != NUL; ++cC, ++strP ) {
	if ( ( ch == ' ' ) ||
	     ( ch == '\t' ) ||
	     ( eolF && ( ch == '\r' ) || ( ch == '\n' ) ) )
	    break;

	if ( cC < bufL )
	    *bufP++ = ch;
    }

    *strPP = strP;
    *bufP = NUL;

    return ( cC );
}
/*

*//* tmpstr( len )

	Access a temporary string buffer

Accepts :

	len		Length of buffer needed

Returns :

	<value>		Pointer to a string buffer,
			NULL if error (warning given)

Notes :

	Returns a pointer to one of several scratch string buffers.
	These are available as *very* temporary registers.  Each call
	to this routine will return the next available buffer in
	round-robin fashion.  Caller must make sure the use is
	indeed temporary.

*/

UBYTE *
tmpstr ARGLIST( (len) )
    FARG( int		len )		/* Length needed */
{

#define	TMPSTRNUM	10
static	int		tmpstrX = -1;
static	struct {
    int ts_len;
    UBYTE *ts_strP;
}  tmpstrs[ TMPSTRNUM ];

    if ( ++tmpstrX == TMPSTRNUM )
	tmpstrX = 0;

    if ( tmpstrs[ tmpstrX ].ts_strP == NULL ) {
	if ( len < 20 )			/* Enforce a minimum */
	    len = 20;
	tmpstrs[ tmpstrX ].ts_strP = emalloc( "tmpstr", "str", len + 1 );
	tmpstrs[ tmpstrX ].ts_len = len;
    }
    else if ( tmpstrs[ tmpstrX ].ts_len < len ) {
	tmpstrs[ tmpstrX ].ts_strP = erealloc( "tmpstr", "str",
						tmpstrs[ tmpstrX ].ts_strP,
						len + 1 );
	tmpstrs[ tmpstrX ].ts_len = len;
    }

    return ( tmpstrs[ tmpstrX ].ts_strP );
}
/*

*//* tmpstr_estr( esP )

	Return a temporary nul-terminated string from an ESTR (with caveats)

Accepts :

	esP		String we want to access nul-terminated

Returns :

	<value>		Pointer to a string buffer.

Notes :

	This returns a pointer to a nul-terminated string that a caller
	can use e.g. in an error message.  It *may* return a pointer
	to the actual string buffer in the estr, or if that's not
	possible it will return a pointer to another string that's
	a subset of the input.  The goal of this routine is to return
	something: not necessarily the entire input string, but
	at least something that can be used for reporting.

*/

UBYTE *
tmpstr_estr ARGLIST( (esP) )
    FARG( ESTR		*esP )		/* The input string */
{
	int		i;
	UBYTE		*strP;		/* Return value */

    /* If the input is a nul-terminated string, just return that */
    if ( esP->es_len < 0 )
	return ( esP->es_strP );

    /* Ditto if it's a counted string that just happens to be
        NUL-terminated
    */
    i = estr_len( esP );
    if ( esP->es_strP[i] == NUL )
	return ( esP->es_strP );

    /* Return up to some arbitrary number in a tmpstr */
    if ( i > 50 )
	i = 50;
    if ( ( strP = tmpstr( i+1 ) ) == NULL )
	return ( (UBYTE *)"" );
    strncpy( (BYTE *)strP, (BYTE *)esP->es_strP, i );
    strP[i] = NUL;
    return ( strP );
}


/***********************************************************************
 *                                                                     *
 *                     Internal support routines                       *
 *                                                                     *
 ***********************************************************************/


/*

*//* bstr_freev( bsP )

	Delete any values of a BSTR

Accepts :

	bsP		Pointer to byte string object

Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :


*/

static BOOL
bstr_freev ARGLIST( (bsP) )
    FARG( BSTR		*bsP )		/* Ptr to byte string object */
{
    if ( bsP->bs_bP != NULL ) {
	dealloc( bsP->bs_bP );
	bsP->bs_bP = NULL;
    }

    return ( TRUE );
}
/*

*//* bstr_newv( noteP, size, max, bsP )

	Create a new byte string object's values

Accepts :

	noteP		A descriptive note for this new object
	size		Initial string size to allocate.
			0 indicates a default value.
	max		The maximum byte length for this object.
			0 indicates a default value.
	bsP		Ptr to BSTR structure to fill in

Returns :

	<value>		TRUE if successful,
			FALSE otherwise

Notes :

	Provides common code for filling in a new BSTR.

*/

static BOOL
bstr_newv ARGLIST( (noteP, size, max, bsP) )
   NFARG( char		*noteP )	/* Descriptive note */
   NFARG( int		size )		/* Initial buffer size */
   NFARG( int		max )		/* Maximum string length, if any */
    FARG( BSTR		*bsP )		/* Ptr to the BSTR to fill in */
{
    /* Allocate the initial size string */
    if ( size <= 0 )
	size = BSTR_CHUNK;

    bsP->bs_bP = emalloc( "bstr_new", noteP, size + BSTR_SLOP );
    if ( bsP->bs_bP == NULL )
	return ( FALSE );

    bsP->bs_bC = 0;
    bsP->bs_bX = 0;
    bsP->bs_bL = size;
    bsP->bs_chunk = 0;		/* Use default chunk size */

    bsP->bs_bM = ( max > 0 ? max : BSTR_MAX );

    if ( size > bsP->bs_bM )
	bsP->bs_bM = size;

    return ( TRUE );
}
/*

*//* estr_match_wild_sub( patP, pX, esP, sX, level, mflags )

	ESTR match: handler routine for estr_match_wild()

Accepts :

	patP		Pattern ESTR
	pX		Initial index into patP
	esP		ESTR to find pattern in
	sX		Initial index into esP
	level		Recursion level
	mflags		Match flagbits

Returns :

	<value>		TRUE if pattern matched
			FALSE if not.

Notes :

	Not all mflags flagbits are relevant

	Modified wildcard :
	    ? matches any single char
	    * matches zero or more chars

*/

static BOOL
estr_match_wild_sub ARGLIST( (patP, pX, esP, sX, level, mflags) )
   NFARG( ESTR		*patP )		/* Pattern (or first string) */
   NFARG( int		pX )		/* Pattern index */
   NFARG( ESTR		*esP )		/* String to test */
   NFARG( int		sX )		/* String index */
   NFARG( int		level )		/* Recursion level */
    FARG( UWORD		mflags )	/* Match flagbits */
{
	BOOL		igncaseF;
	BOOL		matchF;		/* If matched */
	BOOL		doneF;		/* If done */
	int		pch, sch;	/* Chars */
	int		isX;		/* Incoming sX */

    igncaseF = ( (mflags & MF_IGNCASE) != 0 );

    /* Loop through both strings */
    for ( matchF = doneF = FALSE; !doneF; ) {
	pch = estr_nextx( patP, pX );	/* Next pattern char */
	sch = estr_nextx( esP, sX );	/* Next string char */

	/* Check for escaped wild char */
	if ( ( pch == '\\' ) &&
	     ( ( estr_bytex( patP, pX ) == '*' ) ||
	       ( estr_bytex( patP, pX ) == '?' ) ) )
	    pch = estr_nextx( patP, pX );  /* Use this literally */
	else {
	    /* Do wildcard processing */
	    if ( pch == '?' ) {
		/* Match any character except EOF terminator */
		if ( sch == EOF )
		    doneF = TRUE;
		matchF = estr_match_wild_sub( patP, pX,
					      esP, sX,
					      level +1, igncaseF );
		if ( matchF )
		    match_keep( level, esP, sX -1, 1 );
		break;
	    }

	    if ( pch == '*' ) {
		/* Match zero or more characters. */

		/* Try this shortcut first: if this is the end of the
		   pattern, we matched.
		*/
		if ( estr_bytex( patP, pX ) == EOF ) {
		    doneF = matchF = TRUE;
		    --sX;
		    match_keep( level, esP, sX, estr_len( esP ) - sX );
		    break;
		}

		/* Recurse to repeatedly try the rest of the pattern against
		   the rest of the string, skipping one character in the
		   string at each iteration.
		*/
		for ( isX = --sX; estr_bytex( esP, sX ) != EOF; ++sX ) {
		    matchF = estr_match_wild_sub( patP, pX,
						  esP, sX,
						  level +1, igncaseF );
		    if ( matchF ) {
			match_keep( level, esP, isX, sX - isX );
			break;
		    }
		}

		/* Tried each tail until match or done, done at any rate. */
		doneF = TRUE;
		break;
	    }
	}


	/* Not special -- do character match */
	if ( igncaseF ) {
	    /* Fold case */
	    if ( ( pch >= 'A' ) && ( pch <= 'Z' ) ) {
		/* pch is upper, make sch upper */
		if ( ( sch >= 'a' ) && ( sch <= 'z' ) )
		    sch -= ( 'a' - 'A' );
	    }
	    /* pch is not upper, make sch not upper */
	    else if ( ( sch >= 'A' ) && ( sch <= 'Z' ) )
		sch += ( 'a' - 'A' );
	}

	if ( pch != sch )
	    doneF = TRUE;		/* No match */
	else if ( sch == EOF )
	    doneF = matchF = TRUE;	/* End of string -- matched */
    }

    /* Return the result */
    return ( matchF );
}
/*

*//* estr_tmp()

	Allocate a temporary/transient ESTR

Accepts :

	<nothing>

Returns :

	<value>		Ptr to a temporary ESTR

Notes :

	Returns a pointer to one of a round-robin pool of temporary
	ESTRs.  These are to be considered as temporary registers;
	caller shall make sure not to use the returned ESTR for very
	long.  No release or finish is necessary.

*/

static ESTR *
estr_tmp NOARGLIST
{
#define	TMPESTRNUM	10
static	int	tmpestrX = -1;
static	ESTR	tmpestrs[ TMPESTRNUM ];

    if ( ++tmpestrX == TMPESTRNUM )
	tmpestrX = 0;
    return ( &tmpestrs[ tmpestrX ] );
}
/*

*//* match_keep( partN, esP, sX, len )

	Remembers (keeps) a submatch for strmatch functions.

Accepts :

	partN		The subpart number to keep (0 is the entire match)
	esP		Ptr to ESTR containing the string
	sX		Index to first character in the matched string
	len		Length of the area in the matched string


Returns :

	<value>		TRUE if OK
			FALSE if not.

Notes :

	subparts are remembered in BSTR structs and are nul-terminated.

*/

static BOOL
match_keep ARGLIST( (partN, esP, sX, len) )
   NFARG( int		partN )		/* Subpart number */
   NFARG( ESTR		*esP )		/* Ptr to encapsulated string */
   NFARG( int		sX )		/* Start of the matched string */
    FARG( int		len )		/* Number of bytes */
{
	int		n;		/* Scratch */
	BSTR		*bsP;		/* Ptr to BSTR for match part */

    if ( ( partN < 0 ) || ( partN > MATCHKEEP ) ) {
	warning( "match_keep: part number %d out of range.", partN );
	return ( FALSE );
    }

    if ( ( bsP = Bstrmatch[ partN ] ) == NULL ) {
	/* No BSTR has been created for this part number, so make one */
	n = len + 100;			/* Reasonable initial size */
	if ( n < 500 )
	    n = 500;
	bsP = bstr_new( "match_keep part", n, 0 );

	if ( bsP == NULL )
	    return ( FALSE );
	Bstrmatch[ partN ] = bsP;
    }

    /* Save the string */
    bsP->bs_bC = 0;
    if ( bstr_catn( bsP, esP->es_strP + sX, len ) ) {
	bstr_nulterm( bsP );

	if ( partN >= LastmatchC )
	    LastmatchC = partN +1;
	return ( TRUE );
    }

    bstr_nulterm( bsP );
    return ( FALSE );
}
/*

*//* refstr_add( brsP, noteP )

	Add a new REFSTR to a refstr base 

Accepts :

	brsP		REFSTR_BASE to add the REFSTR to
	noteP		Comment for the allocation

Returns :

	<value>		Ptr to newly allocated REFSTR
			NULL if failure

Notes :

	Private routine to localize initialization of REFSTR

*/

static REFSTR *
refstr_add ARGLIST( (brsP, noteP) )
   NFARG( REFSTR_BASE	*brsP )		/* Base for refstr */
    FARG( char		*noteP )	/* Note for the allocation */
{
	REFSTR		*rsP;		/* Ptr to refstr */
	LL_NODE		*llP;		/* Linked list ptr */

    rsP = (REFSTR *)emalloc( "refstr_new", noteP, sizeof( REFSTR ) );
    if ( rsP == NULL )
	return ( NULL );

    /* Attempt to add to the linked list */
    llP = ll_addvt( &brsP->bbs_ll, rsP );
    if ( llP == NULL ) {
	dealloc( rsP );
	return ( NULL );
    }

    /* Fill in defaults */
    rsP->rs_llP = llP;			/* Ptr to linked list node */
    rsP->rs_baseP = brsP;		/* Ptr to base */
    rsP->rs_bX = -1;			/* Current index, uninitialized */
    rsP->rs_refC = 1;			/* 1 ref.. */

       /* caller should fill in start and end */

    return ( rsP );
}
/*

*//* refstr_adjust( rsP, aaX, delta )

	Adjust a REFSTR to account for moved data

Accepts :

	rsP		Ptr to REFSTR to adjust
	aaX		BSTR offset of start of affected area
	delta		How far the affected area moved

Returns :

	<nothing>

Notes :


*/

static void
refstr_adjust ARGLIST( (rsP, aaX, delta) )
   NFARG( REFSTR	*rsP )		/* The REFSTR to adjust */
   NFARG( int		aaX )		/* BSTR offset of affected area */
    FARG( int		delta )		/* How far the area moved */
{
	int		startX;		/* The start of REFSTR within BSTR */

    startX = refstr_startx( rsP );	/* We'll need this */

    /* Adjust the beginning offset if it's relative */
    if ( ( rsP->rs_startX >= 0 ) && ( startX >= aaX ) )
	    rsP->rs_startX += delta;

    /* Adjust the end.. */
    if ( ( rsP->rs_endX >= 0 ) && ( refstr_endx( rsP ) >= aaX ) )
	    rsP->rs_endX += delta;
}
/*

*//* refstr_new_common( brsP, noteP )

	Common part of making a new REFSTR

Accepts :

	brsP		Ptr to refstr_base for new refstr
	noteP		A note to go with the allocation (see emalloc)

Returns :

	<value>		Ptr to REFSTR created if successful,
			NULL if not

Notes :

	Common code for the refstr_new functions; this does the common
	initialization and setup once the refstr_base and its bstr
	have been created.

	If there is a failure, the refstr_base AND its attached BSTR
	are both deallocated.

*/

static REFSTR *
refstr_new_common ARGLIST( (brsP, noteP) )
   NFARG( REFSTR_BASE	*brsP )		/* Base object being made */
    FARG( char		*noteP )	/* Descriptive note */
{
	REFSTR		*rsP;		/* New refstr object */

    /* Init the refstr list */
    ll_init( &brsP->bbs_ll );

    /* Create initial REFSTR */
    rsP = refstr_add( brsP, noteP );
    if ( rsP == NULL ) {
	bstr_freev( &brsP->bbs_bstr );
	dealloc( brsP );
	return ( NULL );
    }

    /* Fill in the offsets to indicate the entire string. */
    rsP->rs_startX = -1;
    rsP->rs_endX = -1;

    return ( rsP );
}
