/* mml-ttio.c -- Terminal I/O stuff

	Copyright 2003,2004,2005,2006,2007
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

Some basic terminal I/O stuff, mainly intended to go with mml-in,
but with some other uses as well I suppose.

Note: this file #includes an operating-system-specific module
depending on the compiler OS_xxx flag.

*/

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <string.h>
#include <ctype.h>

#include <mml/mml.h>

#include <mml/mml-alloc.h>
#include <mml/mml-ll.h>
#include <mml/mml-str.h>
#include <mml/mml-in.h>
#include <mml/mml-ttio.h>

    /* Include the os-specific stuff */
#ifdef	OS_UNIX
# include "mml-ttio.unx"
#endif	/* OS_UNIX */
#ifdef	OS_MSDOS
# include "mml-ttio.msd"
#endif	/* OS_MSDOS */

/* Local definitions */

	/* Internal values for special keys */
#define	KCLEFT		-1		/* Left arrow */
#define	KCRIGHT		-2		/* Right arrow */
#define	KCUP		-3		/* Up arrow */
#define	KCDOWN		-4		/* Down arrow */
#define	KCINSERT	-5		/* Insert */
#define	KCDELETE	-6		/* Delete */
#define	KCHOME		-7		/* Home */
#define	KCEND		-8		/* End */


	/* Editing keys */
#define	EKBEGIN		'\001'		/* Beginning of line */
#define	EKEND		'\005'		/* End of line */
#define	EKLEFT		'\002'		/* Left */
#define	EKRIGHT		'\006'		/* Right */
#define	EKKPWORD	'\027'		/* Kill previous word */
#define	EKKBEG		'\025'		/* Kill to beginning */
#define	EKKEND		'\013'		/* Kill to end of line */
#define	EKREPEAT	'\022'		/* Repeat line */



/* External data referenced */



/* External routines used */



/* Local data publicly available */



/* Local routines and forward references */

    /* Count column position */
static	int	colcount PROTO( (int column, char *bufP, int bufC) );

    /* Output character and update column position */
static	int	coloutc PROTO( (int curcol, char ch) );

    /* Set the cursor to a particular column */
static	void	setcol PROTO( (int curcol, int destcol) );

/* Private data */


/*

*//* tt_cmdline_input( cmlP )

	Input a command line from terminal

Accepts :

	cmlP		Ptr to command line object

Returns :

	<value>		TRUE if OK
			FALSE if not (like EOF)

Notes :


*/

BOOL
tt_cmdline_input ARGLIST( (cmlP) )
    FARG( TTCMDLN	*cmlP )		/* Command line thing */
{
	int		i;		/* Scratch */
	int		ch;		/* Character */
	int		cC;		/* Character count */
	int		bufX;		/* Buffer index */
	int		colC;		/* Column counter */
	int		curcol;		/* Current column */
	int		startcol;	/* Start column */
	int		newcol;		/* New column counter */
	int		fdelX;		/* First delete index */
	int		ldelX;		/* Last delete index */
	int		delC;		/* Deletion count */
	char		*promptP;	/* Ptr to specific prompt used */
	BSTR		*bstrP;		/* Ptr to bstr buffer */

    promptP = (cmlP->tc_contF ? cmlP->tc_cpromptP : cmlP->tc_promptP);
    bstrP = (BSTR *)cmlP->tc_bstrP;

    cC = bufX = 0;			/* Zero counters */

    tt_puts( promptP );			/* Output the prompt */

    colC = startcol = curcol = colcount( 0, promptP, strlen( promptP ) );
    for( ; ; ) {			/* Process input characters */
	ch = tt_inchar();		/* Get character */
	if ( ( ch >= ' ' ) || ( ch == '\011' ) ) {
	    /* "Normal" character */
	    if ( ( ( cC == bufX ) ||
	           ( cmlP->tc_insertF ) ) && ( cC ==  bstrP->bs_bM ) ) {
	    	/* Buffer full */
		tt_outchar( '\007' );	/* Beep */
		continue;		/* Do another one */
	    }

	    curcol = coloutc( curcol, ch );

	    /* Make sure there is room in the buffer */
	    if ( ( bufX == cC ) || cmlP->tc_insertF )
	    	bstr_putb( bstrP, 'x' );

	    if ( bufX == cC )
	    	++cC;
	    else if ( cmlP->tc_insertF )
		for( i = ++cC; i > bufX; --i )
		    bstrP->bs_bP[i] = bstrP->bs_bP[i-1];

	    /* Store the new character */
	    bstrP->bs_bP[bufX++] = ch;

	    /* Update the line, if appropriate */
	    for( newcol = curcol, i = bufX; i < cC; ++i )
	        newcol = coloutc( newcol, bstrP->bs_bP[i] );

	    /* Trim end of line if it needs it */
	    for( i = newcol; i < colC; ++i )
	    	tt_outchar( ' ' );

	    colC = newcol;		/* Update column */
	    if ( i != curcol )		/* Position back ? */
	        setcol( i, curcol );
	}
	else {
	    /* Not "Normal" character */
	    if ( ( ch == '\n' ) || ( ch == '\r' ) ) {
	    	tt_outchar( '\r' );
		tt_outchar( '\n' );
	    	break;
	    }

	    /* Process the character */
	    switch( ch ) {
	        case EKLEFT:		/* Move cursor left */
		case KCLEFT:
		    if ( bufX != 0 ) {
			newcol = colcount( startcol,
					     (BYTE *)bstrP->bs_bP, --bufX );
			setcol( curcol, newcol );
			curcol = newcol;
		    }
		    break;


		case EKRIGHT:		/* Move cursor right */
		case KCRIGHT:
		    if ( bufX < cC )
		        curcol = coloutc( curcol, bstrP->bs_bP[bufX++] );
		    break;


		case EKBEGIN:		/* Move cursor to beginning */
		case KCHOME:
		    setcol( curcol, startcol );
		    curcol = startcol;
		    bufX = 0;
		    break;


		case EKEND:		/* Move cursor to end of line */
		case KCEND:
		    for( ; bufX < cC; ++bufX )
		        curcol = coloutc( curcol, bstrP->bs_bP[bufX] );
		    break;


		case KCINSERT:		/* Toggle insert/overstrike */
		    cmlP->tc_insertF = !cmlP->tc_insertF;
		    break;


		case EKREPEAT:		/* Redraw the line */
		    tt_outchar( '\r' );
		    for( i = 0; i < 79; ++i )
		        tt_outchar( ' ' );
		    tt_outchar( '\r' );
		    tt_puts( promptP );
		    bstrP->bs_bP[cC] = NUL;
		    tt_puts( (BYTE *)bstrP->bs_bP );
		    setcol( colC, curcol );
		    break;


		case KCDELETE:			/* All deletions */
		case EKKPWORD:
		case EKKBEG:
		case EKKEND:
		case '\010':
		    /* Find the range of the deletion */
		    switch( ch ) {
		        case KCDELETE:
			    if ( bufX == cC ) 	/* If at end of line */
			        fdelX = ldelX = bufX-1;
			    else
			    	fdelX = ldelX = bufX;
			    break;

			case '\010':
			    fdelX = ldelX = bufX-1;
			    break;

			case EKKPWORD:
			    /* Skip back over spaces, first */
			    for( i = bufX-1; i >= 0; --i )
			    	if ( ( ( ch = bstrP->bs_bP[i] ) != ' ' ) &&
				     ( ch != '\011' ) )
				    break;

			    if ( i >= 0 ) {
				if ( isalpha(ch) ) {
				    /* On a word, skip back over it */
				    for( ; i >= 0; --i ) {
				        ch = bstrP->bs_bP[i];
					if ( !isalpha(ch) )
					    break;
				    }
				}
				else {
				    /* Not on a word -- skip back. */
				    for( ; i >= 0; --i ) {
				        ch = bstrP->bs_bP[i];
					if ( isalpha(ch) )
					    break;
				    }
				}
			    }

			    fdelX = i+1;
			    ldelX = bufX-1;
			    break;

			case EKKBEG:
			    fdelX = 0;
			    ldelX = bufX-1;
			    break;

			case EKKEND:
			    fdelX = bufX;
			    ldelX = cC-1;
			    break;
		    }

		    /* Delete from fdelX to ldelX */
		    if ( ( fdelX < 0 ) || ( ldelX >= cC ) ||
		         ( fdelX > ldelX ) )
			 break;

		    /* Collapse the buffer and update the display. */
		    delC = (ldelX - fdelX) +1;
		    cC -= delC;
		    bstrP->bs_bC -= delC;
		    bufX = fdelX;
		    newcol = colcount( startcol,
				       (BYTE *)bstrP->bs_bP, bufX );
		    setcol( curcol, newcol );
		    curcol = newcol;
		    for( i = fdelX; i < cC; ++i ) {
		        bstrP->bs_bP[i] = bstrP->bs_bP[i+delC];
			newcol = coloutc( newcol, bstrP->bs_bP[i] );
		    }

		    /* Clear cruft at end of line */
		    for( i = newcol; i < colC; ++i )
		    	tt_outchar( ' ' );
		   
		    colC = newcol;
		    setcol( i, curcol );
		    break;


	    	default:
		    tt_outchar( '\007' );
		    break;
	    }
	}
    }

    bstrP->bs_bP[cC] = NUL;		/* Not that we need it, but... */
    bstrP->bs_bC = cC;			/* Character count */

    if ( ( cC == 0 ) && ( ch == EOF ) )
	return ( FALSE );
    return ( TRUE );
}
/*

*//* in_h_ttcmd_finish( instrP )

	Input stream handler for "ttcmd": finish the stream

Accepts :

	instrP		Ptr to the input stream

Returns :

	<nothing>

Notes :

	This is a routine that is specified as a "finish" function
	for a ttio command line input (i.e., via in_stream_push).
	It follows mml-in input stream semantics.

	This finish function releases the intermediate TTCMDSTR
	object that keeps track of stream input, but does NOT
	release any underlying data (e.g. the TTCMDLN).

*/

void
in_h_ttcmd_finish ARGLIST( (instrP) )
    FARG( MML_INS	*instrP )	/* Input stream pointer */
{
	TTCMDSTR	*tcstrP;	/* ttio command stream thing */

    /* Deallocate the control structure */
    tcstrP = (TTCMDSTR *)instrP->is_handleP;
    dealloc( tcstrP );

    /* done */
}
/*

*//* in_h_ttcmd_getc( instrP )

	Input stream handler for "ttcmd": get char

Accepts :

	instrP		Ptr to the input stream

Returns :

	<value>		character input
			EOF if end of string

Notes :

	This is a routine that is specified as a "get character"
	routine for a ttio command line input (i.e., via in_stream_push).
	It follows mml-in input stream semantics.

	A ttcmd stream inputs directly from the console.

*/

int
in_h_ttcmd_getc ARGLIST( (instrP) )
    FARG( MML_INS	*instrP )	/* Input stream pointer */
{
	char		ch;
	TTCMDSTR	*tcstrP;	/* ttio command stream thing */
	BSTR		*bstrP;		/* Ptr to underlying bstr */

    tcstrP = (TTCMDSTR *)instrP->is_handleP;

    /* A fetch index of -1 means we need a new command line input */
    if ( tcstrP->ts_fetchX == -1 ) {
	/* Input new command line */
	tcstrP->ts_fetchX = 0;
	if ( !tt_cmdline_input( tcstrP->ts_cmlP ) )
	    return ( EOF );
    }

    bstrP = (BSTR *)(tcstrP->ts_cmlP->tc_bstrP);
    if ( tcstrP->ts_fetchX == bstrP->bs_bC ) {
	ch = '\n';			/* End of string, return newline */
	tcstrP->ts_fetchX = -1;		/* Get new line next time */
    }
    else
	ch = bstrP->bs_bP[ tcstrP->ts_fetchX++ ];

    return ( ch );
}
/*

*//* in_stream_push_ttcmd( inP, cmlP, nameP )

	Push a new terminal command stream onto the input 


Accepts :

	inP		Ptr to input handle
	cmlP		Ptr to command line input handle
	nameP		Name for this stream, or NULL if none

Returns :

	<value>		TRUE if OK
			FALSE if not.
			May take error exit on allocation failure.

Notes :

	Pushes a stream using standard input functions a la mml-in .

	nameP's string is copied to make a persistent copy.

*/

BOOL
in_stream_push_ttcmd ARGLIST( (inP, cmlP, nameP) )
   NFARG( MML_IN	*inP )		/* Input handle */
   NFARG( TTCMDLN	*cmlP )		/* Ptr to command line struct */
    FARG( char		*nameP )	/* Name for the stream */
{
	TTCMDSTR	*tcstrP;	/* Ptr to stream object */

    /* Construct the stream control object */
    tcstrP = (TTCMDSTR *)emalloc( "in_stream_push_ttcmd", "stream object",
				      sizeof(TTCMDSTR) );
    if ( tcstrP == NULL )
	return ( FALSE );

    /* Fill it in */
    tcstrP->ts_cmlP = cmlP;
    tcstrP->ts_fetchX = -1;		/* Needs input */

    /* Push this stream onto the stack */
    return ( in_stream_push( inP, in_h_ttcmd_getc, in_h_ttcmd_finish,
			     (void *)tcstrP, nameP ) );
}
/*

*//* tt_cmdline_new( promptP, cpromptP )

	Allocate a new command line structure

Accepts :

	promptP		Primary prompt string
	cpromptP	Continuation line prompt string

Returns :

	<value>		Ptr to TTCMDLN structure
			NULL if error (may take error exit)

Notes :

	Prompt strings are assumed to be persistent (i.e. we don't
	 make a copy of them here).

	Note that the prompt/cprompt duality is merely a convenience
	 that is triggered by somebody setting the "tc_contF" flag.
	 There is no automatic determination of which prompt is
	 supposed to be used.

*/

TTCMDLN *
tt_cmdline_new ARGLIST( (promptP, cpromptP) )
   NFARG( char		*promptP )	/* Main prompt */
    FARG( char		*cpromptP )	/* Continuation prompt */
{
	TTCMDLN		*cmlP;		/* Ptr to command line struct */

    /* Create the new structure */
    cmlP = (TTCMDLN *)emalloc( "tt_cmdline_new", "command line struct",
			sizeof(TTCMDLN) );
    if ( cmlP == NULL )
	return ( NULL );

    /* Allocate bstr to go with it.  We'll just use defaults. */
    cmlP->tc_bstrP = bstr_new( "command line", 0, 0 );
    if ( cmlP->tc_bstrP == NULL ) {
	dealloc( cmlP );
	return ( NULL );
    }

    /* Fill in the rest of the info */
    cmlP->tc_promptP = promptP;
    cmlP->tc_cpromptP = cpromptP;
    cmlP->tc_contF = FALSE;
    cmlP->tc_insertF = TRUE;

    return ( cmlP );
}

/***********************************************************************
 *                                                                     *
 *                     Internal support routines                       *
 *                                                                     *
 ***********************************************************************/


/*

*//* coloutc( curcol, ch )

	Output character, return updated column count

Accepts :

	curcol		Current column
	ch		Character to output

Returns :

	<value>		Updated column number

Notes :

	Strictly in support of command line input

*/

static int
coloutc ARGLIST( (curcol, ch) )
   NFARG( int		curcol )	/* Current column */
    FARG( char		ch )		/* Character to putput */
{
	int		newcol;		/* New column number */

    if ( ch == '\011' ) {		/* If tab */
	newcol = ( curcol + 8 ) & (-8);
	for( ; curcol != newcol; ++curcol )
	    tt_outchar( ' ' );
    }
    else {
        tt_outchar( ch );
        ++curcol;
    }

    return( curcol );
}
/*

*//* colcount( column, bufP, bufC )

	Count column position

Accepts :

	column		Starting column
	bufP		Ptr to characters to count
	bufC		Number of characters to count

Returns :

	<value>		Column count

Notes :

	Strictly in support of command line input

*/

static int
colcount ARGLIST( (column, bufP, bufC) )
   NFARG( int		column )	/* Starting column */
   NFARG( char		*bufP )		/* Ptr to chars to count */
    FARG( int		bufC )		/* Number of chars to count */
{
	char		ch;		/* Character */

    while( bufC-- > 0 ) {
    	ch = *bufP++;			/* Get char */
	if ( ch >= ' ' )		/* Printing character */
	    ++column;
	else if ( ch == '\011' )	/* Tab */
	    column = (column +8) & (-8);
	else if ( ( ch == '\r' ) || ( ch == '\n' ) )
	    column = 0;
	else
	    ;				/* ?? */
    }

    return( column );
}
/*

*//* setcol( curcol, destcol )

	Set the cursor to the desired column position

Accepts :

	curcol		Where we think the cursor is
	destcol		Where we want the cursor to be

Returns :

	<nothing>

Notes :

	Strictly in support of command line input

 	This routine sets the column in a dumb way, assuming no
	direct cursor addressing capability.

*/

static void
setcol ARGLIST( (curcol, destcol) )
   NFARG( int		curcol )	/* Where we think the cursor is */
    FARG( int		destcol )	/* Where we want the cursor to be */
{
    while( curcol-- > destcol )
        tt_outchar( '\010' );		/* Backspace */
}
