/*	avl.h		Definitions for avl routines

	Copyright 2003,2004,2005,2006,2007
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

*/

#ifndef	H_MML_AVL			/* Prevent multiple inclusions */
#define	H_MML_AVL


	/* Structures */

/* Structure of an avl tree node.  Note that this node is meant to
   be used as a header or component of an application-specific structure,
   since there is no key or data information present in the avlnode
   structure.
*/

typedef					/* A node in an AVL tree */
  struct avlnode {
    struct avlnode *n_leftP;		/* Ptr to left subtree */
    struct avlnode *n_rightP;		/* Ptr to right subtree */
    int		n_balance;		/* Balance count */
  } AVLNODE;


typedef					/* The header for an AVL tree */
  struct {
    /* Tree parameters */
    AVLNODE	*t_rootP;		/* Ptr to root node */
    int		t_nodeC;		/* Number of nodes */
    void	*t_infoP;		/* Anything that the user wants
					   to associate with the tree.
					*/
    void	*t_walkP;		/* Something used for walking */

#ifdef	MML_PTHREADS
    pthread_mutex_t t_lock;		/* To lock during critical operations */
#endif	/* MML_PTHREADS */

    /* Handler functions for the tree */
    int		(*t_cmpkey)();		/* Compare two keys */
    AVLNODE	*(*t_mknode)();		/* Node maker */
    int		(*t_rmnode)();		/* Node destroyer */
  } AVLTREE;


/* Function declarations: */

    /* Return the node count for a tree */
int	avlcount PROTO( (AVLTREE *treeP) );

    /* Delete a node from a tree */
int	avldelete PROTO( (AVLTREE *treeP, void *keyP) );

    /* Delete a node from a tree, without locking */
int	avldelete_nolock PROTO( (AVLTREE *treeP, void *keyP) );

    /* Delete a node from a tree, without unlocking */
int	avldelete_nounlock PROTO( (AVLTREE *treeP, void *keyP) );

    /* Delete all nodes from a tree */
void	avldeltree PROTO( (AVLTREE *treeP) );

    /* Find an entry in the tree */
AVLNODE	*avlfind PROTO( (AVLTREE *treeP, void *keyP) );

    /* Find an entry in the tree, without locking */
AVLNODE	*avlfind_nolock PROTO( (AVLTREE *treeP, void *keyP) );

    /* Find an entry in the tree, without unlocking */
AVLNODE	*avlfind_nounlock PROTO( (AVLTREE *treeP, void *keyP) );

    /* Insert a node in a tree */
int	avlinsert PROTO( (AVLTREE *treeP, void *keyP, void *dataP) );

    /* Insert a node in a tree, without locking */
int	avlinsert_nolock PROTO( (AVLTREE *treeP, void *keyP, void *dataP) );

    /* Insert a node in a tree, without unlocking */
int	avlinsert_nounlock PROTO( (AVLTREE *treeP, void *keyP, void *dataP) );

    /* Create a new AVL tree */
AVLTREE	*avlnew PROTO( (AVLTREE *treeP, int (*cmpkey)(),
			AVLNODE *(*mknode)(), int (*rmnode)() ) );

    /* Walk the tree */
void	avlwalk PROTO( (AVLTREE *treeP, int (*rtc)() ) );

    /* Walk the tree, without locking */
void	avlwalk_nolock PROTO( (AVLTREE *treeP, int (*rtc)() ) );

    /* Walk the tree, without unlocking */
void	avlwalk_nounlock PROTO( (AVLTREE *treeP, int (*rtc)() ) );


    /* Handlers for simple string trees: the key is a string, and
       there is no associated data.  Useful for simply putting strings
       into a tree and checking for their presence.  The passed string
       (key) is copied for use by the tree.
    */
int	mml_avl_str_cmpkey();
AVLNODE	*mml_avl_str_mknode();
int	mml_avl_str_rmnode();


	/* Handlers for simple string trees, like mml_avl_str_xxx, but
	   where the assumption is that the passed key is an allocated
	   string that may be pointed to when added, and deallocated
	   when removed.
	*/
int	mml_avl_strp_cmpkey();
AVLNODE	*mml_avl_strp_mknode();
int	mml_avl_strp_rmnode();


#endif	/* H_MML_AVL; */
