/* mml_in.h -- definitions for the mml "mml_in" module

	Copyright 2003,2004,2005,2006,2007
		by Mark E. Mallett, MV Communications, Inc.

	See the "LICENSE" file for terms.

mml_in provides a set of functions allowing layered input.
Input streams can be pushed onto and popped off of the input,
and can include input from a file and from a string.

*/


#ifndef	H_MML_IN		/* For multiple inclusions */
#define	H_MML_IN

#include <sys/types.h>

    /* Note: for the sake of efficiency,some mml modules know the details
       of MML_IN structure(s).  If you make significant changes here,
       check out these other places:

           mml-iopf constructs static MML_IN and MML_INS

    */


    /* The input stream control structure, used internally */
typedef struct mml_ins {
    struct mml_in  *is_inP;		/* Ptr to the MML_IN stack */
    struct mml_ins *is_prevP;		/* Ptr to next one down */
    int		is_pbc;			/* Pushed back character, -1 if none */
    int		(*is_getcP)();		/* Input char routine to call */
    void	(*is_finishP)();	/* Function to finish the handle */
    void	*is_handleP;		/* Handle for input stream */
    char	*is_nameP;		/* Name for the stream, if any */
    long	is_lineN;		/* Line number in stream */
    BOOL	is_poolF;		/* TRUE if this was allocated
					   from the stream pool and should
					   be put back into it (almost always
					   TRUE except for some special
					   insider cases).
					*/

}  MML_INS;


    /* The input stack handle */
typedef struct mml_in {
    BOOL	in_lescF;		/* If last char was escaped */
    BOOL	in_eolF;		/* If last char was end-of-line */
    BOOL	in_bolF;		/* If last char was beginning-of-line */
    MML_INS	*in_streamP;		/* Ptr to stream on top */
    int		in_level;		/* How many streams are on the stack */
    int		in_clineN;		/* Cached line number.  This attempts
					   to remember the last line number;
					   it is set before the next line
					   number is incremented and perhaps
					   at other times.
					*/
    IORESULT	in_result;		/* Extended result info for EOF. */
    int		in_errno;		/* Any errno associated with is_result */
}  MML_IN;


    /* Control structure for a counted file stream */
typedef struct {
    FILE	*cf_fP;			/* File pointer */
    off_t	cf_byteC;		/* Number of bytes remaining */
} IN_CFILE;


    /* Control structure for a string stream */
typedef struct {
    BOOL	cs_bufallocF;		/* TRUE if buffer was alloc'd and
						must be freed. */
    int		cs_bufX;		/* Buffer index */
    int		cs_strL;		/* String length */
    UBYTE	*cs_bufP;		/* Buffer pointer */
} IN_SSTR;


/* Declarations for functions contained in this module: */

    /* Input a char from the current input */
int	in_char PROTO( (MML_IN *inP) );

    /* Push a character back into the input */
void	in_charback PROTO( (MML_IN *inP, int ch) );

    /* Input a char, with escaping */
int	in_charesc PROTO( (MML_IN *inP, char *rawP) );

    /* Destroy an input stack */
void	in_destroy PROTO( (MML_IN *inP) );

    /* Input stream handler function(s): counted file stream */
void	in_h_cfile_finish PROTO( ( MML_INS *instrP ) );
int	in_h_cfile_getc PROTO( ( MML_INS *instrP ) );

    /* Input stream handler function(s): file stream */
void	in_h_file_finish PROTO( ( MML_INS *instrP ) );
int	in_h_file_getc PROTO( ( MML_INS *instrP ) );

    /* Input stream handler function(s): string stream */
void	in_h_sstr_finish PROTO( ( MML_INS *instrP ) );
int	in_h_sstr_getc PROTO( ( MML_INS *instrP ) );

    /* Return the line number of the current stream */
int	in_lineno PROTO( (MML_IN *inP) );

    /* Return the name associated with the current stream */
char	*in_name PROTO( (MML_IN *inP) );

    /* Create a new input stack */
MML_IN	*in_new PROTO( (void) );

    /* Get next non-blank char from input */
int	in_nonblank PROTO( ( MML_IN *inP, BOOL eolF ) );

    /* Print a string representing the location in the stream */
char	*in_stream_location PROTO( (MML_IN *inP) );

    /* Pop a stream from the input */
void	in_stream_pop PROTO( ( MML_IN *inP ) );

    /* Push a stream onto the input */
BOOL	in_stream_push PROTO( (MML_IN *inP,
			       int (*getcP)(), void (*finishP)(),
			       void *handleP, char *nameP) );

    /* Push a counted file stream onto the input, close the file on EOF */
BOOL	in_stream_push_cfile PROTO( (MML_IN *inP, FILE *fP, 
				     off_t byteC, char *nameP) );

    /* Push a file stream onto the input, close the file on EOF */
BOOL	in_stream_push_file PROTO( (MML_IN *inP, FILE *fP, char *nameP) );

    /* Push a file stream onto the input, don't close the file on EOF */
BOOL	in_stream_push_file_nc PROTO( (MML_IN *inP, FILE *fP, char *nameP) );

    /* Push a copy of a string onto the input */
BOOL	in_stream_push_newstr PROTO( (MML_IN *inP, char *strP,
				      int strL, char *nameP) );

    /* Push a reference to a string onto the input */
BOOL	in_stream_push_str PROTO( (MML_IN *inP, char *strP,
				   int strL, char *nameP) );

    /* Get a whitespace-delimited token from input stream */
int	in_token_ws PROTO( (MML_IN *inP, UBYTE *bufP, int bufL, BOOL eolF) );

#endif	/* H_MML_IN */
